/*
 * Copyright 2015 The gRPC Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.alibaba.nacos.shaded.io.grpc.inprocess;

import static com.alibaba.nacos.shaded.com.google.common.base.Preconditions.checkNotNull;
import static com.alibaba.nacos.shaded.io.grpc.internal.GrpcUtil.TIMEOUT_KEY;
import static java.lang.Math.max;

import com.alibaba.nacos.shaded.com.google.common.base.MoreObjects;
import com.alibaba.nacos.shaded.com.google.common.base.Optional;
import com.alibaba.nacos.shaded.com.google.common.util.concurrent.ListenableFuture;
import com.alibaba.nacos.shaded.com.google.common.util.concurrent.SettableFuture;
import com.alibaba.nacos.shaded.io.grpc.Attributes;
import com.alibaba.nacos.shaded.io.grpc.CallOptions;
import com.alibaba.nacos.shaded.io.grpc.Compressor;
import com.alibaba.nacos.shaded.io.grpc.Deadline;
import com.alibaba.nacos.shaded.io.grpc.Decompressor;
import com.alibaba.nacos.shaded.io.grpc.DecompressorRegistry;
import com.alibaba.nacos.shaded.io.grpc.Grpc;
import com.alibaba.nacos.shaded.io.grpc.InternalChannelz.SocketStats;
import com.alibaba.nacos.shaded.io.grpc.InternalLogId;
import com.alibaba.nacos.shaded.io.grpc.InternalMetadata;
import com.alibaba.nacos.shaded.io.grpc.Metadata;
import com.alibaba.nacos.shaded.io.grpc.MethodDescriptor;
import com.alibaba.nacos.shaded.io.grpc.SecurityLevel;
import com.alibaba.nacos.shaded.io.grpc.ServerStreamTracer;
import com.alibaba.nacos.shaded.io.grpc.Status;
import com.alibaba.nacos.shaded.io.grpc.internal.ClientStream;
import com.alibaba.nacos.shaded.io.grpc.internal.ClientStreamListener;
import com.alibaba.nacos.shaded.io.grpc.internal.ConnectionClientTransport;
import com.alibaba.nacos.shaded.io.grpc.internal.GrpcAttributes;
import com.alibaba.nacos.shaded.io.grpc.internal.GrpcUtil;
import com.alibaba.nacos.shaded.io.grpc.internal.InUseStateAggregator;
import com.alibaba.nacos.shaded.io.grpc.internal.InsightBuilder;
import com.alibaba.nacos.shaded.io.grpc.internal.ManagedClientTransport;
import com.alibaba.nacos.shaded.io.grpc.internal.NoopClientStream;
import com.alibaba.nacos.shaded.io.grpc.internal.ObjectPool;
import com.alibaba.nacos.shaded.io.grpc.internal.ServerListener;
import com.alibaba.nacos.shaded.io.grpc.internal.ServerStream;
import com.alibaba.nacos.shaded.io.grpc.internal.ServerStreamListener;
import com.alibaba.nacos.shaded.io.grpc.internal.ServerTransport;
import com.alibaba.nacos.shaded.io.grpc.internal.ServerTransportListener;
import com.alibaba.nacos.shaded.io.grpc.internal.StatsTraceContext;
import com.alibaba.nacos.shaded.io.grpc.internal.StreamListener;
import java.io.InputStream;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.Executor;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;
import com.alibaba.nacos.shaded.javax.annotation.CheckReturnValue;
import com.alibaba.nacos.shaded.javax.annotation.Nullable;
import com.alibaba.nacos.shaded.javax.annotation.concurrent.GuardedBy;
import com.alibaba.nacos.shaded.javax.annotation.concurrent.ThreadSafe;

@ThreadSafe
final class InProcessTransport implements ServerTransport, ConnectionClientTransport {
  private static final Logger log = Logger.getLogger(InProcessTransport.class.getName());

  private final InternalLogId logId;
  private final String name;
  private final int clientMaxInboundMetadataSize;
  private final String authority;
  private final String userAgent;
  private final Optional<ServerListener> optionalServerListener;
  private int serverMaxInboundMetadataSize;
  private final boolean includeCauseWithStatus;
  private ObjectPool<ScheduledExecutorService> serverSchedulerPool;
  private ScheduledExecutorService serverScheduler;
  private ServerTransportListener serverTransportListener;
  private Attributes serverStreamAttributes;
  private ManagedClientTransport.Listener clientTransportListener;
  @GuardedBy("this")
  private boolean shutdown;
  @GuardedBy("this")
  private boolean terminated;
  @GuardedBy("this")
  private Status shutdownStatus;
  @GuardedBy("this")
  private Set<InProcessStream> streams = new HashSet<>();
  @GuardedBy("this")
  private List<ServerStreamTracer.Factory> serverStreamTracerFactories;
  private final Attributes attributes;

  @GuardedBy("this")
  private final InUseStateAggregator<InProcessStream> inUseState =
      new InUseStateAggregator<InProcessStream>() {
        @Override
        protected void handleInUse() {
          clientTransportListener.transportInUse(true);
        }

        @Override
        protected void handleNotInUse() {
          clientTransportListener.transportInUse(false);
        }
      };

  private InProcessTransport(String name, int maxInboundMetadataSize, String authority,
      String userAgent, Attributes eagAttrs,
      Optional<ServerListener> optionalServerListener, boolean includeCauseWithStatus) {
    this.name = name;
    this.clientMaxInboundMetadataSize = maxInboundMetadataSize;
    this.authority = authority;
    this.userAgent = GrpcUtil.getGrpcUserAgent("inprocess", userAgent);
    checkNotNull(eagAttrs, "eagAttrs");
    this.attributes = Attributes.newBuilder()
        .set(GrpcAttributes.ATTR_SECURITY_LEVEL, SecurityLevel.PRIVACY_AND_INTEGRITY)
        .set(GrpcAttributes.ATTR_CLIENT_EAG_ATTRS, eagAttrs)
        .set(Grpc.TRANSPORT_ATTR_REMOTE_ADDR, new InProcessSocketAddress(name))
        .set(Grpc.TRANSPORT_ATTR_LOCAL_ADDR, new InProcessSocketAddress(name))
        .build();
    this.optionalServerListener = optionalServerListener;
    logId = InternalLogId.allocate(getClass(), name);
    this.includeCauseWithStatus = includeCauseWithStatus;
  }

  public InProcessTransport(
      String name, int maxInboundMetadataSize, String authority, String userAgent,
      Attributes eagAttrs, boolean includeCauseWithStatus) {
    this(name, maxInboundMetadataSize, authority, userAgent, eagAttrs,
        Optional.<ServerListener>absent(), includeCauseWithStatus);
  }

  InProcessTransport(
      String name, int maxInboundMetadataSize, String authority, String userAgent,
      Attributes eagAttrs, ObjectPool<ScheduledExecutorService> serverSchedulerPool,
      List<ServerStreamTracer.Factory> serverStreamTracerFactories,
      ServerListener serverListener) {
    this(name, maxInboundMetadataSize, authority, userAgent, eagAttrs,
        Optional.of(serverListener), false);
    this.serverMaxInboundMetadataSize = maxInboundMetadataSize;
    this.serverSchedulerPool = serverSchedulerPool;
    this.serverStreamTracerFactories = serverStreamTracerFactories;
  }

  @CheckReturnValue
  @Override
  public synchronized Runnable start(ManagedClientTransport.Listener listener) {
    this.clientTransportListener = listener;
    if (optionalServerListener.isPresent()) {
      serverScheduler = serverSchedulerPool.getObject();
      serverTransportListener = optionalServerListener.get().transportCreated(this);
    } else {
      InProcessServer server = InProcessServer.findServer(name);
      if (server != null) {
        serverMaxInboundMetadataSize = server.getMaxInboundMetadataSize();
        serverSchedulerPool = server.getScheduledExecutorServicePool();
        serverScheduler = serverSchedulerPool.getObject();
        serverStreamTracerFactories = server.getStreamTracerFactories();
        // Must be semi-initialized; past this point, can begin receiving requests
        serverTransportListener = server.register(this);
      }
    }
    if (serverTransportListener == null) {
      shutdownStatus = Status.UNAVAILABLE.withDescription("Could not find server: " + name);
      final Status localShutdownStatus = shutdownStatus;
      return new Runnable() {
        @Override
        public void run() {
          synchronized (InProcessTransport.this) {
            notifyShutdown(localShutdownStatus);
            notifyTerminated();
          }
        }
      };
    }
    return new Runnable() {
      @Override
      @SuppressWarnings("deprecation")
      public void run() {
        synchronized (InProcessTransport.this) {
          Attributes serverTransportAttrs = Attributes.newBuilder()
              .set(Grpc.TRANSPORT_ATTR_REMOTE_ADDR, new InProcessSocketAddress(name))
              .set(Grpc.TRANSPORT_ATTR_LOCAL_ADDR, new InProcessSocketAddress(name))
              .build();
          serverStreamAttributes = serverTransportListener.transportReady(serverTransportAttrs);
          clientTransportListener.transportReady();
        }
      }
    };
  }

  @Override
  public synchronized ClientStream newStream(
      final MethodDescriptor<?, ?> method, final Metadata headers, final CallOptions callOptions) {
    if (shutdownStatus != null) {
      return failedClientStream(
          StatsTraceContext.newClientContext(callOptions, attributes, headers), shutdownStatus);
    }

    headers.put(GrpcUtil.USER_AGENT_KEY, userAgent);

    if (serverMaxInboundMetadataSize != Integer.MAX_VALUE) {
      int metadataSize = metadataSize(headers);
      if (metadataSize > serverMaxInboundMetadataSize) {
        // Other transports would compute a status with:
        //   GrpcUtil.httpStatusToGrpcStatus(431 /* Request Header Fields Too Large */);
        // However, that isn't handled specially today, so we'd leak HTTP-isms even though we're
        // in-process. We go ahead and make a Status, which may need to be updated if
        // statuscodes.md is updated.
        Status status = Status.RESOURCE_EXHAUSTED.withDescription(
            String.format(
                "Request metadata larger than %d: %d",
                serverMaxInboundMetadataSize,
                metadataSize));
        return failedClientStream(
            StatsTraceContext.newClientContext(callOptions, attributes, headers), status);
      }
    }

    return new InProcessStream(method, headers, callOptions, authority).clientStream;
  }

  private ClientStream failedClientStream(
      final StatsTraceContext statsTraceCtx, final Status status) {
    return new NoopClientStream() {
        @Override
        public void start(ClientStreamListener listener) {
          statsTraceCtx.clientOutboundHeaders();
          statsTraceCtx.streamClosed(status);
          listener.closed(status, new Metadata());
        }
      };
  }

  @Override
  public synchronized void ping(final PingCallback callback, Executor executor) {
    if (terminated) {
      final Status shutdownStatus = this.shutdownStatus;
      executor.execute(new Runnable() {
        @Override
        public void run() {
          callback.onFailure(shutdownStatus.asRuntimeException());
        }
      });
    } else {
      executor.execute(new Runnable() {
        @Override
        public void run() {
          callback.onSuccess(0);
        }
      });
    }
  }

  @Override
  public synchronized void shutdown(Status reason) {
    // Can be called multiple times: once for ManagedClientTransport, once for ServerTransport.
    if (shutdown) {
      return;
    }
    shutdownStatus = reason;
    notifyShutdown(reason);
    if (streams.isEmpty()) {
      notifyTerminated();
    }
  }

  @Override
  public synchronized void shutdown() {
    shutdown(Status.UNAVAILABLE.withDescription("InProcessTransport shutdown by the server-side"));
  }

  @Override
  public void shutdownNow(Status reason) {
    checkNotNull(reason, "reason");
    List<InProcessStream> streamsCopy;
    synchronized (this) {
      shutdown(reason);
      if (terminated) {
        return;
      }
      streamsCopy = new ArrayList<>(streams);
    }
    for (InProcessStream stream : streamsCopy) {
      stream.clientStream.cancel(reason);
    }
  }

  @Override
  public String toString() {
    return MoreObjects.toStringHelper(this)
        .add("logId", logId.getId())
        .add("name", name)
        .toString();
  }

  @Override
  public InternalLogId getLogId() {
    return logId;
  }

  @Override
  public Attributes getAttributes() {
    return attributes;
  }

  @Override
  public ScheduledExecutorService getScheduledExecutorService() {
    return serverScheduler;
  }

  @Override
  public ListenableFuture<SocketStats> getStats() {
    SettableFuture<SocketStats> ret = SettableFuture.create();
    ret.set(null);
    return ret;
  }

  private synchronized void notifyShutdown(Status s) {
    if (shutdown) {
      return;
    }
    shutdown = true;
    clientTransportListener.transportShutdown(s);
  }

  private synchronized void notifyTerminated() {
    if (terminated) {
      return;
    }
    terminated = true;
    if (serverScheduler != null) {
      serverScheduler = serverSchedulerPool.returnObject(serverScheduler);
    }
    clientTransportListener.transportTerminated();
    if (serverTransportListener != null) {
      serverTransportListener.transportTerminated();
    }
  }

  private static int metadataSize(Metadata metadata) {
    byte[][] serialized = InternalMetadata.serialize(metadata);
    if (serialized == null) {
      return 0;
    }
    // Calculate based on SETTINGS_MAX_HEADER_LIST_SIZE in RFC 7540 §6.5.2. We could use something
    // different, but it's "sane."
    long size = 0;
    for (int i = 0; i < serialized.length; i += 2) {
      size += 32 + serialized[i].length + serialized[i + 1].length;
    }
    size = Math.min(size, Integer.MAX_VALUE);
    return (int) size;
  }

  private class InProcessStream {
    private final InProcessClientStream clientStream;
    private final InProcessServerStream serverStream;
    private final CallOptions callOptions;
    private final Metadata headers;
    private final MethodDescriptor<?, ?> method;
    private volatile String authority;

    private InProcessStream(
        MethodDescriptor<?, ?> method, Metadata headers, CallOptions callOptions,
        String authority) {
      this.method = checkNotNull(method, "method");
      this.headers = checkNotNull(headers, "headers");
      this.callOptions = checkNotNull(callOptions, "callOptions");
      this.authority = authority;
      this.clientStream = new InProcessClientStream(callOptions, headers);
      this.serverStream = new InProcessServerStream(method, headers);
    }

    // Can be called multiple times due to races on both client and server closing at same time.
    private void streamClosed() {
      synchronized (InProcessTransport.this) {
        boolean justRemovedAnElement = streams.remove(this);
        if (GrpcUtil.shouldBeCountedForInUse(callOptions)) {
          inUseState.updateObjectInUse(this, false);
        }
        if (streams.isEmpty() && justRemovedAnElement) {
          if (shutdown) {
            notifyTerminated();
          }
        }
      }
    }

    private class InProcessServerStream implements ServerStream {
      final StatsTraceContext statsTraceCtx;
      @GuardedBy("this")
      private ClientStreamListener clientStreamListener;
      @GuardedBy("this")
      private int clientRequested;
      @GuardedBy("this")
      private ArrayDeque<StreamListener.MessageProducer> clientReceiveQueue =
          new ArrayDeque<>();
      @GuardedBy("this")
      private Status clientNotifyStatus;
      @GuardedBy("this")
      private Metadata clientNotifyTrailers;
      // Only is intended to prevent double-close when client cancels.
      @GuardedBy("this")
      private boolean closed;
      @GuardedBy("this")
      private int outboundSeqNo;

      InProcessServerStream(MethodDescriptor<?, ?> method, Metadata headers) {
        statsTraceCtx = StatsTraceContext.newServerContext(
            serverStreamTracerFactories, method.getFullMethodName(), headers);
      }

      private synchronized void setListener(ClientStreamListener listener) {
        clientStreamListener = listener;
      }

      @Override
      public void setListener(ServerStreamListener serverStreamListener) {
        clientStream.setListener(serverStreamListener);
      }

      @Override
      public void request(int numMessages) {
        boolean onReady = clientStream.serverRequested(numMessages);
        if (onReady) {
          synchronized (this) {
            if (!closed) {
              clientStreamListener.onReady();
            }
          }
        }
      }

      // This method is the only reason we have to synchronize field accesses.
      /**
       * Client requested more messages.
       *
       * @return whether onReady should be called on the server
       */
      private synchronized boolean clientRequested(int numMessages) {
        if (closed) {
          return false;
        }
        boolean previouslyReady = clientRequested > 0;
        clientRequested += numMessages;
        while (clientRequested > 0 && !clientReceiveQueue.isEmpty()) {
          clientRequested--;
          clientStreamListener.messagesAvailable(clientReceiveQueue.poll());
        }
        // Attempt being reentrant-safe
        if (closed) {
          return false;
        }
        if (clientReceiveQueue.isEmpty() && clientNotifyStatus != null) {
          closed = true;
          clientStream.statsTraceCtx.clientInboundTrailers(clientNotifyTrailers);
          clientStream.statsTraceCtx.streamClosed(clientNotifyStatus);
          clientStreamListener.closed(clientNotifyStatus, clientNotifyTrailers);
        }
        boolean nowReady = clientRequested > 0;
        return !previouslyReady && nowReady;
      }

      private void clientCancelled(Status status) {
        internalCancel(status);
      }

      @Override
      public synchronized void writeMessage(InputStream message) {
        if (closed) {
          return;
        }
        statsTraceCtx.outboundMessage(outboundSeqNo);
        statsTraceCtx.outboundMessageSent(outboundSeqNo, -1, -1);
        clientStream.statsTraceCtx.inboundMessage(outboundSeqNo);
        clientStream.statsTraceCtx.inboundMessageRead(outboundSeqNo, -1, -1);
        outboundSeqNo++;
        StreamListener.MessageProducer producer = new SingleMessageProducer(message);
        if (clientRequested > 0) {
          clientRequested--;
          clientStreamListener.messagesAvailable(producer);
        } else {
          clientReceiveQueue.add(producer);
        }
      }

      @Override
      public void flush() {}

      @Override
      public synchronized boolean isReady() {
        if (closed) {
          return false;
        }
        return clientRequested > 0;
      }

      @Override
      public void writeHeaders(Metadata headers) {
        if (clientMaxInboundMetadataSize != Integer.MAX_VALUE) {
          int metadataSize = metadataSize(headers);
          if (metadataSize > clientMaxInboundMetadataSize) {
            Status serverStatus = Status.CANCELLED.withDescription("Client cancelled the RPC");
            clientStream.serverClosed(serverStatus, serverStatus);
            // Other transports provide very little information in this case. We go ahead and make a
            // Status, which may need to be updated if statuscodes.md is updated.
            Status failedStatus = Status.RESOURCE_EXHAUSTED.withDescription(
                String.format(
                    "Response header metadata larger than %d: %d",
                    clientMaxInboundMetadataSize,
                    metadataSize));
            notifyClientClose(failedStatus, new Metadata());
            return;
          }
        }

        synchronized (this) {
          if (closed) {
            return;
          }

          clientStream.statsTraceCtx.clientInboundHeaders();
          clientStreamListener.headersRead(headers);
        }
      }

      @Override
      public void close(Status status, Metadata trailers) {
        // clientStream.serverClosed must happen before clientStreamListener.closed, otherwise
        // clientStreamListener.closed can trigger clientStream.cancel (see code in
        // ClientCalls.blockingUnaryCall), which may race with clientStream.serverClosed as both are
        // calling internalCancel().
        clientStream.serverClosed(Status.OK, status);

        if (clientMaxInboundMetadataSize != Integer.MAX_VALUE) {
          int statusSize = status.getDescription() == null ? 0 : status.getDescription().length();
          // Go ahead and throw in the status description's length, since that could be very long.
          int metadataSize = metadataSize(trailers) + statusSize;
          if (metadataSize > clientMaxInboundMetadataSize) {
            // Override the status for the client, but not the server. Transports do not guarantee
            // notifying the server of the failure.

            // Other transports provide very little information in this case. We go ahead and make a
            // Status, which may need to be updated if statuscodes.md is updated.
            status = Status.RESOURCE_EXHAUSTED.withDescription(
                String.format(
                    "Response header metadata larger than %d: %d",
                    clientMaxInboundMetadataSize,
                    metadataSize));
            trailers = new Metadata();
          }
        }

        notifyClientClose(status, trailers);
      }

      /** clientStream.serverClosed() must be called before this method */
      private void notifyClientClose(Status status, Metadata trailers) {
        Status clientStatus = cleanStatus(status, includeCauseWithStatus);
        synchronized (this) {
          if (closed) {
            return;
          }
          if (clientReceiveQueue.isEmpty()) {
            closed = true;
            clientStream.statsTraceCtx.clientInboundTrailers(trailers);
            clientStream.statsTraceCtx.streamClosed(clientStatus);
            clientStreamListener.closed(clientStatus, trailers);
          } else {
            clientNotifyStatus = clientStatus;
            clientNotifyTrailers = trailers;
          }
        }

        streamClosed();
      }

      @Override
      public void cancel(Status status) {
        if (!internalCancel(Status.CANCELLED.withDescription("server cancelled stream"))) {
          return;
        }
        clientStream.serverClosed(status, status);
        streamClosed();
      }

      private synchronized boolean internalCancel(Status clientStatus) {
        if (closed) {
          return false;
        }
        closed = true;
        StreamListener.MessageProducer producer;
        while ((producer = clientReceiveQueue.poll()) != null) {
          InputStream message;
          while ((message = producer.next()) != null) {
            try {
              message.close();
            } catch (Throwable t) {
              log.log(Level.WARNING, "Exception closing stream", t);
            }
          }
        }
        clientStream.statsTraceCtx.streamClosed(clientStatus);
        clientStreamListener.closed(clientStatus, new Metadata());
        return true;
      }

      @Override
      public void setMessageCompression(boolean enable) {
        // noop
      }

      @Override
      public void setCompressor(Compressor compressor) {}

      @Override
      public void setDecompressor(Decompressor decompressor) {}

      @Override public Attributes getAttributes() {
        return serverStreamAttributes;
      }

      @Override
      public String getAuthority() {
        return InProcessStream.this.authority;
      }

      @Override
      public StatsTraceContext statsTraceContext() {
        return statsTraceCtx;
      }

      @Override
      public int streamId() {
        return -1;
      }
    }

    private class InProcessClientStream implements ClientStream {
      final StatsTraceContext statsTraceCtx;
      final CallOptions callOptions;
      @GuardedBy("this")
      private ServerStreamListener serverStreamListener;
      @GuardedBy("this")
      private int serverRequested;
      @GuardedBy("this")
      private ArrayDeque<StreamListener.MessageProducer> serverReceiveQueue =
          new ArrayDeque<>();
      @GuardedBy("this")
      private boolean serverNotifyHalfClose;
      // Only is intended to prevent double-close when server closes.
      @GuardedBy("this")
      private boolean closed;
      @GuardedBy("this")
      private int outboundSeqNo;

      InProcessClientStream(CallOptions callOptions, Metadata headers) {
        this.callOptions = callOptions;
        statsTraceCtx = StatsTraceContext.newClientContext(callOptions, attributes, headers);
      }

      private synchronized void setListener(ServerStreamListener listener) {
        this.serverStreamListener = listener;
      }

      @Override
      public void request(int numMessages) {
        boolean onReady = serverStream.clientRequested(numMessages);
        if (onReady) {
          synchronized (this) {
            if (!closed) {
              serverStreamListener.onReady();
            }
          }
        }
      }

      // This method is the only reason we have to synchronize field accesses.
      /**
       * Client requested more messages.
       *
       * @return whether onReady should be called on the server
       */
      private synchronized boolean serverRequested(int numMessages) {
        if (closed) {
          return false;
        }
        boolean previouslyReady = serverRequested > 0;
        serverRequested += numMessages;
        while (serverRequested > 0 && !serverReceiveQueue.isEmpty()) {
          serverRequested--;
          serverStreamListener.messagesAvailable(serverReceiveQueue.poll());
        }
        if (serverReceiveQueue.isEmpty() && serverNotifyHalfClose) {
          serverNotifyHalfClose = false;
          serverStreamListener.halfClosed();
        }
        boolean nowReady = serverRequested > 0;
        return !previouslyReady && nowReady;
      }

      private void serverClosed(Status serverListenerStatus, Status serverTracerStatus) {
        internalCancel(serverListenerStatus, serverTracerStatus);
      }

      @Override
      public synchronized void writeMessage(InputStream message) {
        if (closed) {
          return;
        }
        statsTraceCtx.outboundMessage(outboundSeqNo);
        statsTraceCtx.outboundMessageSent(outboundSeqNo, -1, -1);
        serverStream.statsTraceCtx.inboundMessage(outboundSeqNo);
        serverStream.statsTraceCtx.inboundMessageRead(outboundSeqNo, -1, -1);
        outboundSeqNo++;
        StreamListener.MessageProducer producer = new SingleMessageProducer(message);
        if (serverRequested > 0) {
          serverRequested--;
          serverStreamListener.messagesAvailable(producer);
        } else {
          serverReceiveQueue.add(producer);
        }
      }

      @Override
      public void flush() {}

      @Override
      public synchronized boolean isReady() {
        if (closed) {
          return false;
        }
        return serverRequested > 0;
      }

      // Must be thread-safe for shutdownNow()
      @Override
      public void cancel(Status reason) {
        Status serverStatus = cleanStatus(reason, includeCauseWithStatus);
        if (!internalCancel(serverStatus, serverStatus)) {
          return;
        }
        serverStream.clientCancelled(reason);
        streamClosed();
      }

      private synchronized boolean internalCancel(
          Status serverListenerStatus, Status serverTracerStatus) {
        if (closed) {
          return false;
        }
        closed = true;

        StreamListener.MessageProducer producer;
        while ((producer = serverReceiveQueue.poll()) != null) {
          InputStream message;
          while ((message = producer.next()) != null) {
            try {
              message.close();
            } catch (Throwable t) {
              log.log(Level.WARNING, "Exception closing stream", t);
            }
          }
        }
        serverStream.statsTraceCtx.streamClosed(serverTracerStatus);
        serverStreamListener.closed(serverListenerStatus);
        return true;
      }

      @Override
      public synchronized void halfClose() {
        if (closed) {
          return;
        }
        if (serverReceiveQueue.isEmpty()) {
          serverStreamListener.halfClosed();
        } else {
          serverNotifyHalfClose = true;
        }
      }

      @Override
      public void setMessageCompression(boolean enable) {}

      @Override
      public void setAuthority(String string) {
        InProcessStream.this.authority = string;
      }

      @Override
      public void start(ClientStreamListener listener) {
        serverStream.setListener(listener);

        synchronized (InProcessTransport.this) {
          statsTraceCtx.clientOutboundHeaders();
          streams.add(InProcessTransport.InProcessStream.this);
          if (GrpcUtil.shouldBeCountedForInUse(callOptions)) {
            inUseState.updateObjectInUse(InProcessTransport.InProcessStream.this, true);
          }
          serverTransportListener.streamCreated(serverStream, method.getFullMethodName(), headers);
        }
      }

      @Override
      public Attributes getAttributes() {
        return attributes;
      }

      @Override
      public void setCompressor(Compressor compressor) {}

      @Override
      public void setFullStreamDecompression(boolean fullStreamDecompression) {}

      @Override
      public void setDecompressorRegistry(DecompressorRegistry decompressorRegistry) {}

      @Override
      public void setMaxInboundMessageSize(int maxSize) {}

      @Override
      public void setMaxOutboundMessageSize(int maxSize) {}

      @Override
      public void setDeadline(Deadline deadline) {
        headers.discardAll(TIMEOUT_KEY);
        long effectiveTimeout = max(0, deadline.timeRemaining(TimeUnit.NANOSECONDS));
        headers.put(TIMEOUT_KEY, effectiveTimeout);
      }

      @Override
      public void appendTimeoutInsight(InsightBuilder insight) {
      }
    }
  }

  /**
   * Returns a new status with the same code and description.
   * If includeCauseWithStatus is true, cause is also included.
   *
   * <p>For InProcess transport to behave in the same way as the other transports,
   * when exchanging statuses between client and server and vice versa,
   * the cause should be excluded from the status.
   * For easier debugging, the status may be optionally included.
   */
  private static Status cleanStatus(Status status, boolean includeCauseWithStatus) {
    if (status == null) {
      return null;
    }
    Status clientStatus = Status
        .fromCodeValue(status.getCode().value())
        .withDescription(status.getDescription());
    if (includeCauseWithStatus) {
      clientStatus = clientStatus.withCause(status.getCause());
    }
    return clientStatus;
  }

  private static class SingleMessageProducer implements StreamListener.MessageProducer {
    private InputStream message;

    private SingleMessageProducer(InputStream message) {
      this.message = message;
    }

    @Nullable
    @Override
    public InputStream next() {
      InputStream messageToReturn = message;
      message = null;
      return messageToReturn;
    }
  }
}
