package com.bitkernel.stream.rapid;

import androidx.annotation.NonNull;

import com.bitkernel.stream.rapid.prt.PrtStatistic;
import com.bitkernel.stream.rapid.utils.LogUtil;

import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * Representation of a stream connection.
 */
public class RapidAgentCall implements RapidAgentSDK.OnPlayListener {
    /**
     * Stream status enum
     */
    public enum Status {
        IDLE, REQUESTING, REQUESTED, CALLING, CONNECTED, READY, ERROR, RELEASED
    }

    private static final String TAG = "RapidAgentCall";
    private static final AtomicInteger GLOBAL_CALL_INDEX = new AtomicInteger(0);
    private static final int IGNORE_CHANNEL_ID = Integer.MIN_VALUE + 1;

    private final PrtStatistic statistic;
    private final int index;
    private final long startTime;
    private int requestIndex;
    private int callIndex;
    private int channelId = IGNORE_CHANNEL_ID;
    private Status status;
    private boolean callSuccess = false;

    RapidAgentCall(RapidAgentUri rapidAgentUri) {
        statistic = new PrtStatistic();
        index = GLOBAL_CALL_INDEX.addAndGet(1);
        startTime = System.nanoTime();
        updateStatus(Status.IDLE);
    }

    /**
     * Gets the index which is unique of the stream connection
     *
     * @return the index
     */
    public int getIndex() {
        return index;
    }

    void setRequestIndex(int requestIndex) {
        this.requestIndex = requestIndex;
        if (requestIndex > 0) {
            channelId = requestIndex;
        }
    }

    /**
     * Gets the request index of the stream connection
     *
     * @return the request index
     */
    public int getRequestIndex() {
        return requestIndex;
    }

    void setCallIndex(int callIndex) {
        this.callIndex = callIndex;
        if (callIndex > 0) {
            channelId = callIndex;
        }
    }

    /**
     * Gets the call index of the stream connection
     *
     * @return the call index
     */
    public int getCallIndex() {
        return callIndex;
    }

    /**
     * Gets the start time of the connection in milliseconds since the Unix epoch
     *
     * @return timestamp
     */
    public long getStartTime() {
        return startTime;
    }

    void updateStatus(Status status) {
        this.status = status;
        if (status == Status.READY) {
            callSuccess = true;
        }
        printState("updateStatus");
    }

    private void printState(String msg) {
        long duration = (System.nanoTime() - startTime) / 1_000_000;
        LogUtil.i(TAG, msg + ":" + this + " " + duration + "ms");
    }

    @Override
    public void onPrtEvent(int event, String params, String desc) {
        switch (event) {
            case RapidAgentSDK.EVENT_PROXY_READY:
                updateStatus(Status.READY);
                break;
            case RapidAgentSDK.EVENT_CREATE_CHANNEL_FAIL:
            case RapidAgentSDK.EVENT_CREATE_CHANNEL_TIMEOUT:
            case RapidAgentSDK.EVENT_PROXY_NO_SIGNAL:
            case RapidAgentSDK.EVENT_CHANGE_STREAM_NO_SIGNAL:
            case RapidAgentSDK.EVENT_PROXY_VERIFY:
            case RapidAgentSDK.EVENT_GET_INDEX_FAIL:
                updateStatus(Status.ERROR);
                break;
        }
        statistic.onEvent(channelId, event, params, desc, callSuccess);
    }

    @Override
    public void onPrtMetric(Map<String, String> params) {
        statistic.onMetric(channelId, params);
    }

    @Override
    public void onPrtState(Map<String, String> state) {

    }

    @Override
    public void onPrtProgress() {
        if (callIndex > 0) {
            statistic.notifyStatusEvent(callIndex);
        }
    }

    @NonNull
    @Override
    public String toString() {
        return String.format("RapidAgentCall[%s,%s,%s] %s", index, requestIndex, callIndex, status);
    }
}
