package com.bitkernel.stream.rapid;

import android.app.ActivityManager;
import android.content.Context;
import android.os.Build;
import android.os.Environment;
import android.os.StatFs;
import android.text.TextUtils;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.core.content.ContextCompat;

import com.bitkernel.stream.rapid.utils.CommonUtil;
import com.bitkernel.stream.rapid.utils.LogUtil;
import com.bitkernel.stream.rapid.utils.SPUtil;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileReader;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.Objects;
import java.util.regex.Pattern;

public final class RapidAgentConstant {
    private static int SDK_VERSION_CODE;
    private static int PLAYER_VERSION_CODE;
    private static int ENGINE_VERSION_CODE;
    private static int SYSTEM_VERSION_CODE;

    private static String APPLICATION_NAME;
    private static String VNO_ID;
    private static String VNO_TAG;
    private static String SECRET_KEY;
    private static String TOKEN;
    private static String ACCOUNT_ID;
    private static String ACCOUNT_AID;

    private static int PLATFORM_TYPE;
    private static String DEVICE_ID;
    private static String ANDROID_ID;
    private static long CID;
    private static String BRAND;
    private static String MODEL;
    private static String CPU_NAME;
    private static String CPU_INFO;
    private static String MEM_INFO;
    private static int STORAGE_SIZE_GB;

    private static String ROOT_DIR;
    private static String SDK_LOG_DIR;
    private static String PRT_WORK_DIR;
    private static String PRT_PM_CACHE_DIR;
    private static String PRT_CFG_FILE_NAME;
    private static String PCT_WORK_DIR;
    private static String RDNS_WORK_DIR;

    static void initWorkPath(@Nullable Context context) {
        String rootDir;
        if (context != null) {
            rootDir = Objects.requireNonNull(ContextCompat.getDataDir(context)).getAbsolutePath() + File.separator + "rapidtv" + File.separator;
        } else if (CommonUtil.externalStorageAvailable()) {
            rootDir = Environment.getExternalStorageDirectory().getAbsolutePath() + File.separator + "rapidtv" + File.separator;
        } else {
            rootDir = File.separator + "rapidtv" + File.separator;
        }
        if (context == null) {
            LogUtil.w("RapidAgentConstant", "Invalid Context Warning : initWorkPath " + rootDir);
        }
        initWorkPath(rootDir);
    }

    static void initWorkPath(@NonNull String rootDir) {
        if (!TextUtils.isEmpty(ROOT_DIR)) {
            return;
        }
        if (rootDir.endsWith(File.separator)) {
            ROOT_DIR = rootDir;
        } else {
            ROOT_DIR = rootDir + File.separator;
        }
        LogUtil.sensitive("base work path : " + ROOT_DIR);
        SDK_LOG_DIR = ROOT_DIR + "sdklog";
        File sdkLogDir = new File(SDK_LOG_DIR);
        if (!sdkLogDir.exists() && !sdkLogDir.mkdirs()) {
            LogUtil.e("RapidAgentConstant", "can not create sdk log dir");
        }
        PRT_WORK_DIR = ROOT_DIR + "prt";
        File prtLogDir = new File(PRT_WORK_DIR);
        if (!prtLogDir.exists() && !prtLogDir.mkdirs()) {
            LogUtil.e("RapidAgentConstant", "can not create prt log dir");
        }
        PRT_PM_CACHE_DIR = ROOT_DIR + "prt_pm" + File.separator + "cache";
        File pmCacheDir = new File(PRT_PM_CACHE_DIR);
        if (!pmCacheDir.exists() && !pmCacheDir.mkdirs()) {
            LogUtil.e("RapidAgentConstant", "can not create pm cache dir");
        }
        PCT_WORK_DIR = ROOT_DIR + "pct";
        File pctWorkPath = new File(PCT_WORK_DIR);
        if (!pctWorkPath.exists() && !pctWorkPath.mkdirs()) {
            LogUtil.e("RapidAgentConstant", "can not create pct work path");
        }
        RDNS_WORK_DIR = ROOT_DIR + "rdns";
        File rdnsWorkPath = new File(RDNS_WORK_DIR);
        if (!rdnsWorkPath.exists() && !rdnsWorkPath.mkdirs()) {
            LogUtil.e("RapidAgentConstant", "can not create rdns work path");
        }
        PRT_CFG_FILE_NAME = "cfgFile.txt";
    }

    static void init(@Nullable Context context, int platform, String appName, String vnoId, String vnoTag,
                     String secretKey, String token, String uid) {
        initWorkPath(context);
        PLATFORM_TYPE = platform;
        BRAND = android.os.Build.BRAND;
        MODEL = android.os.Build.MODEL;
        CPU_NAME = readCpuName() + "-" + Build.HARDWARE;
        CPU_INFO = readCpuInfo();
        MEM_INFO = readMemInfo(context);
        STORAGE_SIZE_GB = readTotalRom();

        APPLICATION_NAME = appName;
        VNO_ID = vnoId;
        VNO_TAG = vnoTag != null ? vnoTag : "";
        SECRET_KEY = secretKey;
        TOKEN = token;
        ACCOUNT_ID = uid;

        DEVICE_ID = SPUtil.getDeviceId();
        ANDROID_ID = SPUtil.getAndroidId();
        ACCOUNT_AID = ACCOUNT_ID + DEVICE_ID;
        CID = SPUtil.getCID(DEVICE_ID);
    }

    static void setVersions(int playerVersionCode, int engineVersionCode) {
        SDK_VERSION_CODE = CommonUtil.parseInt(BuildConfig.VERSION_CODE, 0);
        PLAYER_VERSION_CODE = playerVersionCode;
        ENGINE_VERSION_CODE = engineVersionCode;
        SYSTEM_VERSION_CODE = Build.VERSION.SDK_INT;
        LogUtil.i("RapidAgentConstant", "setVersions prt=" + ENGINE_VERSION_CODE + ", player=" + PLAYER_VERSION_CODE + ", sdk=" + SDK_VERSION_CODE);
    }

    /**
     * Gets the number of cores available in this device, across all processors.
     * Requires: Ability to peruse the filesystem at "/sys/devices/system/cpu"
     *
     * @return The number of cores, or 1 if failed to get result
     */
    private static String readCpuInfo() {
        List<String> freqList = new ArrayList<>();
        try {
            //Get directory containing CPU info
            File dir = new File("/sys/devices/system/cpu/");
            //Filter to only list the devices we care about
            File[] files = dir.listFiles(file -> Pattern.matches("cpu[0-9]", file.getName()));

            if (files != null && files.length > 0) {
                byte[] content = new byte[1024];
                for (File file : files) {
                    try (FileInputStream inStream = new FileInputStream(file.getAbsolutePath() + File.separator + "cpufreq" + File.separator + "cpuinfo_max_freq")) {
                        int len = inStream.read(content);
                        freqList.add(new String(content, 0, len));
                    }
                }
            }
        } catch (Exception e) {
            LogUtil.e("RapidAgentConstant", "fail to read cpu info: " + e);
        }

        String cpuInfo = "[" + CPU_NAME + "] [" + Build.PRODUCT + "] [" + Build.DEVICE + "] " + freqList;
        return cpuInfo.replace("\n", "");
    }

    private static String readCpuName() {
        String str1 = "/proc/cpuinfo";
        String str2;
        String cpuName = "";

        try (BufferedReader bufferedReader = new BufferedReader(new FileReader(str1))) {
            while ((str2 = bufferedReader.readLine()) != null) {
                if (TextUtils.isDigitsOnly(str2)) {
                    continue;
                }
                String[] array = str2.split(":\\s+", 2);
                if (TextUtils.equals(array[0].trim(), "Hardware")) {
                    cpuName = array[1];
                    break;
                }
            }
        } catch (IOException e) {
            LogUtil.e("RapidAgentConstant", "fail to read cpu name: " + e);
        }
        return cpuName;
    }

    private static String readMemInfo(@Nullable Context context) {
        if (context == null) {
            LogUtil.w("RapidAgentConstant", "Invalid Context Warning : readMemInfo");
            return "";
        }
        try {
            ActivityManager manager = (ActivityManager) context.getSystemService(Context.ACTIVITY_SERVICE);
            ActivityManager.MemoryInfo memoryInfo = new ActivityManager.MemoryInfo();
            manager.getMemoryInfo(memoryInfo);
            return String.format(Locale.US, "%.2fG,%.2fG", memoryInfo.totalMem / 1024f / 1024f / 1024f, memoryInfo.availMem / 1024f / 1024f / 1024f);
        } catch (Exception e) {
            LogUtil.e("RapidAgentConstant", "fail to read mem info: " + e);
        }
        return "";
    }

    private static int readTotalRom() {
        File dataDir = Environment.getDataDirectory();
        StatFs stat = new StatFs(dataDir.getPath());
        long blockSize = stat.getBlockSizeLong();
        long totalBlocks = stat.getBlockCountLong();
        long size = totalBlocks * blockSize;
        long GB = 1024 * 1024 * 1024;
        return (int) (size / GB);
    }

    public static int getSdkVersion() {
        return SDK_VERSION_CODE;
    }

    public static int getPlayerVersion() {
        return PLAYER_VERSION_CODE;
    }

    public static int getEngineVersion() {
        return ENGINE_VERSION_CODE;
    }

    public static int getSystemVersion() {
        return SYSTEM_VERSION_CODE;
    }

    public static String getApplicationName() {
        return APPLICATION_NAME;
    }

    public static String getVnoId() {
        return VNO_ID;
    }

    public static String getVnoTag() {
        return VNO_TAG;
    }

    public static String getSecretKey() {
        return SECRET_KEY;
    }

    public static String getToken() {
        return TOKEN;
    }

    public static String getAccountId() {
        return ACCOUNT_ID;
    }

    public static String getAccountAid() {
        return ACCOUNT_AID;
    }

    public static int getPlatformType() {
        return PLATFORM_TYPE;
    }

    public static String getDeviceId() {
        return DEVICE_ID;
    }

    public static String getAndroidId() {
        return ANDROID_ID;
    }

    public static long getCID() {
        return CID;
    }

    public static String getBrand() {
        return BRAND;
    }

    public static String getModel() {
        return MODEL;
    }

    public static String getCpuName() {
        return CPU_NAME;
    }

    public static String getCpuInfo() {
        return CPU_INFO;
    }

    public static String getMemInfo() {
        return MEM_INFO;
    }

    public static int getStorageSizeGb() {
        return STORAGE_SIZE_GB;
    }

    public static String getRootDir() {
        return ROOT_DIR;
    }

    public static String getSdkLogDir() {
        return SDK_LOG_DIR;
    }

    public static String getPrtWorkDir() {
        return PRT_WORK_DIR;
    }

    public static String getPrtCfgFileName() {
        return PRT_CFG_FILE_NAME;
    }

    public static String getPrtPmCacheDir() {
        return PRT_PM_CACHE_DIR;
    }

    public static String getPctWorkDir() {
        return PCT_WORK_DIR;
    }

    public static String getRDnsWorkDir() {
        return RDNS_WORK_DIR;
    }
}
