package com.bitkernel.stream.rapid;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.bitkernel.stream.rapid.config.CommonConfig;
import com.bitkernel.stream.rapid.http.DefaultHttpApi;
import com.bitkernel.stream.rapid.http.HttpApi;
import com.bitkernel.stream.rapid.http.OkHttpApi;
import com.bitkernel.stream.rapid.pct.PctLockException;
import com.bitkernel.stream.rapid.utils.LogUtil;

import java.io.IOException;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.util.Map;

public class RapidAgentHttpClient {
    private static final String TAG = "RapidAgentHttpClient";
    private static final int CONNECT_TIMEOUT_MS = 10_000;
    private static final int READ_TIMEOUT_MS = 5000;
    private static final boolean USE_CACHES = false;
    private static final boolean ALLOW_GZIP = true;
    private static final boolean FOLLOW_REDIRECTS = true;

    public static final HttpClientType INIT_CLIENT_TYPE;

    public enum HttpClientType {
        Default, OkHttp, PCT
    }

    private static HttpClientType httpClientType;
    private final HttpApi httpApi;
    private final HttpApi fallbackHttpApi;
    private final int maxRetryTimes;
    private final int retryIntervalMs;

    static {
        try {
            Class.forName("okhttp3.OkHttpClient");
            httpClientType = HttpClientType.OkHttp;
        } catch (Exception e) {
            LogUtil.i("RapidAgentHttpClient", "OkHttpClient is not implementation");
            httpClientType = HttpClientType.Default;
        }
        INIT_CLIENT_TYPE = httpClientType;
        LogUtil.i("RapidAgentHttpClient", "default http client is " + httpClientType);
    }

    public static void setHttpClientType(HttpClientType httpClientType) {
        RapidAgentHttpClient.httpClientType = httpClientType;
    }

    public RapidAgentHttpClient() {
        this(httpClientType);
    }

    public RapidAgentHttpClient(HttpClientType type) {
        this(type, CONNECT_TIMEOUT_MS, READ_TIMEOUT_MS, USE_CACHES, ALLOW_GZIP, FOLLOW_REDIRECTS);
    }

    public RapidAgentHttpClient(HttpClientType type,
                                int connectTimeoutMillis,
                                int readTimeoutMillis) {
        this(type, connectTimeoutMillis, readTimeoutMillis, USE_CACHES, ALLOW_GZIP, FOLLOW_REDIRECTS);
    }

    public RapidAgentHttpClient(HttpClientType type,
                                int connectTimeoutMillis,
                                int readTimeoutMillis,
                                boolean useCaches,
                                boolean allowGzip,
                                boolean followRedirects) {
        httpApi = createHttpApi(type, connectTimeoutMillis, readTimeoutMillis, useCaches, allowGzip, followRedirects);
        boolean fallback = CommonConfig.getPctFallback();
        if (fallback && type == HttpClientType.PCT) {
            HttpClientType fallbackType = RapidAgentHttpClient.INIT_CLIENT_TYPE == RapidAgentHttpClient.HttpClientType.OkHttp ?
                    HttpClientType.OkHttp : HttpClientType.Default;
            fallbackHttpApi = createHttpApi(fallbackType, connectTimeoutMillis, readTimeoutMillis, useCaches, allowGzip, followRedirects);
        } else {
            fallbackHttpApi = null;
        }
        maxRetryTimes = CommonConfig.getHttpRetryTimes();
        retryIntervalMs = CommonConfig.getHttpRetryInterval();
    }

    private HttpApi createHttpApi(HttpClientType type,
                                  int connectTimeoutMillis,
                                  int readTimeoutMillis,
                                  boolean useCaches,
                                  boolean allowGzip,
                                  boolean followRedirects) {
        HttpApi httpApi = null;
        if (type == HttpClientType.OkHttp) {
            httpApi = new OkHttpApi(connectTimeoutMillis, readTimeoutMillis);
        } else if (type == HttpClientType.PCT) {
            try {
                Class.forName("com.pct.core.api.PctApi");
                Class<?> clazz = Class.forName("com.bitkernel.stream.rapid.http.PCTHttpApi");
                Constructor<?> constructor = clazz.getDeclaredConstructor(int.class);
                Object obj = constructor.newInstance(connectTimeoutMillis);
                httpApi = (HttpApi) obj;
            } catch (ClassNotFoundException | NoSuchMethodException | IllegalAccessException |
                     InvocationTargetException | InstantiationException e) {
                LogUtil.w(TAG, "pct is not implementation");
            }
        }
        if (httpApi == null) {
            httpApi = new DefaultHttpApi(connectTimeoutMillis, readTimeoutMillis, useCaches, allowGzip, followRedirects);
        }
        return httpApi;
    }

    public String post(@NonNull String url, @Nullable Map<String, String> headers, @NonNull String body) throws IOException {
        String response = null;
        IOException errorException = null;
        try {
            response = postWithRetry(httpApi, url, headers, body);
        } catch (IOException e) {
            errorException = e;
        }
        if (response != null) {
            return response;
        }
        if (fallbackHttpApi == null) {
            if (errorException != null) {
                throw new IOException(errorException);
            } else {
                return null;
            }
        }
        response = postWithRetry(fallbackHttpApi, url, headers, body);
        return response;
    }

    private String postWithRetry(@NonNull HttpApi api, @NonNull String url, @Nullable Map<String, String> headers, @NonNull String body) throws IOException {
        String response = null;
        int retryCount = Math.max(maxRetryTimes, 1);
        int sleepMs = Math.min(retryIntervalMs, 3000);
        IOException errorException = null;
        while (retryCount-- > 0) {
            try {
                response = api.post(url, headers, body);
            } catch (PctLockException e) {
                errorException = e;
                LogUtil.e(TAG, "postWithRetry error:" + e.getMessage());
                break;
            } catch (IOException e) {
                errorException = e;
                LogUtil.e(TAG, "postWithRetry error:" + e.getMessage());
            }
            if (response != null) {
                break;
            }
            if (sleepMs > 0) {
                try {
                    Thread.sleep(sleepMs);
                } catch (InterruptedException e) {
                    throw new IOException(e.getMessage());
                }
            }
        }
        if (response != null) {
            return response;
        }
        if (errorException != null) {
            throw new IOException(errorException);
        }
        return null;
    }

    public String get(@NonNull String url, @Nullable Map<String, String> headers) throws IOException {
        String response = null;
        IOException errorException = null;
        try {
            response = getWithRetry(httpApi, url, headers);
        } catch (IOException e) {
            errorException = e;
        }
        if (response != null) {
            return response;
        }
        if (fallbackHttpApi == null) {
            if (errorException != null) {
                throw new IOException(errorException);
            } else {
                return null;
            }
        }
        response = getWithRetry(fallbackHttpApi, url, headers);
        return response;
    }

    private String getWithRetry(@NonNull HttpApi api, @NonNull String url, @Nullable Map<String, String> headers) throws IOException {
        String response = null;
        int retryCount = Math.max(maxRetryTimes, 1);
        int sleepMs = Math.min(retryIntervalMs, 3000);
        IOException errorException = null;
        while (retryCount-- > 0) {
            try {
                response = api.get(url, headers);
            } catch (PctLockException e) {
                errorException = e;
                LogUtil.e(TAG, "getWithRetry error:" + e.getMessage());
                break;
            } catch (IOException e) {
                errorException = e;
                LogUtil.e(TAG, "getWithRetry error:" + e.getMessage());
            }
            if (response != null) {
                break;
            }
            if (sleepMs > 0) {
                try {
                    Thread.sleep(sleepMs);
                } catch (InterruptedException e) {
                    throw new IOException(e.getMessage());
                }
            }
        }
        if (response != null) {
            return response;
        }
        if (errorException != null) {
            throw new IOException(errorException);
        }
        return null;
    }
}
