package com.bitkernel.stream.rapid;

import static com.bitkernel.stream.rapid.RapidAgentUri.DRM_TYPE_NULL;
import static com.bitkernel.stream.rapid.player.RapidAgentPlayer.PLAYER_TYPE_AMP;
import static com.bitkernel.stream.rapid.player.RapidAgentPlayer.PLAYER_TYPE_EXO;
import static com.bitkernel.stream.rapid.player.RapidAgentPlayer.PLAYER_TYPE_IJK;
import static com.bitkernel.stream.rapid.player.RapidAgentPlayer.PLAYER_TYPE_UNKNOWN;

import android.content.Context;
import android.util.Log;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.bitkernel.stream.rapid.config.DeviceState;
import com.bitkernel.stream.rapid.player.DefaultPlayerBuilder;
import com.bitkernel.stream.rapid.player.PlayRecordManager;
import com.bitkernel.stream.rapid.player.PlayerBuilder;
import com.bitkernel.stream.rapid.player.PlayerConfig;
import com.bitkernel.stream.rapid.player.PlayerVersion;
import com.bitkernel.stream.rapid.player.RapidAgentPlayer;
import com.bitkernel.stream.rapid.player.RapidAgentStandardPlayer;
import com.bitkernel.stream.rapid.player.RapidAgentScalablePlayer;
import com.bitkernel.stream.rapid.player.RapidMediaPlayer;
import com.bitkernel.stream.rapid.player.RapidPlayerBuilder;
import com.bitkernel.stream.rapid.player.RapidPlayerBuilderCallback;
import com.bitkernel.stream.rapid.player.RapidPlayerStatistic;
import com.bitkernel.stream.rapid.utils.CommonUtil;
import com.bitkernel.stream.rapid.utils.LogUtil;
import com.bitkernel.stream.rapid.utils.SPUtil;

import java.util.HashMap;

/**
 * Rapid Agent Player SDK
 */
public final class RapidAgentPlayerSDK {
    private static final String TAG = "RapidAgentPlayerSDK";

    private static RapidPlayerBuilderCallback sPlayerBuilderCallback = RapidPlayerBuilderCallback.DEFAULT;

    /**
     * Gets the player version name
     *
     * @return player version name
     */
    public static String playerVersionName() {
        return PlayerVersion.getVersion();
    }

    /**
     * Gets the player version code
     *
     * @return player version code
     */
    public static int playerVersionCode() {
        String versionCode = PlayerVersion.getVersion();
        String[] versions = versionCode.split("_");
        return CommonUtil.parseInt(versions[0], 0);
    }

    static void init(Context context, RapidAgentStatistic statistic) {
        LogUtil.i(TAG, "start init rapid player");
        int h264Rank = SPUtil.getH264MediacodecRank();
        int h265Rank = SPUtil.getH265MediacodecRank();
        RapidMediaPlayer.preload(context, h264Rank, h265Rank);
        RapidPlayerStatistic.setAgentStatistic(statistic);
        DeviceState.setAvcAvailable(DefaultPlayerBuilder.isAvcMediacodecAvailable());
        DeviceState.setHevcAvailable(DefaultPlayerBuilder.isHevcMediacodecAvailable());
        SPUtil.setH264MediacodecRank(DefaultPlayerBuilder.getAvcMediacodecRank());
        SPUtil.setH265MediacodecRank(DefaultPlayerBuilder.getHevcMediacodecRank());
        LogUtil.i(TAG, "finish init rapid player");
    }

    /**
     * Sets default player builder {@link RapidPlayerBuilderCallback}
     *
     * @param callback callback
     */
    public static void setPlayerBuilderCallback(RapidPlayerBuilderCallback callback) {
        sPlayerBuilderCallback = callback;
    }

    /**
     * RapidAgentPlayer builder
     */
    public static class RapidAgentPlayerBuilder {
        private final Context context;
        private final RapidAgentUri rapidAgentUri;
        private int playerType = PLAYER_TYPE_UNKNOWN;
        private boolean enableStartStream = true;
        private boolean enableReleaseAsync = true;
        private boolean enableCacheDrmInfo = false;

        /**
         * RapidAgentPlayer constructor
         *
         * @param context       context
         * @param rapidAgentUri rapidAgentUri
         */
        public RapidAgentPlayerBuilder(@Nullable Context context, @NonNull RapidAgentUri rapidAgentUri) {
            this.context = context;
            this.rapidAgentUri = rapidAgentUri.copySelf();
        }

        /**
         * set player type
         *
         * @param playerType playerType, see
         *                   {@link com.bitkernel.stream.rapid.player.RapidAgentPlayer#PLAYER_TYPE_IJK},
         *                   {@link com.bitkernel.stream.rapid.player.RapidAgentPlayer#PLAYER_TYPE_EXO},
         *                   {@link com.bitkernel.stream.rapid.player.RapidAgentPlayer#PLAYER_TYPE_AMP},
         * @return the builder instance
         */
        public RapidAgentPlayerBuilder setPlayerType(int playerType) {
            this.playerType = playerType;
            return this;
        }

        /**
         * whether start the stream in player internal
         *
         * @param enableStartStream whether start the stream in player internal
         * @return the builder instance
         */
        public RapidAgentPlayerBuilder setEnableStartStream(boolean enableStartStream) {
            this.enableStartStream = enableStartStream;
            return this;
        }

        /**
         * whether enable async-release feature
         *
         * @param enableReleaseAsync whether enable async-release feature
         * @return the builder instance
         */
        public RapidAgentPlayerBuilder setEnableReleaseAsync(boolean enableReleaseAsync) {
            this.enableReleaseAsync = enableReleaseAsync;
            return this;
        }

        /**
         * whether enable cache drm info feature
         *
         * @param enableCacheDrmInfo whether enable cache drm info feature
         * @return the builder instance
         */
        public RapidAgentPlayerBuilder setEnableCacheDrmInfo(boolean enableCacheDrmInfo) {
            this.enableCacheDrmInfo = enableCacheDrmInfo;
            return this;
        }

        /**
         * Creates a media player {@link RapidMediaPlayer}
         *
         * @return media player instance
         */
        public RapidAgentPlayer build() {
            Log.i(TAG, "createMediaPlayer from " + toString());
            HashMap<String, String> params = new HashMap<>();
            if (rapidAgentUri.isAdaptiveBitrate()) {
                params.put(PlayerBuilder.ABR, "1");
            }
            if (rapidAgentUri.getDrmType() != DRM_TYPE_NULL) {
                params.put(PlayerBuilder.DRM, "1");
            }
            if (rapidAgentUri.isMicroBlock()) {
                params.put(PlayerBuilder.MLS, "1");
            }
            PlayRecordManager.getInstance().setForceUsingIjkInterval(PlayerConfig.getForceUsingIjkInterval());
            RapidPlayerStatistic playerStatistic = new RapidPlayerStatistic();
            RapidPlayerBuilder playerBuilder = sPlayerBuilderCallback.createPlayerBuilder(rapidAgentUri);
            playerBuilder.setDefaultPlayerType(PlayerConfig.getDefaultPlayerType());
            RapidAgentPlayer mediaPlayer;
            if (enableStartStream) {
                RapidAgentStandardPlayer standardPlayer = new RapidAgentStandardPlayer(context, playerType, params, playerStatistic, playerBuilder);
                standardPlayer.setRapidAgentUri(rapidAgentUri);
                standardPlayer.setEnableReleaseAsync(enableReleaseAsync);
                standardPlayer.setEnableCacheDrmInfo(rapidAgentUri.getDrmOfflineLicenseKeySetId() == null && enableCacheDrmInfo);
                standardPlayer.enableRebuildFeature(PlayerConfig.enableRebuildFeature() && playerType == PLAYER_TYPE_UNKNOWN);
                standardPlayer.setCustomRebuildEvent(PlayerConfig.getRebuildEvents());
                mediaPlayer = standardPlayer;
            } else {
                RapidAgentScalablePlayer scalablePlayer = new RapidAgentScalablePlayer(context, playerType, params, playerStatistic, playerBuilder);
                scalablePlayer.setRapidAgentUri(rapidAgentUri);
                scalablePlayer.enableRebuildFeature(PlayerConfig.enableRebuildFeature() && playerType == PLAYER_TYPE_UNKNOWN);
                scalablePlayer.setCustomRebuildEvent(PlayerConfig.getRebuildEvents());
                mediaPlayer = scalablePlayer;
            }
            Log.i(TAG, "createMediaPlayer finish playerType:" + mediaPlayer + "--" + getPlayerName(mediaPlayer.getPlayerType()));
            return mediaPlayer;
        }

        @NonNull
        @Override
        public String toString() {
            return "RapidAgentPlayerBuilder{" +
                    "context=" + context +
                    ", rapidAgentUri=" + rapidAgentUri +
                    ", playerType=" + playerType +
                    ", enableStartStream=" + enableStartStream +
                    ", enableReleaseAsync=" + enableReleaseAsync +
                    ", enableCacheDrmInfo=" + enableCacheDrmInfo +
                    '}';
        }
    }

    /**
     * Creates a media player {@link RapidAgentPlayer}
     *
     * @param context       context
     * @param rapidAgentUri rapidAgentUri
     * @return media player instance
     */
    public static RapidAgentPlayer createMediaPlayer(@Nullable Context context, @NonNull RapidAgentUri rapidAgentUri) {
        return new RapidAgentPlayerBuilder(context, rapidAgentUri).build();
    }

    /**
     * Gets the player name
     *
     * @param playerType the player type
     * @return the player name
     * @hidden
     */
    public static String getPlayerName(int playerType) {
        switch (playerType) {
            case PLAYER_TYPE_UNKNOWN:
                return "unknown";
            case PLAYER_TYPE_IJK:
                return "IjkMediaPlayer";
            case PLAYER_TYPE_EXO:
                return "ExoMediaPlayer";
            case PLAYER_TYPE_AMP:
                return "AndroidMediaPlayer";
        }
        return "invalid" + playerType;
    }
}
