package com.bitkernel.stream.rapid;

import android.content.Context;
import android.text.TextUtils;

import androidx.annotation.NonNull;

import com.bitkernel.stream.rapid.config.ConfigParser;
import com.bitkernel.stream.rapid.prt.DefaultJniPrtListener;
import com.bitkernel.stream.rapid.prt.PrtEngine;
import com.bitkernel.stream.rapid.prt.PrtVersion;
import com.bitkernel.stream.rapid.prt.StreamInfoMgr;
import com.bitkernel.stream.rapid.utils.CommonUtil;
import com.bitkernel.stream.rapid.utils.LogUtil;
import com.bitkernel.stream.rapid.utils.NetworkManager;
import com.stream.prt.NetworkState;
import com.stream.prt.PrtEvent;
import com.stream.prt.utils.JniApiImpl;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

/**
 * Rapid Agent SDK
 */
public final class RapidAgentSDK {
    private static final String TAG = "RapidAgentSDK";

    /**
     * Initialization callback
     */
    public interface OnInitListener {
        /**
         * Invoked when it is finished
         *
         * @param success whether init successfully
         * @param message init sdk detail
         */
        void onInit(boolean success, String message);
    }

    /**
     * RapidAgentSDK event callback
     */
    public interface OnPlayListener {
        /**
         * Invoked when receive new event
         *
         * @param event  event type
         * @param params special params of this event
         * @param desc   description of the event
         */
        void onPrtEvent(int event, String params, String desc);

        /**
         * Invoked when receive metric
         *
         * @param params metric detail
         */
        void onPrtMetric(Map<String, String> params);

        /**
         * Invoked after {@link RapidAgentSDK#setPrtEngineLog(boolean, int)}
         *
         * @param state state detail
         */
        void onPrtState(Map<String, String> state);

        /**
         * @hidden
         */
        default void onPrtProgress() {

        }

        /**
         * Invoked by prt engine to get player cache time
         *
         * @param channelId channel id
         * @return player cache time
         */
        default int getPrtPlayerCacheTime(int channelId) {
            return 0;
        }
    }

    private static class OnPlayListenerImpl implements OnPlayListener {
        private final OnPlayListener internalListener;

        public OnPlayListenerImpl(OnPlayListener internalListener) {
            this.internalListener = internalListener;
        }

        @Override
        public void onPrtEvent(int event, String params, String desc) {
            if (internalListener != null) {
                internalListener.onPrtEvent(event, params, desc);
            }
        }

        @Override
        public void onPrtMetric(Map<String, String> params) {
            if (internalListener != null) {
                internalListener.onPrtMetric(params);
            }
        }

        @Override
        public void onPrtState(Map<String, String> state) {
            if (internalListener != null) {
                internalListener.onPrtState(state);
            }
        }

        @Override
        public void onPrtProgress() {
            if (internalListener != null) {
                internalListener.onPrtProgress();
            }
        }

        @Override
        public int getPrtPlayerCacheTime(int channelId) {
            if (internalListener != null) {
                return internalListener.getPrtPlayerCacheTime(channelId);
            }
            return 0;
        }
    }

    /**
     * create channel fail
     */
    public static final int EVENT_CREATE_CHANNEL_FAIL = 0x900;
    /**
     * create channel timeout
     */
    public static final int EVENT_CREATE_CHANNEL_TIMEOUT = 0x901;
    /**
     * long time has no onMetric callback
     */
    public static final int EVENT_HAS_NO_METRIC = 0x902;

    /**
     * The http proxy has not read data for long time
     */
    public static final int EVENT_PROXY_NO_SIGNAL = PrtEvent.event_proxy_no_signa;
    /**
     * The http proxy fail to verify the caller identity
     */
    public static final int EVENT_PROXY_VERIFY = PrtEvent.event_proxy_verify;
    /**
     * Cant read data after change stream
     */
    public static final int EVENT_CHANGE_STREAM_NO_SIGNAL = PrtEvent.event_change_stream_no_signal;
    /**
     * Notify the player to restart after change stream
     */
    public static final int EVENT_CHANGE_STREAM_RESTART_PLAYER = PrtEvent.event_change_stream_restart_player;
    /**
     * Notify the app the prt server id has changed
     */
    public static final int EVENT_CHANGE_PRT_SERVER_ID = PrtEvent.event_change_prt_server_id;
    /**
     * Video buffer reach to max size
     */
    public static final int EVENT_VIDEO_BUFFER_REACH_MAX_SIZE = 0x2002;
    /**
     * The http proxy is ready and the player can start playing
     */
    public static final int EVENT_PROXY_READY = PrtEvent.event_front_start_player;
    /**
     * Notify app to create new channel because need to get channel info from tracker sometimes.
     *
     * @hidden
     */
    public static final int EVENT_FRONT_INFO_TO_PLAY = PrtEvent.event_front_info_to_play;
    /**
     * Fail to get index
     */
    public static final int EVENT_GET_INDEX_FAIL = /*PrtEvent.event_front_get_index_failed*/0x2007;
    /**
     * Notify the app that segment size and download time.
     */
    public static final int EVENT_SEGMENT_DOWNLOADED = PrtEvent.event_abr_info;
    /**
     * The connect token is expired
     */
    public static final int EVENT_CONNECT_TOKEN_EXPIRED = /*PrtEvent.event_connect_token_expired*/0x2101;
    /**
     * Micro block stream is not available
     */
    public static final int EVENT_MB_NOT_AVAILABLE = /*PrtEvent.event_live_micro_restart_index_chan*/0x2104;
    /**
     * Notify the app to restart the stream
     */
    public static final int EVENT_RESTART_STREAM = 0x100001;

    /**
     * The sdk is running on unknown device
     */
    public static final int DEVICE_PLATFORM_UNKNOWN = 0;
    /**
     * The sdk is running on box device
     */
    public static final int DEVICE_PLATFORM_BOX = 1;
    /**
     * The sdk is running on phone device
     */
    public static final int DEVICE_PLATFORM_PHONE = 2;

    /**
     * Ethernet connection
     */
    public static final int NETWORK_STATE_ETHERNET = 0;
    /**
     * Wifi connection
     */
    public static final int NETWORK_STATE_WIFI = 1;
    /**
     * Mobile connection
     */
    public static final int NETWORK_STATE_MOBILE = 2;
    /**
     * No network
     */
    public static final int NETWORK_STATE_DISCONNECTED = 3;
    /**
     * Unknown network state
     */
    public static final int NETWORK_STATE_UNKNOWN = 4;

    /**
     * default http client which use android system api
     */
    public static final int HTTP_CLIENT_TYPE_DEFAULT = 0;
    /**
     * ok http client
     */
    public static final int HTTP_CLIENT_TYPE_OKHTTP = 1;
    /**
     * pct http client
     */
    public static final int HTTP_CLIENT_TYPE_PCT = 2;

    private static volatile RapidAgentWorkThread sRapidAgentWorkThread;
    private static final Map<Integer, PrtEngine> sActiveStreamMap = new HashMap<>();
    private static boolean sHasLoadLib = false;

    /**
     * set so path
     *
     * @param bikSoPath BikCore so path which default as null
     * @param prtSoPath prt engine so path which default as null
     */
    public static void loadLib(String bikSoPath, String prtSoPath) {
        if (sHasLoadLib) {
            return;
        }
        if (bikSoPath == null) {
            bikSoPath = "";
        }
        if (prtSoPath == null) {
            prtSoPath = "";
        }
        LogUtil.i(TAG, "init module bik='" + bikSoPath + "', prt='" + prtSoPath + "'.");
        JniApiImpl.init(bikSoPath, prtSoPath);
        sHasLoadLib = true;
    }

    /**
     * set the work path which the files created by the sdk save to
     *
     * @param path work path
     */
    public static void initWorkPath(@NonNull String path) {
        RapidAgentConstant.initWorkPath(path);
    }

    /**
     * Initialize RapidAgentSDK
     *
     * @param context   application context
     * @param platform  device platform, see DEVICE_PLATFORM_*
     * @param appName   application package name
     * @param vnoId     VNO ID represent a VNO client, and got after register on our website
     * @param secretKey a key used to encrypt content, and got after register on our website
     * @param token     VNO login token
     * @param uid       user id which obtain after register to business system.
     * @param serverUrl server url to init the sdk
     * @param listener  a listener that notify the app if the action is successful.
     */
    public static void init(Context context, int platform, @NonNull String appName, @NonNull String vnoId, @NonNull String secretKey,
                            @NonNull String token, @NonNull String uid, @NonNull String serverUrl, OnInitListener listener) {
        init(context, platform, appName, vnoId, null, secretKey, token, uid, serverUrl, listener);
    }

    /**
     * Initialize RapidAgentSDK
     *
     * @param context   application context
     * @param platform  device platform, see DEVICE_PLATFORM_*
     * @param appName   application package name
     * @param vnoId     VNO ID represent a VNO client, and got after register on our website
     * @param vnoTag    VNO TAG represent a VNO client, and got after register on our website
     * @param secretKey a key used to encrypt content, and got after register on our website
     * @param token     VNO login token
     * @param uid       user id which obtain after register to business system.
     * @param serverUrl server url to init the sdk
     * @param listener  a listener that notify the app if the action is successful.
     */
    public static void init(Context context, int platform, @NonNull String appName, @NonNull String vnoId, String vnoTag,
                            @NonNull String secretKey, @NonNull String token, @NonNull String uid, @NonNull String serverUrl, OnInitListener listener) {
        LogUtil.i(TAG, "start init sdk version " + sdkVersionName() + ", vno:" + vnoId + ", debug:" + BuildConfig.DEBUG);

        if (TextUtils.isEmpty(appName)) {
            if (listener != null) {
                listener.onInit(false, "app name is invalid");
            }
            return;
        }
        if (TextUtils.isEmpty(vnoId)) {
            if (listener != null) {
                listener.onInit(false, "vno id is invalid");
            }
            return;
        }
        if (TextUtils.isEmpty(secretKey)) {
            if (listener != null) {
                listener.onInit(false, "secret key is invalid");
            }
            return;
        }
        if (TextUtils.isEmpty(token)) {
            if (listener != null) {
                listener.onInit(false, "token is invalid");
            }
            return;
        }
        if (TextUtils.isEmpty(uid)) {
            if (listener != null) {
                listener.onInit(false, "uid is invalid");
            }
            return;
        }
        if (TextUtils.isEmpty(serverUrl)) {
            if (listener != null) {
                listener.onInit(false, "server url is invalid");
            }
            return;
        }

        loadLib(null, null);
        if (sRapidAgentWorkThread != null) {
            sRapidAgentWorkThread.quitSafely();
            sRapidAgentWorkThread = null;
        }
        sRapidAgentWorkThread = new RapidAgentWorkThread(context, platform, appName, vnoId, vnoTag, secretKey, token, uid, serverUrl);
        sRapidAgentWorkThread.setListener(listener);
        sRapidAgentWorkThread.start();

        LogUtil.i(TAG, "waiting init work finished");
    }

    /**
     * Starts a new stream
     *
     * @param rapidAgentUri  a uri with special info, see {@link RapidAgentUri}
     * @param onPlayListener listener
     * @return the id of the new stream
     * @throws IOException throw exception when error occur
     */
    public static synchronized RapidAgentCall startStream(final RapidAgentUri rapidAgentUri, final OnPlayListener onPlayListener) throws IOException {
        if (!sRapidAgentWorkThread.hasLoginSuccess()) {
            throw new IOException("must init sdk first");
        }
        return startupEngine(rapidAgentUri, onPlayListener);
    }

    /**
     * Stops the current steam
     *
     * @param call the target stream should stopped
     */
    public static synchronized void stopStream(RapidAgentCall call) {
        if (call == null) {
            return;
        }
        if (call.getRequestIndex() > 0) {
            shutdownEngine(call.getRequestIndex());
            call.setRequestIndex(-1);
        }
        if (call.getCallIndex() > 0) {
            shutdownEngine(call.getCallIndex());
            call.setCallIndex(-1);
        }
        call.updateStatus(RapidAgentCall.Status.RELEASED);
    }

    /**
     * Requests new time shift data, used to seek
     *
     * @param call      the target stream
     * @param delayTime delay
     */
    public static synchronized void requestTimeShiftData(RapidAgentCall call, long delayTime) {
        if (call == null) {
            return;
        }
        requestTimeShiftData(call.getCallIndex(), delayTime);
    }

    /**
     * Requests new time shift data, used to seek
     *
     * @param streamId  the target stream
     * @param delayTime delay in seconds
     */
    public static synchronized void requestTimeShiftData(int streamId, long delayTime) {
        PrtEngine prtEngine = sActiveStreamMap.get(streamId);
        if (prtEngine != null) {
            prtEngine.requestTimeShiftData(delayTime);
        } else {
            LogUtil.e(TAG, "requestTimeShiftData:there is not an instance with streamId " + streamId);
        }
    }

    /**
     * Adds a prt listener
     *
     * @param streamId the target stream
     * @param listener prt listener
     */
    public static void addPrtListener(int streamId, OnPlayListener listener) {
        PrtEngine prtEngine = sActiveStreamMap.get(streamId);
        if (prtEngine != null) {
            prtEngine.addListener(listener);
        } else {
            LogUtil.e(TAG, "addPrtListener:there is not an instance with streamId " + streamId);
        }
    }

    /**
     * Remove a prt listener
     *
     * @param streamId the target stream
     * @param listener prt listener
     */
    public static void removePrtListener(int streamId, OnPlayListener listener) {
        PrtEngine prtEngine = sActiveStreamMap.get(streamId);
        if (prtEngine != null) {
            prtEngine.removeListener(listener);
        } else {
            LogUtil.e(TAG, "removePrtListener:there is not an instance with streamId " + streamId);
        }
    }

    /**
     * Sets the device network state, see NETWORK_STATE_*
     *
     * @param state network state
     */
    public static void setNetworkState(int state) {
        LogUtil.i(TAG, "setNetworkState " + state);
        switch (state) {
            case NETWORK_STATE_ETHERNET:
                PrtEngine.setNetworkState(NetworkState.ETHERNET, NetworkManager.getIpAddress());
                break;
            case NETWORK_STATE_WIFI:
                PrtEngine.setNetworkState(NetworkState.WIFI, NetworkManager.getIpAddress());
                break;
            case NETWORK_STATE_MOBILE:
                PrtEngine.setNetworkState(NetworkState.MOBILE, NetworkManager.getIpAddress());
                break;
            case NETWORK_STATE_DISCONNECTED:
                PrtEngine.setNetworkState(NetworkState.DISCONNECTED, NetworkManager.getIpAddress());
                break;
            case NETWORK_STATE_UNKNOWN:
                PrtEngine.setNetworkState(NetworkState.UNKNOWN, NetworkManager.getIpAddress());
                break;
        }
    }

    /**
     * Sets the fore/background when it is updated
     *
     * @param foreground it is on foreground now
     */
    public static void setForeBackgroundState(boolean foreground) {
        LogUtil.i(TAG, "setForeBackgroundState " + foreground);
        PrtEngine.setRunningMode(foreground ? 0 : 1);
    }

    /**
     * Send buffering msg to engine
     *
     * @param streamId the target stream
     * @param ms       duration in milliseconds
     */
    public static void notifyBuffering(int streamId, int ms) {
        PrtEngine prtEngine = sActiveStreamMap.get(streamId);
        if (prtEngine != null) {
            prtEngine.notifyBuffering(ms);
        } else {
            LogUtil.e(TAG, "notifyBuffering:there is not an instance with streamId " + streamId);
        }
    }

    /**
     * Sets prt engine log switch and period
     *
     * @param enable log switch
     * @param period log print period
     */
    public static void setPrtEngineLog(boolean enable, int period) {
        PrtEngine.setLog(enable, period);
    }

    public static void sendFrontLog(int streamId, String log) {
        PrtEngine prtEngine = sActiveStreamMap.get(streamId);
        if (prtEngine != null) {
            prtEngine.sendFrontLog(log);
        } else {
            LogUtil.e(TAG, "sendFrontLog:there is not an instance with streamId " + streamId);
        }
    }

    /**
     * Sets the device network state, see HTTP_CLIENT_TYPE_*
     *
     * @param type http client type
     */
    public static void setHttpClientType(int type) {
        switch (type) {
            case HTTP_CLIENT_TYPE_DEFAULT:
                RapidAgentHttpClient.setHttpClientType(RapidAgentHttpClient.HttpClientType.Default);
                break;
            case HTTP_CLIENT_TYPE_OKHTTP:
                RapidAgentHttpClient.setHttpClientType(RapidAgentHttpClient.HttpClientType.OkHttp);
                break;
            case HTTP_CLIENT_TYPE_PCT:
                RapidAgentHttpClient.setHttpClientType(RapidAgentHttpClient.HttpClientType.PCT);
                break;
        }
    }

    /**
     * Gets the sdk version name
     *
     * @return the sdk version name
     */
    public static String sdkVersionName() {
        return BuildConfig.VERSION_CODE + "_" + BuildConfig.GIT_HASH;
    }

    /**
     * Gets the engine version name
     *
     * @return the engine version name
     */
    public static String engineVersionName() {
        return PrtVersion.getVersionName();
    }

    private static synchronized RapidAgentCall startupEngine(final RapidAgentUri rapidAgentUri, final OnPlayListener onPlayListener) throws IOException {
        if (rapidAgentUri.isMicroBlock() && rapidAgentUri.getDrmType() == RapidAgentUri.DRM_TYPE_GOOSE) {
            rapidAgentUri.setMicroBlock(false, 0);
            LogUtil.w(TAG, "startupEngine: mb is not support goose drm now");
        }
        RapidAgentCall call = new RapidAgentCall(rapidAgentUri);
        if (rapidAgentUri instanceof RapidAgentLiveUri) {
            callEngine(rapidAgentUri, onPlayListener, call);
        } else {
            requestEngine(rapidAgentUri, onPlayListener, call);
        }
        return call;
    }

    private static synchronized void requestEngine(final RapidAgentUri rapidAgentUri, final OnPlayListener onPlayListener, RapidAgentCall call) throws IOException {
        LogUtil.i(TAG, "requestEngine agentUri=" + rapidAgentUri);
        call.updateStatus(RapidAgentCall.Status.REQUESTING);
        PrtEngine prtEngine = new PrtEngine();
        DefaultJniPrtListener jniPrtListener = new DefaultJniPrtListener();
        jniPrtListener.addPlayListener(new OnPlayListenerImpl(onPlayListener) {
            @Override
            public void onPrtEvent(int event, String params, String desc) {
                if (event == RapidAgentSDK.EVENT_FRONT_INFO_TO_PLAY) {
                    call.updateStatus(RapidAgentCall.Status.REQUESTED);
                    final Map<String, Object> frontInfo = CommonUtil.parseMapFromString(params);
                    PrtEngine.runOnWorkThread(() -> {
                        LogUtil.i(TAG, "start vod channel with front info");
                        rapidAgentUri.setFrontInfo(frontInfo);
                        shutdownEngine(call.getRequestIndex());
                        call.setRequestIndex(-1);
                        try {
                            callEngine(rapidAgentUri, onPlayListener, call);
                        } catch (IOException e) {
                            e.printStackTrace();
                        }
                        rapidAgentUri.setFrontInfo(null);
                    });
                } else {
                    if (onPlayListener != null) {
                        onPlayListener.onPrtEvent(event, params, desc);
                    }
                }
            }
        });
        jniPrtListener.addPlayListener(call);
        int channelId = prtEngine.startup(rapidAgentUri, jniPrtListener);
        call.setRequestIndex(channelId);
        if (channelId > 0) {
            if (sActiveStreamMap.containsKey(channelId)) {
                throw new IllegalStateException("There is already an instance with streamId " + channelId);
            }
            sActiveStreamMap.put(channelId, prtEngine);
            LogUtil.i(TAG, "requestEngine new active stream " + channelId + ", current stream size is " + sActiveStreamMap.size());
        } else {
            jniPrtListener.onEvent(channelId, EVENT_CREATE_CHANNEL_FAIL, "", "create channel fail");
        }
    }

    private static synchronized void callEngine(final RapidAgentUri rapidAgentUri, final OnPlayListener onPlayListener, RapidAgentCall call) throws IOException {
        LogUtil.i(TAG, "callEngine agentUri=" + rapidAgentUri);
        call.updateStatus(RapidAgentCall.Status.CALLING);
        PrtEngine prtEngine = new PrtEngine();
        DefaultJniPrtListener jniPrtListener = new DefaultJniPrtListener();
        jniPrtListener.addPlayListener(new OnPlayListenerImpl(onPlayListener) {
            @Override
            public void onPrtEvent(int event, String params, String desc) {
                if (event == RapidAgentSDK.EVENT_CONNECT_TOKEN_EXPIRED) {
                    LogUtil.i(TAG, "engine connect token is expired");
                    sRapidAgentWorkThread.triggerUpdate();
                    return;
                }
                if (onPlayListener != null) {
                    onPlayListener.onPrtEvent(event, params, desc);
                }
            }
        });
        jniPrtListener.addPlayListener(call);
        int channelId = prtEngine.startup(rapidAgentUri, jniPrtListener);
        call.setCallIndex(channelId);
        if (channelId > 0) {
            call.updateStatus(RapidAgentCall.Status.CONNECTED);
            if (rapidAgentUri instanceof RapidAgentCatchupUri || rapidAgentUri instanceof RapidAgentVodUri) {
                LogUtil.i(TAG, "notify a dummy proxy ready event");
                jniPrtListener.onEvent(channelId, RapidAgentSDK.EVENT_PROXY_READY, null, null);
            }
            StreamInfoMgr.getStreamInfo(channelId).startupTime = call.getStartTime();
            if (sActiveStreamMap.containsKey(channelId)) {
                throw new IllegalStateException("There is already an instance with streamId " + channelId);
            }
            sActiveStreamMap.put(channelId, prtEngine);
            LogUtil.i(TAG, "callEngine new active stream " + channelId + ", current stream size is " + sActiveStreamMap.size());
        } else {
            jniPrtListener.onEvent(channelId, EVENT_CREATE_CHANNEL_FAIL, "", "create channel fail");
        }
    }

    private static synchronized void shutdownEngine(int streamId) {
        PrtEngine prtEngine = sActiveStreamMap.remove(streamId);
        if (prtEngine != null) {
            prtEngine.shutdown();
            LogUtil.i(TAG, "shutdownEngine stream " + streamId + ", current stream size is " + sActiveStreamMap.size());
        } else {
            LogUtil.e(TAG, "shutdownEngine:there is not an instance with streamId " + streamId);
        }
    }

    static void notifyConfigChanged() throws IOException {
        LogUtil.i(TAG, "notifyConfigChanged");
        ConfigParser.printAllConfig();
        for (Map.Entry<Integer, PrtEngine> entry : sActiveStreamMap.entrySet()) {
            entry.getValue().updateToken();
        }
    }

    static void triggerLogcat(int what, int extra) {
        sRapidAgentWorkThread.triggerLogcat(what, extra);
    }
}
