package com.bitkernel.stream.rapid;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import java.util.Map;

/**
 * Representation of a vod stream
 */
public final class RapidAgentVodUri extends RapidAgentUri {
    /**
     * Constructs a new instance
     *
     * @param streamId the stream id
     */
    private RapidAgentVodUri(String streamId) {
        super(streamId);
    }

    /**
     * Creates a {@link RapidAgentVodUri} whose protocol is RPD_PORTAL
     *
     * @param streamId the stream id
     * @return RapidAgentVodUri
     */
    public static RapidAgentVodUri createRpdUri(@NonNull String streamId) {
        RapidAgentVodUri uri = new RapidAgentVodUri(streamId);
        uri.setProtocol(PROTOCOL_RPD_PORTAL);
        return uri;
    }

    /**
     * Creates a {@link RapidAgentVodUri} whose protocol is MPT_PORTAL
     *
     * @param streamId the stream id
     * @param routeId  the special info for back-to-resource
     * @param fileSize the file size of the stream
     * @return RapidAgentVodUri
     */
    public static RapidAgentVodUri createMptUri(@NonNull String streamId, @NonNull String routeId, long fileSize) {
        RapidAgentVodUri uri = new RapidAgentVodUri(streamId);
        uri.setRouteId(routeId);
        uri.customInfo.put("routeId", routeId);
        uri.customInfo.put("fsize", fileSize);
        uri.setProtocol(PROTOCOL_MPT_PORTAL);
        return uri;
    }

    /**
     * Creates a {@link RapidAgentVodUri} relate to net disk
     *
     * @param streamId   the stream id
     * @param fileSize   the size of vod file
     * @param customInfo must contain info such below
     *                   "customInfo": {
     *                      "resNetDisk": "REDBOXES",
     *                      "resTtNum": "tt number",
     *                      "fid": "test file id",
     *                      "fkey": "test fle key",
     *                      "netDiskSite": "REDBOXES",
     *                      "nduser": "test user",
     *                      "ndpwd": "test password",
     *                      "ndlgtyp": "test",
     *                      "pt": "mpq/mpt"
     *                   }
     * @return RapidAgentVodUri
     */
    public static RapidAgentVodUri createNetDiskUri(@NonNull String streamId, long fileSize, @Nullable Map<String, Object> customInfo) {
        if (customInfo == null || !customInfo.containsKey("pt")) {
            throw new IllegalArgumentException("custom info must contain pt");
        }
        Object protocol = customInfo.get("pt");
        if (!(protocol instanceof String)) {
            throw new IllegalArgumentException("custom info must contain invalid pt:" + protocol);
        }
        String protoString = (String) protocol;
        protoString = protoString.toLowerCase();
        if (protoString.contains("mpq")) {
            return createNetDiskMpqUri(streamId, fileSize, customInfo);
        }
        if (protoString.contains("mpt")) {
            return createNetDiskMptUri(streamId, fileSize, customInfo);
        }
        throw new IllegalArgumentException("custom info must contain invalid pt:" + protocol);
    }

    /**
     * Creates a {@link RapidAgentVodUri} whose protocol is mpt
     *
     * @param streamId   the stream id
     * @param fileSize   the size of vod file
     * @param customInfo must contain info such below
     *                   "customInfo": {
     *                      "resNetDisk": "REDBOXES",
     *                      "resTtNum": "tt number",
     *                      "fid": "test file id",
     *                      "fkey": "test fle key",
     *                      "netDiskSite": "REDBOXES",
     *                      "nduser": "test user",
     *                      "ndpwd": "test password",
     *                      "ndlgtyp": "test"
     *                   }
     * @return RapidAgentVodUri
     */
    public static RapidAgentVodUri createNetDiskMptUri(@NonNull String streamId, long fileSize, @Nullable Map<String, Object> customInfo) {
        RapidAgentVodUri uri = new RapidAgentVodUri(streamId);
        if (customInfo != null && !customInfo.isEmpty()) {
            uri.customInfo.putAll(customInfo);
        }
        uri.customInfo.put("fsize", fileSize);
        uri.setProtocol(PROTOCOL_MPT);
        return uri;
    }

    /**
     * Creates a {@link RapidAgentVodUri} whose protocol is mpq
     *
     * @param streamId   the stream id
     * @param fileSize   the size of vod file
     * @param customInfo must contain info such below
     *                   "customInfo": {
     *                      "resNetDisk": "REDBOXES",
     *                      "resTtNum": "tt number",
     *                      "fid": "test file id",
     *                      "fkey": "test fle key",
     *                      "netDiskSite": "REDBOXES",
     *                      "nduser": "test user",
     *                      "ndpwd": "test password",
     *                      "ndlgtyp": "test"
     *                   }
     * @return RapidAgentVodUri
     */
    public static RapidAgentVodUri createNetDiskMpqUri(@NonNull String streamId, long fileSize, @Nullable Map<String, Object> customInfo) {
        RapidAgentVodUri uri = new RapidAgentVodUri(streamId);
        if (customInfo != null && !customInfo.isEmpty()) {
            uri.customInfo.putAll(customInfo);
        }
        uri.customInfo.put("fsize", fileSize);
        uri.setProtocol(PROTOCOL_MPQ);
        return uri;
    }

    @Override
    public RapidAgentUri copySelf() {
        RapidAgentVodUri other = new RapidAgentVodUri(streamId);
        copySelf(other);
        return other;
    }

    @NonNull
    @Override
    public String toString() {
        return "RapidAgentVodUri{" +
                "streamId='" + streamId + '\'' +
                ", routeId=" + routeId +
                ", customInfo=" + customInfo +
                ", frontInfo='" + frontInfo + '\'' +
                ", drmType=" + drmType +
                '}';
    }
}
