package com.bitkernel.stream.rapid;

import android.Manifest;
import android.content.Context;
import android.content.pm.PackageManager;
import android.os.Handler;
import android.os.HandlerThread;
import android.os.Message;

import androidx.annotation.NonNull;
import androidx.core.content.ContextCompat;

import com.bitkernel.stream.rapid.config.CommonConfig;
import com.bitkernel.stream.rapid.config.ConfigParser;
import com.bitkernel.stream.rapid.config.DeviceState;
import com.bitkernel.stream.rapid.debug.DebugConfig;
import com.bitkernel.stream.rapid.debug.LogcatHelper;
import com.bitkernel.stream.rapid.prt.PrtConfig;
import com.bitkernel.stream.rapid.prt.PrtEngine;
import com.bitkernel.stream.rapid.prt.PrtStatistic;
import com.bitkernel.stream.rapid.prt.PrtVersion;
import com.bitkernel.stream.rapid.utils.LogUtil;
import com.bitkernel.stream.rapid.utils.NetworkManager;
import com.bitkernel.stream.rapid.utils.SPUtil;

import org.json.JSONException;

import java.io.IOException;
import java.lang.ref.WeakReference;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;

public final class RapidAgentWorkThread extends HandlerThread {
    private static class WorkHandler extends Handler {
        private static final int START = 100;
        private static final int INIT_WORK_DONE = 101;
        private static final int AUTH_WORK_DONE = 102;
        private static final int PARSE_WORK_DONE = 103;
        private static final int ERROR = 110;
        private static final int UPDATE = 120;

        private final RapidAgentWorkThread outside;

        private WorkHandler(RapidAgentWorkThread outside) {
            super(outside.getLooper());
            this.outside = outside;
        }

        @Override
        public void handleMessage(@NonNull Message msg) {
            switch (msg.what) {
                case START:
                    outside.init();
                    break;
                case INIT_WORK_DONE:
                    outside.auth();
                    break;
                case AUTH_WORK_DONE:
                    outside.parse((String) msg.obj);
                    break;
                case PARSE_WORK_DONE:
                    outside.success();
                    break;
                case ERROR:
                    outside.error(msg.obj);
                    break;
                case UPDATE:
                    outside.update();
                    break;
            }
        }
    }

    private static final String TAG = "RapidAgentWorkThread";

    private static final String[] REQUIRED_PERMISSIONS = new String[]{
            Manifest.permission.INTERNET,
            Manifest.permission.ACCESS_WIFI_STATE,
            Manifest.permission.ACCESS_NETWORK_STATE
    };

    private final WeakReference<Context> contextRef;
    private final int platform;
    private final String appName;
    private final String vnoId;
    private final String vnoTag;
    private final String secretKey;
    private final String token;
    private final String uid;
    private final String serverUrl;
    private Handler workHandler;

    private RapidAgentSDK.OnInitListener listener;
    private boolean hasLoginSuccess = false;
    private volatile boolean updatingToken = false;

    RapidAgentWorkThread(Context context, int platform, String appName, String vnoId, String vnoTag,
                         String secretKey, String token, String uid, String serverUrl) {
        super("RapidWorkThread");
        this.contextRef = new WeakReference<>(context);
        this.platform = platform;
        this.appName = appName;
        this.vnoId = vnoId;
        this.vnoTag = vnoTag;
        this.secretKey = secretKey;
        this.token = token;
        this.uid = uid;
        this.serverUrl = serverUrl;
    }

    @Override
    protected void onLooperPrepared() {
        super.onLooperPrepared();
        workHandler = new WorkHandler(this);
        workHandler.sendEmptyMessage(WorkHandler.START);
        LogUtil.i(TAG, "start initWork");
    }

    @Override
    public void run() {
        LogUtil.i(TAG, "enter RapidAgentWorkThread");
        super.run();
        LogUtil.i(TAG, "quit RapidAgentWorkThread");
    }

    public void setListener(RapidAgentSDK.OnInitListener listener) {
        this.listener = listener;
    }

    public boolean hasLoginSuccess() {
        return hasLoginSuccess;
    }

    public void triggerUpdate() {
        if (workHandler == null) {
            LogUtil.e(TAG, "please waiting for the work thread prepared");
            return;
        }
        if (updatingToken) {
            return;
        }
        workHandler.removeMessages(WorkHandler.UPDATE);
        workHandler.sendEmptyMessage(WorkHandler.UPDATE);
    }

    public void triggerLogcat(int what, int extra) {
        workHandler.post(() -> {
            boolean running = LogcatHelper.getInstance().startReadLogcat(RapidAgentConstant.getSdkLogDir(), "logcat.log", what, extra);
            if (running) {
                int logSeconds = DebugConfig.getLogcatSeconds();
                workHandler.postDelayed(() -> LogcatHelper.getInstance().stopReadLogcat(), logSeconds * 1000L);
            }
        });
    }

    private void init() {
        Context context = contextRef.get();
        if (context != null) {
            for (String permission : REQUIRED_PERMISSIONS) {
                if (ContextCompat.checkSelfPermission(context, permission) != PackageManager.PERMISSION_GRANTED) {
                    Message.obtain(workHandler, WorkHandler.ERROR, new IllegalAccessException(permission + " is requested")).sendToTarget();
                    return;
                }
            }
        }
        SPUtil.init(context);
        NetworkManager.init(context);
        RapidAgentConstant.init(context, platform, appName, vnoId, vnoTag, secretKey, token, uid);
        PrtEngine.init(RapidAgentConstant.getPrtWorkDir(), RapidAgentConstant.getPrtCfgFileName());
        PrtStatistic.setAgentStatistic(RapidAgentStatistic.getInstance());
        RapidAgentServer.init(serverUrl);
        int playerVersionCode = 0;
        int engineVersionCode = PrtVersion.getVersionCode();
        try {
            Class.forName("com.bitkernel.stream.rapid.player.RapidMediaPlayer");
            Class<?> clazz = Class.forName("com.bitkernel.stream.rapid.RapidAgentPlayerSDK");
            Method methodInit = clazz.getDeclaredMethod("init", Context.class, RapidAgentStatistic.class);
            methodInit.invoke(null, context, RapidAgentStatistic.getInstance());
            Method methodGetVersion = clazz.getDeclaredMethod("playerVersionCode");
            Integer playerVersionCodeObj = (Integer) methodGetVersion.invoke(null);
            if (playerVersionCodeObj != null) {
                playerVersionCode = playerVersionCodeObj;
            }
        } catch (ClassNotFoundException | NoSuchMethodException | IllegalAccessException |
                 InvocationTargetException e) {
            LogUtil.w(TAG, "RapidMediaPlayer is not implementation");
        }
        RapidAgentConstant.setVersions(playerVersionCode, engineVersionCode);
        LogUtil.i(TAG, "initWork done");
        workHandler.sendEmptyMessage(WorkHandler.INIT_WORK_DONE);
    }

    private void auth() {
        try {
            String response = RapidAgentServer.request();
            LogUtil.i(TAG, "authWork done");
            Message.obtain(workHandler, WorkHandler.AUTH_WORK_DONE, response).sendToTarget();
        } catch (IOException e) {
            e.printStackTrace();
            LogUtil.e(TAG, "authWork fail : " + e.getMessage());
            Message.obtain(workHandler, WorkHandler.ERROR, e).sendToTarget();
        }
    }

    private void parse(String json) {
        boolean success = false;
        Exception exception = null;
        try {
            success = ConfigParser.parseConfig(json);
            LogUtil.i(TAG, "parseConfigWork done " + success);
        } catch (JSONException e) {
            LogUtil.e(TAG, "parseConfigWork fail : " + e.getMessage());
            exception = e;
        }
        RapidAgentStatistic.init();
        try {
            Class.forName("com.pct.core.api.PctApi");
            Class<?> clazz = Class.forName("com.bitkernel.stream.rapid.pct.PctHandler");
            Method methodInit = clazz.getDeclaredMethod("init", RapidAgentStatistic.class);
            methodInit.invoke(null, RapidAgentStatistic.getInstance());
        } catch (ClassNotFoundException | NoSuchMethodException | IllegalAccessException |
                 InvocationTargetException e) {
            LogUtil.w(TAG, "pct is not implementation");
        }
        ConfigParser.printAllConfig();
        if (success && !PrtConfig.hasValidSecretKey()) {
            LogUtil.e(TAG, "parseConfigWork cant get valid EngineSecretKey");
            exception = new IllegalStateException("cant get valid EngineSecretKey");
            success = false;
        }
        if (success) {
            workHandler.sendEmptyMessage(WorkHandler.PARSE_WORK_DONE);
        } else {
            Message.obtain(workHandler, WorkHandler.ERROR, exception).sendToTarget();
        }
    }

    private void success() {
        LogUtil.i(TAG, "init sdk successfully");
        long delay = CommonConfig.serverRefreshInterval() * 1000L;
        workHandler.sendEmptyMessageDelayed(WorkHandler.UPDATE, delay);
        hasLoginSuccess = true;
        RapidAgentSDK.OnInitListener tmpListener = listener;
        if (tmpListener != null) {
            tmpListener.onInit(true, "success");
        }
        listener = null;
        if (!DeviceState.isAvcAvailable() || !DeviceState.isHevcAvailable()) {
            RapidAgentStatistic.getInstance().notifyLowMediaCodec(DeviceState.isAvcAvailable(), DeviceState.isHevcAvailable());
        }
        DeviceState.setLoginSuccess(true);
    }

    private void error(Object obj) {
        LogUtil.e(TAG, "init sdk fail");
        String desc;
        try {
            Exception exception = (Exception) obj;
            desc = exception.getClass().getSimpleName() + "," + exception.getMessage();
            exception.printStackTrace();
        } catch (Exception e) {
            desc = "unknown";
        }
        hasLoginSuccess = false;
        RapidAgentSDK.OnInitListener tmpListener = listener;
        if (tmpListener != null) {
            tmpListener.onInit(false, desc);
        }
        listener = null;
        RapidAgentStatistic.init();
        RapidAgentStatistic.getInstance().notifySDKInitError(desc);
        DeviceState.setLoginSuccess(false);
    }

    private void update() {
        LogUtil.d(TAG, "start server refresh");
        workHandler.removeMessages(WorkHandler.UPDATE);
        try {
            updatingToken = true;
            String response = RapidAgentServer.update();
            boolean changed = ConfigParser.parseConfig(response, true);
            if (changed) {
                RapidAgentSDK.notifyConfigChanged();
            } else {
                LogUtil.d(TAG, "server config is not updated");
            }
        } catch (IOException | JSONException e) {
            e.printStackTrace();
            LogUtil.e(TAG, "server refresh error: " + e.getMessage());
            RapidAgentStatistic.getInstance().notifySDKUpdateError(e.getMessage());
        } finally {
            updatingToken = false;
            long delay = CommonConfig.serverRefreshInterval() * 1000L;
            workHandler.sendEmptyMessageDelayed(WorkHandler.UPDATE, delay);
        }
    }
}
