package com.bitkernel.stream.rapid.player;

import static com.bitkernel.stream.rapid.utils.CommonUtil.parseChannelIdFromUrl;

import android.content.Context;
import android.os.Build;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.bitkernel.stream.rapid.RapidAgentSDK;
import com.bitkernel.stream.rapid.RapidAgentUri;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

public class RapidAgentScalablePlayer extends RapidMediaPlayer implements RapidAgentPlayer, RapidAgentSDK.OnPlayListener {
    private final RapidPlayerBuilder rapidPlayerBuilder;
    private int channelId;

    private OnStreamMetricListener streamMetricListener;
    private OnStreamStateListener streamStateListener;

    public RapidAgentScalablePlayer(@Nullable Context context, int playerType, Map<String, String> params,
                                    RapidPlayerStatistic playerStatistic, RapidPlayerBuilder playerBuilder) {
        super(context, playerType, params, playerStatistic, playerBuilder);
        this.rapidPlayerBuilder = playerBuilder;
    }

    public void setRapidAgentUri(@NonNull RapidAgentUri rapidAgentUri) {
        if (rapidAgentUri.getDrmType() == DRM_TYPE_WIDEVINE) {
            boolean supportMultiSession = Build.VERSION.SDK_INT > Build.VERSION_CODES.M;
            String drmLicenseUri = rapidAgentUri.getDrmLicenseUrl();
            Map<String, String> headers = new HashMap<>();
            if (rapidAgentUri.getDrmHeaders() != null) {
                headers.putAll(rapidAgentUri.getDrmHeaders());
            }
            setDrmInfo(DRM_TYPE_WIDEVINE, supportMultiSession, drmLicenseUri, headers, rapidAgentUri.getDrmMethod(), rapidAgentUri.getDrmOfflineLicenseKeySetId());
        }
    }

    @Override
    public void setDataSource(String path) throws IOException, IllegalArgumentException, SecurityException, IllegalStateException {
        super.setDataSource(path);
        channelId = parseChannelIdFromUrl(path);
        RapidAgentSDK.addPrtListener(channelId, this);
    }

    @Override
    public void release() {
        RapidAgentSDK.removePrtListener(channelId, this);
        super.release();
    }

    @Override
    public void setOnStreamMetricListener(OnStreamMetricListener listener) {
        streamMetricListener = listener;
    }

    @Override
    public void setOnStreamStateListener(OnStreamStateListener listener) {
        streamStateListener = listener;
    }

    @NonNull
    @Override
    public Map<String, Object> getPlayMetric() {
        return new HashMap<>();
    }

    @Override
    public void sendFrontLog(String msg) {

    }

    @Override
    public void onPrtEvent(int event, String params, String desc) {
        if (event == RapidAgentSDK.EVENT_SEGMENT_DOWNLOADED) {
            final SegmentDownloadedListener listener = rapidPlayerBuilder.getSegmentDownloadedListener();
            if (listener != null) {
                listener.onSegmentDownloaded(params);
            }
        }
    }

    @Override
    public void onPrtMetric(Map<String, String> params) {
        if (streamMetricListener != null) {
            streamMetricListener.onStreamMetric(params);
        }
    }

    @Override
    public void onPrtState(Map<String, String> state) {
        if (streamStateListener != null) {
            streamStateListener.onStreamState(state);
        }
    }
}
