package com.bitkernel.stream.rapid.player;

import android.content.Context;

import androidx.annotation.NonNull;

import com.bitkernel.stream.rapid.RapidAgentLiveUri;
import com.bitkernel.stream.rapid.RapidAgentUri;
import com.bitkernel.stream.rapid.prt.PrtEngine;
import com.bitkernel.stream.rapid.utils.LogUtil;

import java.lang.ref.WeakReference;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.Map;

import tv.danmaku.ijk.media.player.IMediaPlayer;
import tv.danmaku.ijk.media.player.IjkMediaPlayer;

public class RapidPlayerBuilderImpl extends DefaultPlayerBuilder implements RapidPlayerBuilder {
    private final String urlSuffix;
    private final WeakReference<RapidAgentUri> rapidAgentUriRef;
    private SegmentDownloadedListener segmentDownloadedListener;

    public RapidPlayerBuilderImpl(@NonNull RapidAgentUri rapidAgentUri) {
        this.urlSuffix = PrtEngine.getProxyUrlSuffix(rapidAgentUri);
        this.rapidAgentUriRef = new WeakReference<>(rapidAgentUri);
    }

    @Override
    public SegmentDownloadedListener getSegmentDownloadedListener() {
        return segmentDownloadedListener;
    }

    @Override
    public IMediaPlayer createIjkMediaPlayer() {
        IjkMediaPlayer ijkMediaPlayer = new IjkMediaPlayer();
        int dflLogLevel = PlayerConfig.getDefaultLogLevel();
        if (dflLogLevel >= IjkMediaPlayer.IJK_LOG_VERBOSE && dflLogLevel <= IjkMediaPlayer.IJK_LOG_SILENT) {
            RapidMediaPlayer.setLogLevel(dflLogLevel);
        } else {
            RapidMediaPlayer.setLogLevel(IjkMediaPlayer.IJK_LOG_INFO);
        }
        boolean ignoreOutOfOrderBlock = false;
        int outOfOrderLength = 0;

        RapidAgentUri rapidAgentUri = rapidAgentUriRef != null ? rapidAgentUriRef.get() : null;
        if (rapidAgentUri != null) {
            ijkMediaPlayer.setOption(IjkMediaPlayer.OPT_CATEGORY_PLAYER, "start-on-prepared", 1);
            ijkMediaPlayer.setOption(IjkMediaPlayer.OPT_CATEGORY_PLAYER, "framedrop", 1);
            ijkMediaPlayer.setOption(IjkMediaPlayer.OPT_CATEGORY_PLAYER, "audio-write-policy", 0);
            ijkMediaPlayer.setOption(IjkMediaPlayer.OPT_CATEGORY_FORMAT, "handle_codec_changed", 1);
            if (!(rapidAgentUri instanceof RapidAgentLiveUri)) {
                ijkMediaPlayer.setOption(IjkMediaPlayer.OPT_CATEGORY_FORMAT, "max_retry_times", 3);
            }
            if (rapidAgentUri.isMicroBlock()) {
                ijkMediaPlayer.setOption(IjkMediaPlayer.OPT_CATEGORY_PLAYER, "check-jitter-buffer", 30);
                ijkMediaPlayer.setOption(IjkMediaPlayer.OPT_CATEGORY_PLAYER, "jitter-buffer-buffering-packets", 2);
                ijkMediaPlayer.setOption(IjkMediaPlayer.OPT_CATEGORY_FORMAT, "mls-video-prior-frame", 2);
                ijkMediaPlayer.setOption(IjkMediaPlayer.OPT_CATEGORY_PLAYER, "jitter-buffer-cost-percent", 110);
                ijkMediaPlayer.setOption(IjkMediaPlayer.OPT_CATEGORY_PLAYER, "jitter-buffer-max-percent", 30);
                ijkMediaPlayer.setOption(IjkMediaPlayer.OPT_CATEGORY_PLAYER, "jitter-buffer-min-percent", 20);
                ijkMediaPlayer.setOption(IjkMediaPlayer.OPT_CATEGORY_PLAYER, "jitter-buffer-safe-multiple", 16);
                ijkMediaPlayer.setOption(IjkMediaPlayer.OPT_CATEGORY_PLAYER, "jitter-buffer-capacity-packets", 300);
                if (rapidAgentUri.getMicroBlockLevel() == RapidAgentUri.MICRO_BLOCK_ULTRA_LOW) {
                    ijkMediaPlayer.setOption(IjkMediaPlayer.OPT_CATEGORY_PLAYER, "jitter-buffer-safe-frames", 20);
                    ijkMediaPlayer.setOption(IjkMediaPlayer.OPT_CATEGORY_PLAYER, "jitter-buffer-target-frames", 30);
                    ijkMediaPlayer.setOption(IjkMediaPlayer.OPT_CATEGORY_FORMAT, "mls-fragment-interval", 1800);
                    ignoreOutOfOrderBlock = true;
                    outOfOrderLength = 5;
                } else if (rapidAgentUri.getMicroBlockLevel() == RapidAgentUri.MICRO_BLOCK_LOW) {
                    ijkMediaPlayer.setOption(IjkMediaPlayer.OPT_CATEGORY_PLAYER, "jitter-buffer-safe-frames", 30);
                    // ijkMediaPlayer.setOption(IjkMediaPlayer.OPT_CATEGORY_PLAYER, "jitter-buffer-target-frames", 80);
                    ijkMediaPlayer.setOption(IjkMediaPlayer.OPT_CATEGORY_FORMAT, "mls-fragment-interval", 60);
                } else if (rapidAgentUri.getMicroBlockLevel() == RapidAgentUri.MICRO_BLOCK_NORMAL) {
                    ijkMediaPlayer.setOption(IjkMediaPlayer.OPT_CATEGORY_PLAYER, "jitter-buffer-cost-percent", 200);
                    ijkMediaPlayer.setOption(IjkMediaPlayer.OPT_CATEGORY_PLAYER, "jitter-buffer-safe-frames", 300);
                    ijkMediaPlayer.setOption(IjkMediaPlayer.OPT_CATEGORY_PLAYER, "jitter-buffer-max-percent", 80);
                    ijkMediaPlayer.setOption(IjkMediaPlayer.OPT_CATEGORY_PLAYER, "jitter-buffer-min-percent", 60);
                    ijkMediaPlayer.setOption(IjkMediaPlayer.OPT_CATEGORY_PLAYER, "jitter-buffer-safe-multiple", 0);
                    ijkMediaPlayer.setOption(IjkMediaPlayer.OPT_CATEGORY_PLAYER, "jitter-buffer-enough-packets", 100);
                    ijkMediaPlayer.setOption(IjkMediaPlayer.OPT_CATEGORY_PLAYER, "jitter-buffer-capacity-packets", 500);
                    ijkMediaPlayer.setOption(IjkMediaPlayer.OPT_CATEGORY_FORMAT, "mls-fragment-interval", 20);
                    ijkMediaPlayer.setOption(IjkMediaPlayer.OPT_CATEGORY_FORMAT, "mls_read_sleep", 200);
                } else if (rapidAgentUri.getMicroBlockLevel() == RapidAgentUri.MICRO_BLOCK_GOP3) {
                    ijkMediaPlayer.setOption(IjkMediaPlayer.OPT_CATEGORY_PLAYER, "jitter-buffer-cost-percent", 200);
                    ijkMediaPlayer.setOption(IjkMediaPlayer.OPT_CATEGORY_PLAYER, "jitter-buffer-safe-frames", 90);
                    ijkMediaPlayer.setOption(IjkMediaPlayer.OPT_CATEGORY_PLAYER, "jitter-buffer-max-percent", 80);
                    ijkMediaPlayer.setOption(IjkMediaPlayer.OPT_CATEGORY_PLAYER, "jitter-buffer-min-percent", 60);
                    ijkMediaPlayer.setOption(IjkMediaPlayer.OPT_CATEGORY_PLAYER, "jitter-buffer-safe-multiple", 0);
                    ijkMediaPlayer.setOption(IjkMediaPlayer.OPT_CATEGORY_PLAYER, "jitter-buffer-enough-packets", 100);
                    ijkMediaPlayer.setOption(IjkMediaPlayer.OPT_CATEGORY_FORMAT, "mls-fragment-interval", 20);
                    ijkMediaPlayer.setOption(IjkMediaPlayer.OPT_CATEGORY_FORMAT, "mls_read_sleep", 200);
                }
                if (rapidAgentUri.isAdaptiveBitrate()) {
                    ijkMediaPlayer.setOption(IjkMediaPlayer.OPT_CATEGORY_PLAYER, "swithc-ideal-stream-percentile", 0);
                    ijkMediaPlayer.setOption(IjkMediaPlayer.OPT_CATEGORY_PLAYER, "swithc-ideal-stream-duration", 0);
                }
                // ijkMediaPlayer.setOption(IjkMediaPlayer.OPT_CATEGORY_FORMAT, "enable_dump_file", 9981);
            }
        }

        HashMap<Integer, Map<String, Object>> optionMap = new HashMap<>();
        optionMap.put(IjkMediaPlayer.OPT_CATEGORY_FORMAT, PlayerConfig.getIjkFormatOptions());
        optionMap.put(IjkMediaPlayer.OPT_CATEGORY_CODEC, PlayerConfig.getIjkCodecOptions());
        optionMap.put(IjkMediaPlayer.OPT_CATEGORY_SWS, PlayerConfig.getIjkSwsOptions());
        optionMap.put(IjkMediaPlayer.OPT_CATEGORY_PLAYER, PlayerConfig.getIjkPlayerOptions());
        if (rapidAgentUri != null && rapidAgentUri.isMicroBlock()) {
            Map<Integer, Map<String, Object>> jitterBufferMap = PlayerConfig.getJitterBufferConfig();
            Map<String, Object> jitterBufferConfig = jitterBufferMap.get(rapidAgentUri.getMicroBlockLevel());
            if (jitterBufferConfig != null) {
                Map<String, Object> playerMap = optionMap.get(IjkMediaPlayer.OPT_CATEGORY_PLAYER);
                if (playerMap != null) {
                    playerMap.putAll(jitterBufferConfig);
                } else {
                    optionMap.put(IjkMediaPlayer.OPT_CATEGORY_PLAYER, jitterBufferConfig);
                }
            }
        }
        for (Map.Entry<Integer, Map<String, Object>> optionMapEntry : optionMap.entrySet()) {
            int category = optionMapEntry.getKey();
            Map<String, Object> options = optionMapEntry.getValue();
            for (Map.Entry<String, Object> optionEntry : options.entrySet()) {
                String key = optionEntry.getKey();
                Object value = optionEntry.getValue();
                if (value instanceof String) {
                    ijkMediaPlayer.setOption(category, key, (String) value);
                } else if (value instanceof Long) {
                    ijkMediaPlayer.setOption(category, key, (Long) value);
                } else if (value instanceof Integer) {
                    ijkMediaPlayer.setOption(category, key, (Integer) value);
                }
            }
        }

        if (rapidAgentUri != null) {
            if (rapidAgentUri instanceof RapidAgentLiveUri) {
                ijkMediaPlayer.setOption(IjkMediaPlayer.OPT_CATEGORY_PLAYER, "live-quick-start", 1);
            } else {
                ijkMediaPlayer.setOption(IjkMediaPlayer.OPT_CATEGORY_PLAYER, "live-quick-start", 0);
            }
            if (rapidAgentUri.isAdaptiveBitrate()) {
                ijkMediaPlayer.setOption(IjkMediaPlayer.OPT_CATEGORY_PLAYER, "is-abr-source", 1);
                ijkMediaPlayer.setOption(IjkMediaPlayer.OPT_CATEGORY_FORMAT, "first_variant_only", 1);
            } else {
                ijkMediaPlayer.setOption(IjkMediaPlayer.OPT_CATEGORY_PLAYER, "is-abr-source", 0);
                ijkMediaPlayer.setOption(IjkMediaPlayer.OPT_CATEGORY_FORMAT, "first_variant_only", 0);
            }
            if (!rapidAgentUri.isMicroBlock()) {
                ijkMediaPlayer.setOption(IjkMediaPlayer.OPT_CATEGORY_PLAYER, "check-jitter-buffer", 0);
            }
        }

        RapidIjkBandwidthMeter bandwidthMeter = new RapidIjkBandwidthMeter(ignoreOutOfOrderBlock, outOfOrderLength);
        segmentDownloadedListener = bandwidthMeter;
        ijkMediaPlayer.setBandwidthMeter(bandwidthMeter);
        return ijkMediaPlayer;
    }

    @Override
    public IMediaPlayer createExoMediaPlayer(@NonNull Context context) {
        try {
            Class.forName("tv.danmaku.ijk.media.exo.IjkExoMediaPlayer");
            Class<?> clazz = Class.forName("com.bitkernel.stream.rapid.player.RapidExoMediaPlayer");
            Constructor<?> constructor = clazz.getDeclaredConstructor(Context.class, String.class);
            Object obj = constructor.newInstance(context, urlSuffix);
            Method method = clazz.getDeclaredMethod("getBandwidthMeter");
            segmentDownloadedListener = (SegmentDownloadedListener) method.invoke(obj);
            LogUtil.w("RapidPlayerBuilderImpl", "ExoPlayer bandwidth meter is: " + segmentDownloadedListener);
            return (IMediaPlayer) obj;
        } catch (ClassNotFoundException | NoSuchMethodException | IllegalAccessException |
                 InstantiationException | InvocationTargetException e) {
            LogUtil.w("RapidPlayerBuilderImpl", "ExoPlayer is no implementation");
            return createIjkMediaPlayer();
        }
    }
}
