package com.bitkernel.stream.rapid.prt;

import static com.bitkernel.stream.rapid.utils.CommonUtil.parseServerId;

import android.os.Handler;
import android.os.HandlerThread;
import android.os.Message;
import android.text.TextUtils;

import androidx.annotation.NonNull;

import com.bitkernel.stream.rapid.RapidAgentSDK;
import com.bitkernel.stream.rapid.config.CommonConfig;
import com.bitkernel.stream.rapid.utils.LogUtil;
import com.stream.prt.JniPrtListener;
import com.stream.prt.PrtEvent;
import com.stream.prt.PrtMetric;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;

public final class DefaultJniPrtListener implements JniPrtListener {
    private static final String TAG = "DefaultJniPrtListener";

    private static final HandlerThread WORK_THREAD;

    static {
        WORK_THREAD = new HandlerThread("DefaultJniPrtListener") {
            @Override
            public void run() {
                LogUtil.i(TAG, "enter work thread");
                super.run();
                LogUtil.i(TAG, "quit work thread");
            }
        };
        WORK_THREAD.start();
    }

    private static String sPrtServerId;
    private static String sTrackServerId;

    private final List<RapidAgentSDK.OnPlayListener> playListenerList;

    private int channelId;
    private String proxyUrl;
    private final AtomicBoolean hasNotifyFront = new AtomicBoolean();
    private final AtomicBoolean hasNotifyReady = new AtomicBoolean();
    private final AtomicInteger waitingNewToken = new AtomicInteger(0);

    private static final int MSG_STARTUP_TIMEOUT = 1;
    private static final int MSG_TIMER = 2;
    private final Handler workHandler;

    public DefaultJniPrtListener() {
        this.playListenerList = new CopyOnWriteArrayList<>();
        this.workHandler = new Handler(WORK_THREAD.getLooper()) {
            @Override
            public void handleMessage(@NonNull Message msg) {
                if (msg.what == MSG_STARTUP_TIMEOUT) {
                    onEvent(channelId, RapidAgentSDK.EVENT_CREATE_CHANNEL_TIMEOUT, "", "startup timeout");
                } else if (msg.what == MSG_TIMER) {
                    removeMessages(MSG_TIMER);
                    sendEmptyMessageDelayed(MSG_TIMER, 10_000);
                    onProgress();
                }
            }
        };
        this.workHandler.sendEmptyMessageDelayed(MSG_STARTUP_TIMEOUT, CommonConfig.engineStartupTimeout() * 1000L);
    }

    public void addPlayListener(RapidAgentSDK.OnPlayListener listener) {
        if (playListenerList.contains(listener)) {
            return;
        }
        playListenerList.add(listener);
        LogUtil.i(TAG, "addPlayListener " + listener + ", current size=" + playListenerList.size());
    }

    public void removePlayListener(RapidAgentSDK.OnPlayListener listener) {
        if (!playListenerList.contains(listener)) {
            return;
        }
        playListenerList.remove(listener);
        LogUtil.i(TAG, "removePlayListener " + listener + ", current size=" + playListenerList.size());
    }

    public void removeAllPlayListener() {
        playListenerList.clear();
        LogUtil.i(TAG, "removeAllPlayListener current size=" + playListenerList.size());
    }

    public void setChannelId(int channelId) {
        this.channelId = channelId;
    }

    public void setProxyUrl(String proxyUrl) {
        this.proxyUrl = proxyUrl;
        this.hasNotifyReady.set(false);
    }

    @Override
    public int onDataAvail(int channId, int blockId, byte[] data) {
        LogUtil.v(TAG, "onDataAvail 1");
        return 0;
    }

    @Override
    public int onDataAvail(int channId, int blockId, byte[] data, int offset, int len) {
        LogUtil.v(TAG, "onDataAvail 2");
        return 0;
    }

    @Override
    public int onDataAvail(int channId, int blockId, byte[] data, int offset, int len, long timeout) {
        LogUtil.v(TAG, "onDataAvail 3");
        return 0;
    }

    @Override
    public int onDataAvail(int channelId, int requestId, int blockId, byte[] data, long offsetInFile, int timeout) {
        LogUtil.v(TAG, "onDataAvail 4");
        return 0;
    }

    @Override
    public int onCheckRecvDataBuffer(int requestId, int blockId, int len, long offset) {
        LogUtil.d(TAG, "onCheckRecvDataBuffer");
        return 0;
    }

    @Override
    public int onMetric(int var1, PrtMetric metric) {
        LogUtil.d(TAG, "prtMetric");
        return 0;
    }

    @Override
    public void onVersion(String version) {
        LogUtil.d(TAG, "onVersion");
    }

    @Override
    public void onNatReq(int channelId, String externHost, int port) {
        LogUtil.d(TAG, "onNatReq");
    }

    @Override
    public int getPlayerCacheTime(int channelId) {
        LogUtil.d(TAG, "getPlayerCacheTime");
        int cacheTime = 0;
        List<RapidAgentSDK.OnPlayListener> tmpList = new ArrayList<>(playListenerList);
        for (int i = tmpList.size() - 1; i >= 0; i--) {
            RapidAgentSDK.OnPlayListener playListener = tmpList.get(i);
            cacheTime = playListener.getPrtPlayerCacheTime(channelId);
            if (cacheTime > 0) {
                break;
            }
        }
        return cacheTime;
    }

    @Override
    public int onState(int channelId, Map<String, String> state) {
        List<RapidAgentSDK.OnPlayListener> tmpList = new ArrayList<>(playListenerList);
        for (RapidAgentSDK.OnPlayListener playListener : tmpList) {
            playListener.onPrtState(state);
        }
        return 0;
    }

    @Override
    public void onEvent(int channelId, int event, String params, String desc) {
        LogUtil.d(TAG, "onEvent channelId=" + channelId + ",event=" + event + ",params=" + params + ",desc=" + desc);
        if (event == PrtEvent.event_switch_chan_continue || event == 0x2006/* event_start_fast_keypoints, 起播的关键点, 前端认为metric不好区分要求通过event上报 */) {
            //do not report internal event to app
            return;
        }
        if (channelId != this.channelId) {
            return;
        }
        switch (event) {
            case RapidAgentSDK.EVENT_FRONT_INFO_TO_PLAY:
                if (!hasNotifyFront.compareAndSet(false, true)) {
                    LogUtil.w(TAG, "has notify front message");
                    return;
                }
                break;
            case RapidAgentSDK.EVENT_PROXY_READY:
                if (!hasNotifyReady.compareAndSet(false, true)) {
                    LogUtil.w(TAG, "has notify ready message");
                    return;
                }
                StreamInfoMgr.StreamInfo streamInfo = StreamInfoMgr.getStreamInfo(channelId);
                streamInfo.engineReadyTime = System.nanoTime();
                params = proxyUrl;
                workHandler.removeMessages(MSG_STARTUP_TIMEOUT);
                workHandler.sendEmptyMessageDelayed(MSG_TIMER, 13_000);
                break;
            case RapidAgentSDK.EVENT_CHANGE_PRT_SERVER_ID:
                params = parseServerId(params);
                desc = sTrackServerId;
                sPrtServerId = params;
                break;
            case RapidAgentSDK.EVENT_CONNECT_TOKEN_EXPIRED:
                waitingNewToken.compareAndSet(0, 1);
                if (waitingNewToken.incrementAndGet() == 30) {
                    //already update the token, but engine still cant start the stream
                    event = RapidAgentSDK.EVENT_RESTART_STREAM;
                    params = null;
                    desc = null;
                }
                break;
            case RapidAgentSDK.EVENT_CHANGE_STREAM_RESTART_PLAYER:
                if (!hasNotifyReady.get()) {
                    LogUtil.w(TAG, "should not notify restart player msg before ready");
                    return;
                }
            case RapidAgentSDK.EVENT_CREATE_CHANNEL_FAIL:
            case RapidAgentSDK.EVENT_CREATE_CHANNEL_TIMEOUT:
            case RapidAgentSDK.EVENT_PROXY_NO_SIGNAL:
            case RapidAgentSDK.EVENT_CHANGE_STREAM_NO_SIGNAL:
            case RapidAgentSDK.EVENT_PROXY_VERIFY:
            case RapidAgentSDK.EVENT_GET_INDEX_FAIL:
                workHandler.removeMessages(MSG_STARTUP_TIMEOUT);
                break;
        }
        List<RapidAgentSDK.OnPlayListener> tmpList = new ArrayList<>(playListenerList);
        for (RapidAgentSDK.OnPlayListener playListener : tmpList) {
            playListener.onPrtEvent(event, params, desc);
        }
    }

    @Override
    public int onMetric(int channelId, Map<String, String> params) {
        LogUtil.d(TAG, "onMetric " + params);
        String serverId = PrtConfig.getTrackerServerId(params.get("trAddr"));
        List<RapidAgentSDK.OnPlayListener> tmpList = new ArrayList<>(playListenerList);
        for (RapidAgentSDK.OnPlayListener playListener : tmpList) {
            if (!TextUtils.equals(sTrackServerId, serverId)) {
                playListener.onPrtEvent(RapidAgentSDK.EVENT_CHANGE_PRT_SERVER_ID, sPrtServerId, serverId);
            }
            playListener.onPrtMetric(params);
        }
        sTrackServerId = serverId;
        return 0;
    }

    public void onProgress() {
        List<RapidAgentSDK.OnPlayListener> tmpList = new ArrayList<>(playListenerList);
        for (RapidAgentSDK.OnPlayListener playListener : tmpList) {
            playListener.onPrtProgress();
        }
    }

    public void release() {
        workHandler.removeCallbacksAndMessages(null);
    }

    public void onTokenUpdated() {
        waitingNewToken.compareAndSet(1, 2);
    }
}
