package com.bitkernel.stream.rapid.prt;

import static com.bitkernel.stream.rapid.prt.PrtEngine.SOLUTION_TYPE_LIVE_DRM_DASH;
import static com.bitkernel.stream.rapid.prt.PrtEngine.SOLUTION_TYPE_LIVE_DRM_HLS;
import static com.bitkernel.stream.rapid.prt.PrtEngine.SOLUTION_TYPE_LIVE_PLAIN_RPD;
import static com.bitkernel.stream.rapid.prt.PrtEngine.SOLUTION_TYPE_LIVE_VALO_HLS;
import static com.bitkernel.stream.rapid.prt.PrtEngine.SOLUTION_TYPE_NONE;
import static com.bitkernel.stream.rapid.prt.PrtEngine.SOLUTION_TYPE_VOD_DRM_DASH;
import static com.bitkernel.stream.rapid.prt.PrtEngine.SOLUTION_TYPE_VOD_DRM_HLS;
import static com.bitkernel.stream.rapid.prt.PrtEngine.SOLUTION_TYPE_VOD_MPQ;
import static com.bitkernel.stream.rapid.prt.PrtEngine.SOLUTION_TYPE_VOD_MPT;
import static com.bitkernel.stream.rapid.prt.PrtEngine.SOLUTION_TYPE_VOD_PLAIN_RPD;
import static com.bitkernel.stream.rapid.prt.PrtEngine.SOLUTION_TYPE_VOD_TIME_SHIFT_DRM_DASH;
import static com.bitkernel.stream.rapid.prt.PrtEngine.SOLUTION_TYPE_VOD_TIME_SHIFT_DRM_HLS;
import static com.bitkernel.stream.rapid.prt.PrtEngine.SOLUTION_TYPE_VOD_TIME_SHIFT_PLAIN_RPD;
import static com.bitkernel.stream.rapid.prt.PrtEngine.translate;

import android.text.TextUtils;

import androidx.annotation.NonNull;

import com.bitkernel.stream.rapid.RapidAgentCatchupUri;
import com.bitkernel.stream.rapid.RapidAgentConstant;
import com.bitkernel.stream.rapid.RapidAgentUri;
import com.bitkernel.stream.rapid.RapidAgentVodUri;
import com.bitkernel.stream.rapid.config.ServerConfig;
import com.bitkernel.stream.rapid.utils.AES;
import com.bitkernel.stream.rapid.utils.LogUtil;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import java.util.Random;

class TokenGenerator {
    private String TOKEN_SECRET_KEY;

    void setTokenSecretKey(String tokenSecretKey) {
        TOKEN_SECRET_KEY = tokenSecretKey;
    }

    String generatePlayToken(String channelId, int solutionType, int solutionBrand, RapidAgentUri rapidAgentUri) throws IOException {
        if (solutionType == SOLUTION_TYPE_VOD_MPQ ||
                solutionType == SOLUTION_TYPE_VOD_MPT ||
                solutionType == SOLUTION_TYPE_VOD_PLAIN_RPD ||
                solutionType == SOLUTION_TYPE_VOD_DRM_HLS ||
                solutionType == SOLUTION_TYPE_VOD_DRM_DASH) {
            return generateVodToken(channelId, ServerConfig.getPlayToken(), solutionType, solutionBrand, rapidAgentUri);
        } else if (solutionType == SOLUTION_TYPE_VOD_TIME_SHIFT_PLAIN_RPD ||
                solutionType == SOLUTION_TYPE_VOD_TIME_SHIFT_DRM_HLS ||
                solutionType == SOLUTION_TYPE_VOD_TIME_SHIFT_DRM_DASH) {
            return generateTimeShiftToken(channelId, ServerConfig.getPlayToken(), solutionType, solutionBrand, (RapidAgentCatchupUri) rapidAgentUri);
        } else {
            return generateCommonToken(channelId, ServerConfig.getPlayToken(), solutionType, solutionBrand, rapidAgentUri);
        }
    }

    String generateCommonToken(String channelId, String brtToken, int solutionType, int solutionBrand, @NonNull RapidAgentUri rapidAgentUri) {
        Map<String, String> params = new HashMap<>();
        params.put("TOKEN", brtToken);
        params.put("CODE", "/" + channelId);
        if (rapidAgentUri.getPlayStreamId() != null) {
            params.put("STREAM_ID", rapidAgentUri.getPlayStreamId());
        } else {
            params.put("STREAM_ID", channelId);
        }
        params.put("CHNID", channelId);
        if (rapidAgentUri.getFormat() >= 0) {
            params.put("FORMAT", String.valueOf(rapidAgentUri.getFormat()));
        } else if (rapidAgentUri.isAdaptiveBitrate() || (rapidAgentUri instanceof RapidAgentCatchupUri)) {
            params.put("FORMAT", "1");
        }
        if (!TextUtils.isEmpty(rapidAgentUri.getRouteId())) {
            params.put("ROUTE_ID", rapidAgentUri.getRouteId());
        }
        params.put("VNO_ID", RapidAgentConstant.getVnoId());
        return generateToken(solutionType, solutionBrand, params);
    }

    String generateVodToken(String channelId, String brtToken, int solutionType, int solutionBrand, @NonNull RapidAgentUri rapidAgentUri) throws IOException {
        Map<String, String> params = new HashMap<>();
        String playType = PrtConfig.getPlaySource(solutionType, rapidAgentUri.getProtocol());
        params.put("TOKEN", brtToken);
        params.put("CODE", "/" + channelId);
        params.put("STREAM_ID", channelId);
        params.put("CHNID", channelId);
        params.put("vnoId", RapidAgentConstant.getVnoId());
        params.put("PLAYTYPE", playType);
        params.put("playSource", playType);
        if (rapidAgentUri instanceof RapidAgentVodUri) {
            String routeId = rapidAgentUri.getRouteId();
            if (TextUtils.isEmpty(routeId)) {
                if ("MTP_PORTAL".equals(playType)) {
                    throw new IOException("generate token error: invalid routeId");
                }
            } else {
                params.put("routeId", routeId);
            }
        } else if (rapidAgentUri instanceof RapidAgentCatchupUri) {
            if (rapidAgentUri.getFormat() >= 0) {
                params.put("FORMAT", String.valueOf(rapidAgentUri.getFormat()));
            } else {
                params.put("FORMAT", "1");
            }
        }

        return generateToken(solutionType, solutionBrand, params);
    }

    String generateTimeShiftToken(String channelId, String brtToken, int solutionType, int solutionBrand, @NonNull RapidAgentCatchupUri rapidAgentUri) {
        Map<String, String> params = new HashMap<>();
        String playType = PrtConfig.getPlaySource(solutionType, rapidAgentUri.getProtocol());
        params.put("TOKEN", brtToken);
        params.put("CODE", "/" + channelId);
        params.put("STREAM_ID", channelId);
        params.put("CHNID", channelId);
        params.put("PLAYTYPE", playType);
        params.put("playSource", playType);
        if (rapidAgentUri.getFormat() >= 0) {
            params.put("FORMAT", String.valueOf(rapidAgentUri.getFormat()));
        } else {
            params.put("FORMAT", "1");
        }

        return generateToken(solutionType, solutionBrand, params);
    }

    private String generateToken(int solutionType, int solutionBrand, @NonNull Map<String, String> params) {
        if (TextUtils.isEmpty(TOKEN_SECRET_KEY)) {
            throw new IllegalStateException("tokenSecretKey is null");
        }

        int _nRandNumLess30 = lrand48() % 15;
        byte[] randBytes1 = generateRandomData(_nRandNumLess30 + 6);
        String strRand1 = new String(randBytes1);

        int _nRandNumLess57 = lrand48() % 20;
        byte[] randBytes2 = generateRandomData(_nRandNumLess57 + 6);
        String strRand2 = new String(randBytes2);

        StringBuilder sb = new StringBuilder(getMacKeyName(solutionType) + "=" + RapidAgentConstant.getDeviceId() +
                "&APPVER=" + RapidAgentConstant.getSdkVersion() +
                "&ADID=" + RapidAgentConstant.getAndroidId() +
                "&RELEASEID=" + RapidAgentConstant.getBrand().toLowerCase() +
                "&UID=" + RapidAgentConstant.getAccountId() +
                "&RANDSTRING1=" + strRand1 +
                "&RANDSTRING2=" + strRand2 +
                "&DTYP=0" +
                "&ENCRYPTIONTYPE=" + solutionBrand);
        for (Map.Entry<String, String> entry : params.entrySet()) {
            sb.append("&").append(entry.getKey()).append("=").append(entry.getValue());
        }
        LogUtil.sensitive("generateToken by source: " + sb);
        return AES.encrypt(sb.toString(), translate(TOKEN_SECRET_KEY));
    }

    private static int lrand48() {
        int random = new Random().nextInt(Integer.MAX_VALUE);
        return Math.abs(random);
    }

    private static byte[] generateRandomData(int len) {
        int currentIndex = 0;

        byte[] pDataBytes;
        int target;
        for (pDataBytes = new byte[len]; currentIndex < len; pDataBytes[currentIndex++] = (byte) target) {
            int rand1;
            int rand2;
            if (lrand48() % 3 > 0) {
                if ((lrand48() & 1) > 0) {
                    rand1 = lrand48();
                    rand2 = rand1 % 10;
                    target = rand2 + 48;
                } else {
                    rand1 = lrand48();
                    rand2 = rand1 % 26;
                    target = rand2 + 65;
                }
            } else {
                rand1 = lrand48();
                rand2 = rand1 % 26;
                target = rand2 + 97;
            }
        }

        return pDataBytes;
    }

    private static String getMacKeyName(int solutionType) {
        switch (solutionType) {
            case SOLUTION_TYPE_NONE:
            case SOLUTION_TYPE_LIVE_VALO_HLS:
            case SOLUTION_TYPE_LIVE_PLAIN_RPD:
            case SOLUTION_TYPE_LIVE_DRM_HLS:
            case SOLUTION_TYPE_LIVE_DRM_DASH:
                return "APKSN";
            case SOLUTION_TYPE_VOD_MPT:
            case SOLUTION_TYPE_VOD_MPQ:
            case SOLUTION_TYPE_VOD_PLAIN_RPD:
            case SOLUTION_TYPE_VOD_DRM_HLS:
            case SOLUTION_TYPE_VOD_DRM_DASH:
            case SOLUTION_TYPE_VOD_TIME_SHIFT_PLAIN_RPD:
            case SOLUTION_TYPE_VOD_TIME_SHIFT_DRM_HLS:
            case SOLUTION_TYPE_VOD_TIME_SHIFT_DRM_DASH:
                return "ASN";
        }

        throw new IllegalStateException("invalid solutionType " + solutionType);
    }
}
