package com.bitkernel.stream.rapid.prt;

import static com.bitkernel.stream.rapid.RapidAgentStatistic.CATEGORY_STARTUP;
import static com.bitkernel.stream.rapid.RapidAgentStatistic.CATEGORY_UNSPECIFIED;
import static com.bitkernel.stream.rapid.utils.CommonUtil.parseInt;
import static com.bitkernel.stream.rapid.utils.CommonUtil.parseLong;

import android.text.TextUtils;

import com.bitkernel.stream.rapid.RapidAgentSDK;
import com.bitkernel.stream.rapid.RapidAgentStatistic;
import com.bitkernel.stream.rapid.config.CommonConfig;
import com.stream.prt.PrtEvent;

import java.util.LinkedHashMap;
import java.util.Map;

public final class PrtStatistic {
    private static RapidAgentStatistic sRapidAgentStatistic;

    private boolean engineReady = false;
    private boolean notifyStartupInfo = false;

    private int totalSuccessIndex = 0;
    private int totalFailIndex = 0;
    private int totalServerTs = 0;
    private int lastTotalServerTs = 0;
    private int totalPeerTs = 0;
    private int lastTotalPeerTs = 0;
    private int totalFailTs = 0;
    private int lastTotalFailTs = 0;
    private int totalSuccessPiece = 0;
    private int totalFailPiece = 0;
    private long totalServerBytes = 0;
    private long totalPeerBytes = 0;
    private long totalTurboBytes = 0;
    private int totalLostRtt = 0;

    private int maxFwdDelay = 0;
    private int maxEngDelay = 0;
    private int maxPopDelay = 0;
    private int maxDownloadMs = 0;
    private int maxDecodeFEC = 0;
    private int downCount = 0;
    private int downWithFEC = 0;
    private int downWithoutFEC = 0;

    private final Map<String, Object> statusMap = new LinkedHashMap<>();
    private long lastStatusReportTime;

    public static void setAgentStatistic(RapidAgentStatistic statistic) {
        sRapidAgentStatistic = statistic;
    }

    public void onEvent(int channelId, int event, String params, String desc, boolean callSuccess) {
        int category = callSuccess ? CATEGORY_UNSPECIFIED : CATEGORY_STARTUP;
        switch (event) {
            case RapidAgentSDK.EVENT_PROXY_READY:
                engineReady = true;
                break;
            case RapidAgentSDK.EVENT_CREATE_CHANNEL_FAIL:
            case RapidAgentSDK.EVENT_CREATE_CHANNEL_TIMEOUT:
            case RapidAgentSDK.EVENT_PROXY_NO_SIGNAL:
            case RapidAgentSDK.EVENT_CHANGE_STREAM_NO_SIGNAL:
            case RapidAgentSDK.EVENT_PROXY_VERIFY:
            case RapidAgentSDK.EVENT_GET_INDEX_FAIL:
                sRapidAgentStatistic.notifyEngineError(channelId, event, 0, desc, category);
                break;
            case PrtEvent.event_change_prt_server_id:
                sRapidAgentStatistic.notifyPrtServerIdChanged(params);
        }
    }

    public synchronized void onMetric(int channelId, Map<String, String> params) {
        sRapidAgentStatistic.notifyTrackerServerIdChanged(PrtConfig.getTrackerServerId(params.get("trAddr")));
        checkCrash(params);
        if (!engineReady) {
            return;
        }
        if (!notifyStartupInfo) {
            notifyStartupInfo = true;
            sRapidAgentStatistic.notifyEngineStartupTime(channelId, params);
        }

        StreamInfoMgr.StreamInfo streamInfo = StreamInfoMgr.getStreamInfo(channelId);
        if (streamInfo.isVod()) {
            // 本次播放的总索引数
            totalSuccessIndex = 1;
            totalFailIndex = 0;
            // 本次播放收到的ts总数
            totalServerTs = parseInt(params.get("recvBkFromPmT"), 0);
            totalPeerTs = parseInt(params.get("recvBkFromPeerT"), 0) + parseInt(params.get("recvBkFromStorageT"), 0);
            totalFailTs = 0;//点播不会下载失败，如果下载不到会一直尝试
            // 过去一分钟内的piece数
            totalSuccessPiece = parseInt(params.get("recv_ps_all_p"), 0);
            totalFailPiece = parseInt(params.get("recv_ps_try_p"), 0);
            // 一个上报周期内的字节数
            totalServerBytes += parseLong(params.get("recvBkBytesFromPmP"), 0);
            totalPeerBytes += parseLong(params.get("recvBkBytesFromPeerP"), 0);
        } else {
            // 本次播放的总索引数
            totalSuccessIndex = parseInt(params.get("m_Period_Ok"), 0);
            totalFailIndex = parseInt(params.get("m_Period_Failed"), 0);
            // 本次播放收到的ts总数
            totalServerTs = parseInt(params.get("recv_ts_ok_t_fs"), 0);
            totalPeerTs = parseInt(params.get("recv_ts_ok_t_fp"), 0);
            totalFailTs = parseInt(params.get("recv_ts_failed_t"), 0);
            // 过去一分钟内的piece数
            totalSuccessPiece = parseInt(params.get("recv_ps_all_p"), 0);
            totalFailPiece = parseInt(params.get("recv_ps_try_p"), 0);
            // 一个上报周期内的字节数
            totalServerBytes += parseLong(params.get("recv_ts_ok_bytes_prt_p"), 0);
            totalPeerBytes += parseLong(params.get("recv_ts_ok_bytes_peer_p"), 0);
            totalTurboBytes += parseLong(params.get("recv_ts_ok_bytes_turbo_p"), 0);
        }
        totalLostRtt = parseInt(params.get("longLostRtT"), 0);

        Map<String, Object> newParams = statusMap;
        newParams.put("tr_st", parseInt(params.get("trState"), 0));
        int tsSvrState = parseInt(params.get("tsSvrState"), 0);
        newParams.put("ts_svr_st", tsSvrState == 0 ? 1 : 0);
        newParams.put("conn_peers", parseInt(params.get("connectedPeers"), 0));
        newParams.put("peer_num", parseInt(params.get("PeerNums"), 0));
        newParams.put("cdn", parseInt(params.get("cdnMode"), 0));

        newParams.put("index_total_times", totalSuccessIndex);
        newParams.put("index_fail_times", totalFailIndex);
        newParams.put("recv_ps_all_p", totalSuccessPiece);
        newParams.put("recv_ps_try_p", totalFailPiece);
        newParams.put("recv_ts_ok_svr_p", totalServerTs - lastTotalServerTs);
        newParams.put("recv_ts_ok_peer_p", totalPeerTs - lastTotalPeerTs);
        newParams.put("recv_ts_fail", totalFailTs - lastTotalFailTs);
        newParams.put("recv_ts_ok_bytes_svr_p", totalServerBytes);
        newParams.put("recv_ts_ok_bytes_peer_p", totalPeerBytes);
        newParams.put("recv_ts_ok_bytes_turbo_p", totalTurboBytes);

        newParams.put("total_lost_rtt", totalLostRtt);

        if (streamInfo.isLive()) {
            newParams.put("m_play_ok_1min", parseInt(params.get("m_playOk_1min"), 0));
            newParams.put("m_play_rate", parseInt(params.get("m_play_Rate"), 0));
        }
        if (streamInfo.isMicroBlock()) {
            int engineDelay1 = parseInt(params.get("ts_avg_fwd_delay"), 0);
            int engineDelay2 = parseInt(params.get("ts_avg_eng_delay"), 0);
            if (engineDelay1 > 0 && engineDelay2 >= 0) {
                newParams.put("engine_delay", engineDelay1 + engineDelay2);
            }
            if (CommonConfig.getStatisticLevel() >= CommonConfig.STATISTIC_LEVEL_DEBUG) {
                maxFwdDelay = Math.max(maxFwdDelay, parseInt(params.get("ts_max_fwd_delay"), 0));
                maxEngDelay = Math.max(maxEngDelay, parseInt(params.get("ts_max_eng_delay"), 0));
                maxPopDelay = Math.max(maxPopDelay, parseInt(params.get("ts_max_player_delay"), 0));
                maxDownloadMs = Math.max(maxDownloadMs, parseInt(params.get("down_ts_ok_max_use"), 0));
                maxDecodeFEC = Math.max(maxDecodeFEC, parseInt(params.get("fecDecodeMaxMs"), 0));
                downCount += parseInt(params.get("recvOkP"), 0);
                downWithFEC += parseInt(params.get("recvNoRetryFecOk"), 0);
                downWithoutFEC += parseInt(params.get("recvNoRetryNoFecOk"), 0);
                if (maxFwdDelay > 0) {
                    newParams.put("max_fwd_delay", maxFwdDelay);
                    newParams.put("max_eng_delay", maxEngDelay);
                    newParams.put("max_pop_delay", maxPopDelay);
                    newParams.put("max_down_ms", maxDownloadMs);
                    newParams.put("max_dec_fec", maxDecodeFEC);
                    newParams.put("down_count", downCount);
                    newParams.put("down_with_fec", downWithFEC);
                    newParams.put("down_without_fec", downWithoutFEC);
                }
            }
        }
    }

    public synchronized void notifyStatusEvent(int channelId) {
        int interval = CommonConfig.statusEventReportInterval();
        long currentTime = System.nanoTime();
        if (currentTime - lastStatusReportTime < interval * 1_000_000_000L) {
            return;
        }
        lastStatusReportTime = currentTime;
        if (engineReady && statusMap.isEmpty()) {
            sRapidAgentStatistic.notifyEngineError(channelId, RapidAgentSDK.EVENT_HAS_NO_METRIC, 0, "has no onMetric callback", CATEGORY_UNSPECIFIED);
            return;
        }

        StreamInfoMgr.StreamInfo streamInfo = StreamInfoMgr.getStreamInfo(channelId);
        int playDuration = (int) ((currentTime - streamInfo.playerReadyTime) / 1000_000_000);
        statusMap.put("play_dur", playDuration);

        sRapidAgentStatistic.notifyEngineStatus(channelId, statusMap);

        statusMap.clear();
        totalSuccessIndex = 0;
        totalFailIndex = 0;
        lastTotalServerTs = totalServerTs;
        lastTotalPeerTs = totalPeerTs;
        lastTotalFailTs = totalFailTs;
        totalServerBytes = 0;
        totalPeerBytes = 0;
        totalTurboBytes = 0;
        totalSuccessPiece = 0;
        totalFailPiece = 0;
        totalLostRtt = 0;
        maxFwdDelay = 0;
        maxEngDelay = 0;
        maxPopDelay = 0;
        maxDownloadMs = 0;
        maxDecodeFEC = 0;
        downCount = 0;
        downWithFEC = 0;
        downWithoutFEC = 0;
    }

    private void checkCrash(Map<String, String> params) {
        String preCrashThreadName = params.containsKey("preCrashThreadName") ? params.get("preCrashThreadName") : null;
        if (!TextUtils.isEmpty(preCrashThreadName)) {
            String curMetricSec = params.get("curMetricSec");
            String preCrashSec = params.get("preCrashSec");
            String preCrashAddr = params.get("preCrashAddr");
            String preCrashRunSec = params.get("preCrashRunSec");
            String desc = curMetricSec + "," + preCrashSec + "," + preCrashAddr + "," + preCrashRunSec;
            sRapidAgentStatistic.notifySDKNativeCrash(desc);
        }
    }
}
