package com.bitkernel.stream.rapid;

import androidx.annotation.NonNull;

import java.util.Map;

/**
 * Representation of a media source.
 */
public class RapidAgentSource {
    /**
     * Clear stream
     */
    public static final int DRM_TYPE_NULL = 0;
    /**
     * Widevine stream
     */
    public static final int DRM_TYPE_WIDEVINE = 1;
    /**
     * GooseDrm stream
     */
    public static final int DRM_TYPE_GOOSE = 4;

    /**
     * media source path
     */
    protected String path;
    /**
     * Required by encrypted steam.
     * see {@link RapidAgentSource#DRM_TYPE_NULL}, {@link RapidAgentSource#DRM_TYPE_WIDEVINE}, {@link RapidAgentSource#DRM_TYPE_GOOSE}
     */
    protected int drmType = DRM_TYPE_NULL;
    /**
     * Required by encrypted steam. DRM license server URI.
     */
    protected String drmLicenseUrl;
    /**
     * Required by encrypted steam. The optional request headers attached to DRM license requests.
     */
    protected Map<String, String> drmHeaders;
    /**
     * The method of DRM license requests. MUST be POST or GET
     */
    protected String drmMethod;
    /**
     * The offline license key set id
     */
    protected byte[] drmOfflineLicenseKeySetId;
    /**
     * The offline license drm init info
     */
    protected String drmOfflineLicenseInitInfo;
    /**
     * Whether it's an adaptive bitrate stream.
     */
    protected boolean isAdaptiveBitrate;

    /**
     * Constructs a new instance
     *
     * @param path the media source path
     */
    public RapidAgentSource(@NonNull String path) {
        this.path = path;
    }

    /**
     * Gets the media source path
     *
     * @return the path
     */
    public String getPath() {
        return path;
    }

    /**
     * Gets the drm type, return {@link RapidAgentUri#DRM_TYPE_NULL}, {@link RapidAgentUri#DRM_TYPE_WIDEVINE}, {@link RapidAgentUri#DRM_TYPE_GOOSE}
     *
     * @return the drm type
     */
    public int getDrmType() {
        return drmType;
    }

    /**
     * Sets the drm type, see {@link RapidAgentUri#DRM_TYPE_NULL}, {@link RapidAgentUri#DRM_TYPE_WIDEVINE}, {@link RapidAgentUri#DRM_TYPE_GOOSE}
     *
     * @param drmType the drm type
     */
    public void setDrmType(int drmType) {
        this.drmType = drmType;
    }

    /**
     * Gets the default DRM license server URI.
     *
     * @return url
     */
    public String getDrmLicenseUrl() {
        return drmLicenseUrl;
    }

    /**
     * Sets the default DRM license server URI.
     *
     * @param drmLicenseUrl url
     */
    public void setDrmLicenseUrl(String drmLicenseUrl) {
        this.drmLicenseUrl = drmLicenseUrl;
    }

    /**
     * Gets the optional request headers attached to DRM license requests.
     *
     * @return the request headers
     */
    public Map<String, String> getDrmHeaders() {
        return drmHeaders;
    }

    /**
     * Sets the optional request headers attached to DRM license requests.
     *
     * @param drmHeaders the request headers
     */
    public void setDrmHeaders(Map<String, String> drmHeaders) {
        this.drmHeaders = drmHeaders;
    }

    /**
     * Gets the method of DRM license requests.
     *
     * @return http MUST be POST or GET
     */
    public String getDrmMethod() {
        return drmMethod;
    }

    /**
     * Sets the method of DRM license requests. MUST be POST or GET
     *
     * @param drmMethod http method
     */
    public void setDrmMethod(String drmMethod) {
        this.drmMethod = drmMethod;
    }

    /**
     * set offline license key set id
     *
     * @param set offline license set
     */
    public void setDrmOfflineLicenseKeySetId(byte[] set) {
        if (set != null) {
            this.drmOfflineLicenseKeySetId = set.clone();
        } else {
            this.drmOfflineLicenseKeySetId = null;
        }
    }

    /**
     * get offline license key set id
     *
     * @return offline license key set id
     */
    public byte[] getDrmOfflineLicenseKeySetId() {
        return drmOfflineLicenseKeySetId;
    }

    /**
     * set offline license drm init info
     *
     * @param drmOfflineLicenseInitInfo offline license drm init info
     */
    public void setDrmOfflineLicenseInitInfo(String drmOfflineLicenseInitInfo) {
        this.drmOfflineLicenseInitInfo = drmOfflineLicenseInitInfo;
    }

    /**
     * get offline license drm init info
     *
     * @return offline license drm init info
     */
    public String getDrmOfflineLicenseInitInfo() {
        return drmOfflineLicenseInitInfo;
    }

    /**
     * Sets whether this is an adaptive bitrate stream.
     *
     * @param adaptiveBitrate whether this is an adaptive bitrate stream.
     */
    public void setAdaptiveBitrate(boolean adaptiveBitrate) {
        isAdaptiveBitrate = adaptiveBitrate;
    }

    /**
     * Returns whether this is an adaptive bitrate stream.
     *
     * @return whether this is an adaptive bitrate stream.
     */
    public boolean isAdaptiveBitrate() {
        return isAdaptiveBitrate;
    }

    @NonNull
    @Override
    public String toString() {
        return "RapidAgentSource{" +
                "path='" + path + '\'' +
                ", drmType='" + drmType + '\'' +
                ", abr='" + isAdaptiveBitrate + '\'' +
                '}';
    }
}
