package com.bitkernel.stream.rapid.player;

import android.content.Context;
import android.os.Build;
import android.os.Bundle;
import android.os.Handler;
import android.os.HandlerThread;
import android.view.Surface;
import android.view.SurfaceHolder;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.bitkernel.stream.rapid.RapidAgentCall;
import com.bitkernel.stream.rapid.RapidAgentCatchupUri;
import com.bitkernel.stream.rapid.RapidAgentLiveUri;
import com.bitkernel.stream.rapid.RapidAgentSDK;
import com.bitkernel.stream.rapid.RapidAgentUri;
import com.bitkernel.stream.rapid.config.CommonConfig;
import com.bitkernel.stream.rapid.utils.CommonUtil;
import com.bitkernel.stream.rapid.utils.LogUtil;
import com.bitkernel.stream.rapid.utils.SPUtil;

import org.json.JSONException;
import org.json.JSONObject;

import java.io.IOException;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;

import tv.danmaku.ijk.media.player.IMediaPlayer;
import tv.danmaku.ijk.media.player.IjkMediaPlayer;

public class RapidAgentStandardPlayer extends RapidMediaPlayer implements RapidAgentPlayer, RapidAgentSDK.OnPlayListener {
    private static final String TAG = "RapidAgentStandardPlayer";

    private static final Map<String, String> DRM_INFO_MAP = new HashMap<>();
    private static final HandlerThread WORK_THREAD;
    private static final Handler WORK_HANDLER;

    static {
        WORK_THREAD = new HandlerThread("RapidPlayerWork");
        WORK_THREAD.start();
        WORK_HANDLER = new Handler(WORK_THREAD.getLooper());
    }

    private final RapidPlayerStatistic rapidPlayerStatistic;
    private final RapidPlayerBuilder rapidPlayerBuilder;
    private boolean enableReleaseAsync;
    private boolean enableCacheDrmInfo;
    private RapidAgentUri rapidAgentUri;
    private RapidAgentCall rapidAgentCall;
    private long timeShiftDurationMs;
    private long timeShiftStartMs;
    private final AtomicBoolean timeShiftSeeking = new AtomicBoolean();
    private boolean isStreamReady = false;
    private long prepareStartTime;
    private long bufferingStartTime;

    private float mFpsOutput = 0.0f;
    private float mFpsDecode = 0.0f;
    private int mPlayDelayMs = 0;

    private OnStreamMetricListener streamMetricListener;
    private OnStreamStateListener streamStateListener;
    private RapidAgentCall.OnStateChangedListener callStateChangedListener;
    private final Map<String, Object> playMetricMap = new HashMap<>();
    private long lastMetricCallbackTime = 0;
    private int fwdDelayMs;
    private int engDelayMs;
    private int downCount = 0;
    private int downWithFEC = 0;
    private int downWithoutFEC = 0;

    private long downloadBlockCount = 0;
    private long downloadTime = 0;

    private int currentIdealPackets = 0;
    private long currentBandwidth = 0;

    public RapidAgentStandardPlayer(@Nullable Context context, int playerType, Map<String, String> params,
                                    RapidPlayerStatistic playerStatistic, RapidPlayerBuilder playerBuilder) {
        super(context, playerType, params, playerStatistic, playerBuilder);
        this.rapidPlayerStatistic = playerStatistic;
        this.rapidPlayerBuilder = playerBuilder;
    }

    public void setRapidAgentUri(@NonNull RapidAgentUri rapidAgentUri) {
        this.rapidAgentUri = rapidAgentUri;
        if (rapidAgentUri instanceof RapidAgentLiveUri) {
            setStartupTimeoutMs(CommonConfig.playerStartupTimeout() * 1000L);
        }
        if (rapidAgentUri.getDrmType() == RapidAgentUri.DRM_TYPE_WIDEVINE) {
            boolean supportMultiSession = Build.VERSION.SDK_INT > Build.VERSION_CODES.M;
            String drmLicenseUri = rapidAgentUri.getDrmLicenseUrl();
            Map<String, String> headers = new HashMap<>();
            if (rapidAgentUri.getDrmHeaders() != null) {
                headers.putAll(rapidAgentUri.getDrmHeaders());
            }
            setDrmInfo(IMediaPlayer.DRM_TYPE_WIDEVINE, supportMultiSession, drmLicenseUri, headers, rapidAgentUri.getDrmMethod(),
                    rapidAgentUri.getDrmOfflineLicenseKeySetId(), rapidAgentUri.getDrmOfflineLicenseInitInfo());
        }
        if (rapidAgentUri instanceof RapidAgentCatchupUri) {
            RapidAgentCatchupUri catchupUri = (RapidAgentCatchupUri) rapidAgentUri;
            if (catchupUri.isTimeShift()) {
                long startMs = Long.parseLong(catchupUri.getEpgStart());
                timeShiftDurationMs = System.currentTimeMillis() - startMs;
                timeShiftStartMs = timeShiftDurationMs - catchupUri.getTimeShiftDelay() * 1000;
            }
        }
    }

    public void setEnableReleaseAsync(boolean enableReleaseAsync) {
        this.enableReleaseAsync = enableReleaseAsync;
    }

    public void setEnableCacheDrmInfo(boolean enableCacheDrmInfo) {
        this.enableCacheDrmInfo = enableCacheDrmInfo;
    }

    private void runOnWorkThread(Runnable runnable) {
        if (enableReleaseAsync) {
            WORK_HANDLER.post(runnable);
        } else {
            runnable.run();
        }
    }

    @Override
    public long getDuration() {
        if (timeShiftDurationMs > 0) {
            RapidAgentCatchupUri catchupUri = (RapidAgentCatchupUri) rapidAgentUri;
            long startMs = Long.parseLong(catchupUri.getEpgStart());
            return (timeShiftDurationMs = System.currentTimeMillis() - startMs);
        }
        return super.getDuration();
    }

    @Override
    public long getCurrentPosition() {
        long pos = super.getCurrentPosition();
        if (timeShiftDurationMs > 0) {
            pos = pos + timeShiftStartMs;
        }
        return pos;
    }

    @Override
    public void setDisplay(SurfaceHolder sh) {
        runOnWorkThread(() -> RapidAgentStandardPlayer.super.setDisplay(sh));
    }

    @Override
    public void setSurface(Surface surface) {
        runOnWorkThread(() -> RapidAgentStandardPlayer.super.setSurface(surface));
    }

    @Override
    public void prepareAsync() throws IllegalStateException {
        prepareStartTime = System.nanoTime() / 1000_000;
        if (enableCacheDrmInfo) {
            if (DRM_INFO_MAP.containsKey(rapidAgentUri.getStreamId())) {
                updateDrmInitInfo(DRM_INFO_MAP.get(rapidAgentUri.getStreamId()));
            }
        }
        startStream();
    }

    @Override
    public void seekTo(long msec) throws IllegalStateException {
        if (rapidAgentUri instanceof RapidAgentCatchupUri) {
            RapidAgentCatchupUri catchupUri = (RapidAgentCatchupUri) rapidAgentUri;
            if (catchupUri.isTimeShift() && timeShiftSeeking.compareAndSet(false, true)) {
                long delay = (timeShiftDurationMs - msec) / 1000;
                LogUtil.i(TAG, "timeShiftSeekTo " + delay);
                catchupUri.setTimeShiftDelay(delay);
                rapidAgentCall.requestTimeShiftData(delay);
                timeShiftStartMs = timeShiftDurationMs - catchupUri.getTimeShiftDelay() * 1000;
                rebuildInternalPlayer(getPlayerType(), RAPID_ERROR_CATEGORY, RAPID_ERROR_TIME_SHIFT_SEEK_TIMEOUT);
                return;
            }
        }
        super.seekTo(msec);
    }

    @Override
    public void stop() throws IllegalStateException {
        // release player first to avoid prt engine crash.
        // when stop channel, prt will release buffer first, but http proxy still working.
        // then http proxy trigger heap-use-after-free error.
        RapidAgentStandardPlayer.super.stop();
        stopStream();
    }

    @Override
    public void release() {
        if (currentIdealPackets > 0) {
            SPUtil.setMBIdealPacket(rapidAgentUri.getMicroBlockLevel(), currentIdealPackets);
        }
        if (currentBandwidth > 0) {
            SPUtil.setMBBandwidth(currentBandwidth);
        }
        runOnWorkThread(RapidAgentStandardPlayer.super::release);
    }

    @Override
    protected boolean onInfo(IMediaPlayer mp, int what, int extra, String info) {
        if (what == IMediaPlayer.MEDIA_INFO_FIND_STREAM_INFO) {
            long startupDuration = System.nanoTime() / 1000_000 - prepareStartTime;
            LogUtil.i("RapidAgentCall", "loaded       " + rapidAgentCall + " " + startupDuration + "ms");
        } else if (what == IMediaPlayer.MEDIA_INFO_VIDEO_RENDERING_START || (what == IMediaPlayer.MEDIA_INFO_AUDIO_RENDERING_START && extra == 1)) {
            long startupDuration = System.nanoTime() / 1000_000 - prepareStartTime;
            LogUtil.i("RapidAgentCall", "playing      " + rapidAgentCall + " " + startupDuration + "ms");
            if (timeShiftSeeking.compareAndSet(true, false)) {
                LogUtil.i(TAG, "timeShiftSeekCompleted");
                onSeekComplete(mp);
            }
        } else if (what == RapidAgentPlayer.MEDIA_INFO_JBUF_PLAY_RATE_CHANGED) {
            LogUtil.i(TAG, "jbuf play rate changed:" + info);
            try {
                JSONObject jsonObject = new JSONObject(info);
                currentIdealPackets = jsonObject.getInt("ideal");
            } catch (JSONException e) {
                throw new RuntimeException(e);
            }
        } else if (what == MEDIA_INFO_BUFFERING_START) {
            bufferingStartTime = System.nanoTime() / 1000_000;
        } else if (what == MEDIA_INFO_BUFFERING_END) {
            rapidAgentCall.notifyBuffering((int) (System.nanoTime() / 1000_000 - bufferingStartTime));
        }
        return super.onInfo(mp, what, extra, info);
    }

    @Override
    protected void onDrmInitInfo(IMediaPlayer mp, String drmInitInfo) {
        super.onDrmInitInfo(mp, drmInitInfo);
        if (enableCacheDrmInfo) {
            DRM_INFO_MAP.put(rapidAgentUri.getStreamId(), drmInitInfo);
        }
    }

    @Override
    protected boolean onNativeInvoke(int what, Bundle args) {
        if (rapidAgentUri.isMicroBlock() && what == IjkMediaPlayer.OnNativeInvokeListener.EVENT_DID_HTTP_READ_END) {
            IjkMediaPlayer ijkMediaPlayer = (IjkMediaPlayer) getInternalPlayer();
            int blockCount = args.getInt(IjkMediaPlayer.OnNativeInvokeListener.ARG_HTTP_CODE);
            long frameCount = ijkMediaPlayer.getVideoReadPackets();
            downloadBlockCount += blockCount;
            checkMicroBlockLevelMismatch(downloadBlockCount, frameCount);
            if (CommonConfig.getStatisticLevel() >= CommonConfig.STATISTIC_LEVEL_DEBUG) {
                long readVideoCount = downloadBlockCount;
                long dropVideoCount = ijkMediaPlayer.getMlsVideoDropCount();
                long maxVideoCount = ijkMediaPlayer.getJitterMaxPackets();
                long minVideoCount = ijkMediaPlayer.getJitterMinPackets();
                long maxBlockCost = ijkMediaPlayer.getJitterMaxCostMs();
                long now = System.nanoTime() / 1000_000;
                int frameCost = args.getInt(IjkMediaPlayer.OnNativeInvokeListener.ARG_ERROR);
                int segmentCost = downloadTime > 0 ? (int) (now - downloadTime) : 0;
                downloadTime = now;
                Map<String, Object> map = new HashMap<>();
                map.put("frame_cost", frameCost);
                map.put("segment_cost", segmentCost);
                map.put("read_block_count", readVideoCount);
                map.put("drop_block_count", dropVideoCount);
                map.put("max_block_count", maxVideoCount);
                map.put("min_block_count", minVideoCount);
                map.put("max_block_cost", maxBlockCost);
                playerStatistic.downloadPerformance(map);
            }
        }
        return super.onNativeInvoke(what, args);
    }

    @Override
    public void setOnStreamMetricListener(OnStreamMetricListener listener) {
        streamMetricListener = listener;
    }

    @Override
    public void setOnStreamStateListener(OnStreamStateListener listener) {
        streamStateListener = listener;
    }

    @Override
    public void setOnCallStateChangedListener(RapidAgentCall.OnStateChangedListener listener) {
        callStateChangedListener = listener;
    }

    @Override
    public void updateTokens(String newUserToken, String newStreamToken) throws IOException {
        if (rapidAgentCall == null) {
            return;
        }
        rapidAgentCall.updateTokens(newUserToken, newStreamToken);
    }

    @Override
    public void sendFrontLog(String msg) {
        if (rapidAgentCall == null) {
            return;
        }
        rapidAgentCall.sendFrontLog(msg);
    }

    @NonNull
    @Override
    public Map<String, Object> getPlayMetric() {
        updatePlayMetric();
        return playMetricMap;
    }

    @Override
    public void onSegmentDownloaded(String params) {
        LogUtil.e(TAG, "should not call onSegmentDownloaded");
    }

    private void internalSegmentDownloaded(String params) {
        final SegmentDownloadedListener listener = rapidPlayerBuilder.getSegmentDownloadedListener();
        if (listener != null) {
            listener.onSegmentDownloaded(params);
            currentBandwidth = listener.getCurrentBandwidth();
        }
    }

    @Override
    public void onPrtStarted(RapidAgentCall call) {
        call.setStateChangedListener(callStateChangedListener);
    }

    @Override
    public void onPrtEvent(int event, String params, String desc) {
        try {
            handleStreamEvent(event, params, desc);
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    @Override
    public void onPrtMetric(Map<String, String> params) {
        fwdDelayMs = CommonUtil.parseInt(params.get("ts_avg_fwd_delay"), 0);
        engDelayMs = CommonUtil.parseInt(params.get("ts_avg_eng_delay"), 0);
        downCount += CommonUtil.parseInt(params.get("recvOkP"), 0);
        downWithFEC += CommonUtil.parseInt(params.get("recvNoRetryFecOk"), 0);
        downWithoutFEC += CommonUtil.parseInt(params.get("recvNoRetryNoFecOk"), 0);
        long now = System.nanoTime() / 1000_000;
        if (lastMetricCallbackTime > 0) {
            long totalSize = 0;
            totalSize += CommonUtil.parseLong(params.get("recv_ts_ok_bytes_prt_p"), 0);
            totalSize += CommonUtil.parseLong(params.get("recv_ts_ok_bytes_peer_p"), 0);
            totalSize += CommonUtil.parseLong(params.get("recv_ts_ok_bytes_turbo_p"), 0);
            float bitrate = (float) totalSize / ((now - lastMetricCallbackTime) / 1000f) / 131072;
            playMetricMap.put("bitrate", bitrate);
        }
        lastMetricCallbackTime = now;
        playMetricMap.put("engineFwdDelayMs", fwdDelayMs);
        playMetricMap.put("engineEngDelayMs", engDelayMs);
        playMetricMap.put("downCount", downCount);
        playMetricMap.put("downWithFEC", downWithFEC);
        playMetricMap.put("downWithoutFEC", downWithoutFEC);
        if (fwdDelayMs > 0 && engDelayMs >= 0 && mPlayDelayMs > 0) {
            playMetricMap.put("totalDelayMs", fwdDelayMs + engDelayMs + mPlayDelayMs);
        }
        if (streamMetricListener != null) {
            streamMetricListener.onStreamMetric(params);
        }
    }

    @Override
    public void onPrtState(Map<String, String> state) {
        if (streamStateListener != null) {
            streamStateListener.onStreamState(state);
        }
    }

    @Override
    public void onPrtProgress() {
        updateStatus();
    }

    @Override
    public int getPrtPlayerCacheTime(int channelId) {
        if (isRebuilding()) {
            return 0;
        }
        IMediaPlayer internalPlayer = getInternalPlayer();
        if (internalPlayer instanceof IjkMediaPlayer) {
            IjkMediaPlayer ijkMediaPlayer = (IjkMediaPlayer) internalPlayer;
            return (int) (ijkMediaPlayer.getVideoCachedDuration() / 1000);
        }
        return 0;
    }

    private void startStream() {
        try {
            rapidAgentCall = RapidAgentSDK.startStream(rapidAgentUri, this);
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    private void handleStreamEvent(int event, String params, String desc) throws IOException {
        switch (event) {
            case RapidAgentSDK.EVENT_CREATE_CHANNEL_FAIL:
            case RapidAgentSDK.EVENT_CREATE_CHANNEL_TIMEOUT:
            case RapidAgentSDK.EVENT_PROXY_NO_SIGNAL:
            case RapidAgentSDK.EVENT_CHANGE_STREAM_NO_SIGNAL:
            case RapidAgentSDK.EVENT_PROXY_VERIFY:
            case RapidAgentSDK.EVENT_GET_INDEX_FAIL:
                stop();
                onError(getInternalPlayer(), STREAM_ERROR_MSG, event);
                break;
            case RapidAgentSDK.EVENT_MB_NOT_AVAILABLE:
                LogUtil.i(TAG, "restart the stream");
                if (appHandler != null) {
                    appHandler.post(() -> {
                        if (isStreamReady) {
                            outsideInfoListener.onInfo(this, MEDIA_INFO_BUFFERING_START, 0, String.valueOf(FFP_BUFFERING_REASON_START_REBUILD));
                        }
                        rapidAgentUri.setMicroBlock(false, 0);
                        RapidAgentStandardPlayer.super.stop();
                        stopStream();
                        startStream();
                    });
                }
                break;
            case RapidAgentSDK.EVENT_CHANGE_PRT_SERVER_ID:
            case RapidAgentSDK.EVENT_VIDEO_BUFFER_REACH_MAX_SIZE:
            case RapidAgentSDK.EVENT_CONNECT_TOKEN_EXPIRED:
                onInfo(getInternalPlayer(), STREAM_INFO_MSG, event, params + "," + desc);
                break;
            case RapidAgentSDK.EVENT_PROXY_READY:
                onStreamReady(params);
                break;
            case RapidAgentSDK.EVENT_CHANGE_STREAM_RESTART_PLAYER:
                if (appHandler != null) {
                    rebuildInternalPlayer(getPlayerType(), RAPID_ERROR_CATEGORY, RapidAgentSDK.EVENT_CHANGE_STREAM_RESTART_PLAYER);
                }
                break;
            case RapidAgentSDK.EVENT_SEGMENT_DOWNLOADED:
                internalSegmentDownloaded(params);
                break;
        }
    }

    private void onStreamReady(String url) throws IOException {
        if (isStreamReady) {
            outsideInfoListener.onInfo(this, MEDIA_INFO_BUFFERING_END, 0, String.valueOf(FFP_BUFFERING_REASON_END_REBUILD));
            rebuildInternalPlayer(getPlayerType(), url, RAPID_ERROR_CATEGORY, RapidAgentSDK.EVENT_MB_NOT_AVAILABLE);
        } else {
            setDataSource(url);
            runOnWorkThread(RapidAgentStandardPlayer.super::prepareAsync);
        }
        isStreamReady = true;
        downloadTime = lastMetricCallbackTime = System.nanoTime() / 1000_000;
    }

    private void stopStream() {
        LogUtil.e(TAG, "stop stream");
        if (rapidAgentCall != null) {
            RapidAgentSDK.stopStream(rapidAgentCall);
            rapidAgentCall = null;
        }
    }

    private void updateStatus() {
        IMediaPlayer internalPlayer = getInternalPlayer();
        if ((internalPlayer instanceof IjkMediaPlayer) && internalPlayer.isPlaying()) {
            IjkMediaPlayer ijkMediaPlayer = (IjkMediaPlayer) internalPlayer;
            float fpsOutput = ijkMediaPlayer.getVideoOutputFramesPerSecond();
            float fpsDecode = ijkMediaPlayer.getVideoDecodeFramesPerSecond();
            Map<String, String> map = new HashMap<>();
            if (fpsOutput > 0 && fpsOutput < 50) {
                mFpsOutput = mFpsOutput > 0 ? (mFpsOutput + fpsOutput) / 2 : fpsOutput;
                map.put("outputFPS", String.format(Locale.US, "%.2f", mFpsOutput));
            }
            if (fpsDecode > 0 && fpsDecode < 50) {
                mFpsDecode = mFpsDecode > 0 ? (mFpsDecode + fpsDecode) / 2 : fpsDecode;
                map.put("decFPS", String.format(Locale.US, "%.2f", mFpsDecode));
            }
            if (rapidAgentUri.isMicroBlock()) {
                int playDelayMs = (int) (ijkMediaPlayer.getBufDelay() * 1000);
                if (mPlayDelayMs > 60_000) {
                    mPlayDelayMs = playDelayMs;
                } else {
                    mPlayDelayMs = mPlayDelayMs > 0 ? (mPlayDelayMs + playDelayMs) / 2 : playDelayMs;
                }
                map.put("playDelayMs", String.valueOf(mPlayDelayMs));
            }
            if (rapidAgentUri.isMicroBlock() && CommonConfig.getStatisticLevel() >= CommonConfig.STATISTIC_LEVEL_DEBUG) {
                int maxCostMs = (int) ijkMediaPlayer.getJitterMaxCostMs();
                long readVideoPackets = downloadBlockCount;
                long dropVideoPackets = ijkMediaPlayer.getMlsVideoDropCount();
                int minVideoBuf = (int) ijkMediaPlayer.getJitterMinPackets();
                int maxVideoBuf = (int) ijkMediaPlayer.getJitterMaxPackets();
                int idealVideoBuf = (int) ijkMediaPlayer.getJitterIdealPackets();
                if (maxCostMs > 0) {
                    map.put("maxCostMs", String.valueOf(maxCostMs));
                }
                if (readVideoPackets + dropVideoPackets > 0) {
                    map.put("dropRate", String.format(Locale.US, "%.3f", dropVideoPackets * 1.0f / (readVideoPackets + dropVideoPackets)));
                }
                map.put("minVideoBuf", String.valueOf(minVideoBuf));
                map.put("maxVideoBuf", String.valueOf(maxVideoBuf));
                map.put("idealVideoBuf", String.valueOf(idealVideoBuf));
            }
            rapidPlayerStatistic.onStatusChanged(map);
        }
    }

    private void updatePlayMetric() {
        IMediaPlayer internalPlayer = getInternalPlayer();
        if (internalPlayer instanceof IjkMediaPlayer) {
            IjkMediaPlayer ijkMediaPlayer = (IjkMediaPlayer) internalPlayer;
            float fpsOutput = ijkMediaPlayer.getVideoOutputFramesPerSecond();
            float playSpeed = ijkMediaPlayer.getSpeed(1.0f);
            long curVideoBuf = ijkMediaPlayer.getVideoTotalPackets();
            long curAudioBuf = ijkMediaPlayer.getAudioTotalPackets();
            long minVideoBuf = ijkMediaPlayer.getVideoMinPackets();
            long maxVideoBuf = ijkMediaPlayer.getVideoMaxPackets();
            long minAudioBuf = ijkMediaPlayer.getAudioMinPackets();
            long maxAudioBuf = ijkMediaPlayer.getAudioMaxPackets();
            long dropVideoCount = ijkMediaPlayer.getMlsVideoDropCount();
            long dropAudioCount = ijkMediaPlayer.getMlsAudioDropCount();
            long idealBuf = ijkMediaPlayer.getJitterIdealPackets();
            long maxCost = ijkMediaPlayer.getJitterMaxCostMs();
            playMetricMap.put("outputFPS", fpsOutput);
            playMetricMap.put("speed", playSpeed);
            playMetricMap.put("curVideoBuf", curVideoBuf);
            playMetricMap.put("curAudioBuf", curAudioBuf);
            playMetricMap.put("minVideoBuf", minVideoBuf);
            playMetricMap.put("maxVideoBuf", maxVideoBuf);
            playMetricMap.put("minAudioBuf", minAudioBuf);
            playMetricMap.put("maxAudioBuf", maxAudioBuf);
            playMetricMap.put("dropVideoBlock", dropVideoCount);
            playMetricMap.put("dropAudioBlock", dropAudioCount);
            playMetricMap.put("idealBuf", idealBuf);
            playMetricMap.put("maxCost", maxCost);
            if (mPlayDelayMs > 0) {
                playMetricMap.put("playerDelayMs", mPlayDelayMs);
            }
            if (fwdDelayMs > 0 && engDelayMs >= 0 && mPlayDelayMs > 0) {
                playMetricMap.put("totalDelayMs", fwdDelayMs + engDelayMs + mPlayDelayMs);
            }
        }
    }

    private void checkMicroBlockLevelMismatch(long blockCount, long frameCount) {
        float avg = frameCount * 1.0f / blockCount;
        int level;
        if (avg < 10) {
            level = RapidAgentUri.MICRO_BLOCK_ULTRA_LOW;
        } else if (avg < 40) {
            level = RapidAgentUri.MICRO_BLOCK_LOW;
        } else {
            level = RapidAgentUri.MICRO_BLOCK_NORMAL;
        }
        if (level != rapidAgentUri.getMicroBlockLevel()) {
            LogUtil.w(TAG, "micro block level may be mismatch, block:" + blockCount + ", frame:" + frameCount);
        }
    }
}
