package com.bitkernel.stream.rapid;

import androidx.annotation.NonNull;

import com.bitkernel.stream.rapid.prt.PrtEngine;
import com.bitkernel.stream.rapid.prt.PrtStatistic;
import com.bitkernel.stream.rapid.utils.LogUtil;

import java.io.IOException;
import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * Representation of a stream connection.
 */
public class RapidAgentCall implements RapidAgentSDK.OnPlayListener {
    /**
     * Stream status enum
     */
    public enum Status {
        IDLE, REQUESTING, REQUESTED, CALLING, CONNECTED, READY, ERROR, RELEASED
    }

    /**
     * Call status changed listener
     */
    public interface OnStateChangedListener {
        void onStateChanged(Status currentStatus, int channelId, RapidAgentUri rapidAgentUri);
    }

    private static final String TAG = "RapidAgentCall";
    private static final AtomicInteger GLOBAL_CALL_INDEX = new AtomicInteger(0);
    private static final int IGNORE_CHANNEL_ID = Integer.MIN_VALUE + 1;

    private OnStateChangedListener stateChangedListener;

    private PrtEngine requestEngine;
    private PrtEngine callEngine;

    private final RapidAgentUri rapidAgentUri;
    private final PrtStatistic statistic;
    private final int index;
    private final long startTime;
    private final String streamType;
    private int channelId = IGNORE_CHANNEL_ID;
    private Status status;
    private boolean callSuccess = false;

    RapidAgentCall(@NonNull RapidAgentUri rapidAgentUri) {
        this.rapidAgentUri = rapidAgentUri;
        statistic = new PrtStatistic();
        index = GLOBAL_CALL_INDEX.addAndGet(1);
        startTime = System.nanoTime();
        streamType = getStreamType();
    }

    /**
     * Gets the start time of the connection in milliseconds since the Unix epoch
     *
     * @return timestamp
     */
    public long getStartTime() {
        return startTime;
    }

    void updateStatus(Status status) {
        this.status = status;
        if (status == Status.REQUESTED) {
            requestEngine.shutdown();
            requestEngine = null;
        } else if (status == Status.READY) {
            callSuccess = true;
        }
        printState("updateStatus");
        if (stateChangedListener != null) {
            stateChangedListener.onStateChanged(status, channelId, rapidAgentUri);
        }
    }

    void setRequestEngine(int channelId, PrtEngine prtEngine) {
        this.requestEngine = prtEngine;
    }

    void setCallEngine(int channelId, PrtEngine prtEngine) {
        this.channelId = channelId;
        this.callEngine = prtEngine;
    }

    /**
     * Sets call status changed listener
     *
     * @param stateChangedListener call status changed listener
     */
    public void setStateChangedListener(OnStateChangedListener stateChangedListener) {
        this.stateChangedListener = stateChangedListener;
    }

    /**
     * Update the user token and stream token
     *
     * @param newUserToken   new user token
     * @param newStreamToken new stream token
     * @throws IOException throw exception if fail to update the token
     */
    public void updateTokens(String newUserToken, String newStreamToken) throws IOException {
        printState("updateTokens");
        PrtEngine engine = null;
        if (requestEngine != null) {
            engine = requestEngine;
        }
        if (callEngine != null) {
            engine = callEngine;
        }
        if (engine == null) {
            LogUtil.e(TAG, "updateTokens:prt engine is invalid");
            return;
        }
        engine.updateTokens(newUserToken, newStreamToken);
    }

    /**
     * Requests new time shift data, used to seek
     *
     * @param delayTime delay in seconds
     */
    public void requestTimeShiftData(long delayTime) {
        RapidAgentCatchupUri catchupUri = null;
        if (rapidAgentUri instanceof RapidAgentCatchupUri) {
            catchupUri = (RapidAgentCatchupUri) rapidAgentUri;
        }
        if (catchupUri == null || !catchupUri.isTimeShift()) {
            return;
        }
        if (callEngine == null) {
            LogUtil.e(TAG, "requestTimeShiftData:prt engine is invalid");
            return;
        }
        callEngine.requestTimeShiftData(delayTime);
    }

    /**
     * Send buffering msg to engine
     *
     * @param ms duration in milliseconds
     */
    public void notifyBuffering(int ms) {
        if (callEngine == null) {
            LogUtil.w(TAG, "notifyBuffering:prt engine is invalid");
            return;
        }
        callEngine.notifyBuffering(ms);
    }

    /**
     * send app log to prt log server
     *
     * @param log message to send
     * @hidden
     */
    public void sendFrontLog(String log) {
        PrtEngine engine = null;
        if (requestEngine != null) {
            engine = requestEngine;
        }
        if (callEngine != null) {
            engine = callEngine;
        }
        if (engine != null) {
            engine.sendFrontLog(log);
        }
    }

    /**
     * Shutdown the call
     */
    public void shutdown() {
        printState("shutdown    ");
        if (requestEngine != null) {
            requestEngine.shutdown();
            requestEngine = null;
        }
        if (callEngine != null) {
            callEngine.shutdown();
            callEngine = null;
        }
        updateStatus(RapidAgentCall.Status.RELEASED);
    }

    @Override
    public void onPrtEvent(int event, String params, String desc) {
        switch (event) {
            case RapidAgentSDK.EVENT_PROXY_READY:
                updateStatus(Status.READY);
                break;
            case RapidAgentSDK.EVENT_CREATE_CHANNEL_FAIL:
            case RapidAgentSDK.EVENT_CREATE_CHANNEL_TIMEOUT:
            case RapidAgentSDK.EVENT_PROXY_NO_SIGNAL:
            case RapidAgentSDK.EVENT_CHANGE_STREAM_NO_SIGNAL:
            case RapidAgentSDK.EVENT_PROXY_VERIFY:
            case RapidAgentSDK.EVENT_GET_INDEX_FAIL:
                updateStatus(Status.ERROR);
                break;
        }
        statistic.onEvent(channelId, event, params, desc, callSuccess);
    }

    @Override
    public void onPrtMetric(Map<String, String> params) {
        if (channelId > 0) {
            statistic.onMetric(channelId, params);
        }
    }

    @Override
    public void onPrtState(Map<String, String> state) {

    }

    @Override
    public void onPrtProgress() {
        if (channelId > 0) {
            statistic.notifyStatusEvent(channelId);
        }
    }

    private String getStreamType() {
        if (rapidAgentUri instanceof RapidAgentLiveUri) {
            if (rapidAgentUri.isMicroBlock) {
                switch (rapidAgentUri.getMicroBlockLevel()) {
                    case RapidAgentUri.MICRO_BLOCK_ULTRA_LOW:
                        return "ultra_low";
                    case RapidAgentUri.MICRO_BLOCK_LOW:
                        return "gop1_low";
                    case RapidAgentUri.MICRO_BLOCK_NORMAL:
                        return "normal_low";
                    case RapidAgentUri.MICRO_BLOCK_GOP3:
                    default:
                        return "gop3_low";
                }
            } else {
                return "live";
            }
        } else if (rapidAgentUri instanceof RapidAgentCatchupUri) {
            return "catchup";
        } else if (rapidAgentUri instanceof RapidAgentVodUri) {
            return "vod";
        }
        return "unknown";
    }

    private void printState(String msg) {
        long duration = (System.nanoTime() - startTime) / 1_000_000;
        LogUtil.i(TAG, msg + ":" + this + " " + duration + "ms");
    }

    @NonNull
    @Override
    public String toString() {
        return String.format("RapidAgentCall[%s,%s,%s,%s] %s", rapidAgentUri.getStreamId(), streamType, index, channelId, status);
    }
}
