package com.bitkernel.stream.rapid;

import static com.bitkernel.stream.rapid.utils.CommonUtil.parseFloat;
import static com.bitkernel.stream.rapid.utils.CommonUtil.parseInt;

import androidx.annotation.NonNull;

import com.bitkernel.stream.rapid.config.CommonConfig;
import com.bitkernel.stream.rapid.debug.DebugConfig;
import com.bitkernel.stream.rapid.prt.StreamInfoMgr;
import com.bitkernel.stream.rapid.utils.CommonUtil;
import com.bitkernel.stream.rapid.utils.LogUtil;
import com.bitkernel.stream.rapid.utils.SPUtil;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;

public final class RapidAgentStatistic {
    private static final String METRIC_APP_ID = "rapid_sdk";
    private static final String METRIC_APP_KEY = "rapid_sdk_key";

    private static final String METRIC_EVENT_NAME_STARTUP = "rapid_agent_startup";
    private static final String METRIC_EVENT_NAME_STATUS = "rapid_agent_status";
    private static final String METRIC_EVENT_NAME_EXCEPTION = "rapid_agent_exception";
    private static final String METRIC_EVENT_NAME_BUFFERING = "rapid_agent_buffering";
    private static final String METRIC_EVENT_NAME_COMMON = "rapid_agent_common";

    private static final int MODULE_UNSPECIFIED = 0;
    private static final int MODULE_SDK = 1;
    private static final int MODULE_ENGINE = 2;
    private static final int MODULE_PLAYER = 3;

    public static final int CATEGORY_UNSPECIFIED = 0;
    public static final int CATEGORY_LOGIN = 1;
    public static final int CATEGORY_STARTUP = 2;

    private static final int EXCEPTION_SUBTYPE_SDK_INIT_FAIL = 6;
    private static final int EXCEPTION_SUBTYPE_SDK_UPDATE_FAIL = 7;
    private static final int EXCEPTION_SUBTYPE_SDK_NATIVE_CRASH = 8;

    private static final int COMMON_SUBTYPE_PLAYER_SEEK = 1;
    private static final int COMMON_SUBTYPE_PLAYER_RELEASE = 2;
    private static final int COMMON_SUBTYPE_PLAYER_LIFECYCLE = 3;
    private static final int COMMON_SUBTYPE_LOW_MEDIACODEC = 4;
    private static final int COMMON_SUBTYPE_LIVE_COMPLETED = 5;
    private static final int COMMON_SUBTYPE_AUTO_SWITCH_TRACK = 6;

    private static final int IGNORE_CHANNEL_ID = Integer.MIN_VALUE + 1;

    private static final String DEFAULT_AWS_METRIC_HOST = BuildConfig.DEBUG ? "http://15.204.22.126:2086" : "http://appfmetric.te999.site:2095";
    private static final String DEFAULT_PCT_METRIC_HOST = BuildConfig.DEBUG ? "http://15.204.22.126:2086" : "http://appfmetric.te999.site:2095";

    private static volatile RapidAgentStatistic sInstance;
    private static final AtomicBoolean INIT_REF = new AtomicBoolean();

    private final long initTimestamp;
    private final long initNanoTime;
    private final Map<String, Object> baseInfoMap = new LinkedHashMap<>();
    private final Map<Integer, Map<String, Integer>> startupMap = new LinkedHashMap<>();
    private final Map<Integer, Map<String, Object>> statusMap = new LinkedHashMap<>();

    static void init() {
        if (INIT_REF.compareAndSet(false, true)) {
            HashMap<String, String[]> servers = new HashMap<>();
            servers.put("aws", getRapidMetricHosts());
            servers.put("pct", getPCTMetricHosts());
            LogUtil.sensitive("aws metric url is " + Arrays.toString(servers.get("aws")));
            LogUtil.sensitive("pct metric url is " + Arrays.toString(servers.get("pct")));
            int interval = CommonConfig.metricReportInterval();
            LogUtil.i("RapidAgentStatistic", "metric report interval " + interval);
            RapidMetricSender.init(METRIC_APP_ID, METRIC_APP_KEY, servers, 100, interval * 1000, BuildConfig.DEBUG);

            RapidAgentStatistic.getInstance().initBaseInfo();
        }
    }

    @NonNull
    private static String[] getRapidMetricHosts() {
        String[] metricHosts = null;
        String metricHostString = CommonConfig.getMetricServers();
        if (metricHostString != null) {
            metricHosts = CommonUtil.parseHostArray(metricHostString);
        }
        if (metricHosts != null && metricHosts.length > 0) {
            SPUtil.setRapidMetricHost(metricHostString);
            return metricHosts;
        }
        metricHostString = SPUtil.getRapidMetricHost(DEFAULT_AWS_METRIC_HOST);
        metricHosts = CommonUtil.parseHostArray(metricHostString);
        return metricHosts;
    }

    @NonNull
    private static String[] getPCTMetricHosts() {
        String[] metricHosts = null;
        String metricHostString = CommonConfig.getPctMetricServers();
        if (metricHostString != null) {
            metricHosts = CommonUtil.parseHostArray(metricHostString);
        }
        if (metricHosts != null && metricHosts.length > 0) {
            SPUtil.setPCTMetricHost(metricHostString);
            return metricHosts;
        }
        metricHostString = SPUtil.getPCTMetricHost(DEFAULT_PCT_METRIC_HOST);
        metricHosts = CommonUtil.parseHostArray(metricHostString);
        return metricHosts;
    }

    private RapidAgentStatistic() {
        initTimestamp = System.currentTimeMillis();
        initNanoTime = System.nanoTime();
    }

    static RapidAgentStatistic getInstance() {
        if (sInstance == null) {
            synchronized (RapidAgentStatistic.class) {
                if (sInstance == null) {
                    sInstance = new RapidAgentStatistic();
                }
            }
        }
        return sInstance;
    }

    private void initBaseInfo() {
        baseInfoMap.put("vno_id", RapidAgentConstant.getVnoId());
        baseInfoMap.put("app_name", RapidAgentConstant.getApplicationName());
        baseInfoMap.put("vno_uid", RapidAgentConstant.getAccountId());
        baseInfoMap.put("did", RapidAgentConstant.getDeviceId());
        baseInfoMap.put("cid", RapidAgentConstant.getCID());
        baseInfoMap.put("platform", 1);
        baseInfoMap.put("aver", RapidAgentConstant.getSystemVersion());
        baseInfoMap.put("ever", RapidAgentConstant.getEngineVersion());
        baseInfoMap.put("pver", RapidAgentConstant.getPlayerVersion());
        baseInfoMap.put("sver", RapidAgentConstant.getSdkVersion());
        baseInfoMap.put("brand", RapidAgentConstant.getBrand());
        baseInfoMap.put("model", RapidAgentConstant.getModel());
        baseInfoMap.put("cpu", RapidAgentConstant.getCpuName());
        baseInfoMap.put("init_time", initTimestamp);
        LogUtil.sensitive("initBaseInfo " + baseInfoMap);
    }

    private void sendToServer(String key, String event, Map<String, Object> params) {
        LogUtil.sensitive("start sendToServer(" + key + ") " + event + ":" + params);
        RapidMetricSender.put(key, event, params);
    }

    private void notifyEvent(int channelId, String event, Map<String, Object> params) {
        LogUtil.i("RapidAgentStatistic", "notifyEvent " + event);
        LinkedHashMap<String, Object> data = new LinkedHashMap<>(params);
        if (channelId != IGNORE_CHANNEL_ID) {
            StreamInfoMgr.StreamInfo streamInfo = StreamInfoMgr.getStreamInfo(channelId);
            if (streamInfo.type != StreamInfoMgr.INVALID) {
                data.put("stream_id", streamInfo.streamId);
                data.put("stream_type", streamInfo.type);
            } else if (METRIC_EVENT_NAME_EXCEPTION.equals(event)) {
                data.put("stream_type", StreamInfoMgr.INVALID);
            } else {
                LogUtil.e("RapidAgentStatistic", "send to server fail:invalid channel id");
                return;
            }
        } else {
            data.put("stream_type", StreamInfoMgr.INVALID);
        }
        data.putAll(baseInfoMap);
        data.put("ref_time", (System.nanoTime() - initNanoTime) / 1_000_000);
        sendToServer("aws", event, data);
    }

    public void notifyPCTEvent(String event, Map<String, Object> params) {
        LogUtil.i("RapidAgentStatistic", "notifyPCTEvent " + event);
        LinkedHashMap<String, Object> data = new LinkedHashMap<>(params);
        data.put("app_name", RapidAgentConstant.getApplicationName());
        data.put("vno_uid", RapidAgentConstant.getAccountId());
        data.put("did", RapidAgentConstant.getDeviceId());
        data.put("aver", RapidAgentConstant.getSystemVersion());
        data.put("sver", RapidAgentConstant.getSdkVersion());
        data.put("brand", RapidAgentConstant.getBrand());
        data.put("model", RapidAgentConstant.getModel());
        data.put("cpu", RapidAgentConstant.getCpuName());
        data.put("init_time", initTimestamp);
        data.put("ref_time", (System.nanoTime() - initNanoTime) / 1_000_000);
        sendToServer("pct", event, data);
    }

    public void notifyTrackerServerIdChanged(String serverId) {
        baseInfoMap.put("tr_sv_id", serverId);
    }

    public void notifyPrtServerIdChanged(String serverId) {
        baseInfoMap.put("prt_sv_id", serverId);
    }

    public synchronized void notifyPlayerStartupTime(int channelId, int playerType, JSONObject jsonObject) throws JSONException {
        int streamInfo = jsonObject.has("find_streaminfo") ? jsonObject.getInt("find_streaminfo") : 0;
        int decOpen = jsonObject.has("component_open") ? jsonObject.getInt("component_open") : 0;
        int audioDec = jsonObject.has("audio_decode") ? jsonObject.getInt("audio_decode") : 0;
        int videoDec = jsonObject.has("video_decode") ? jsonObject.getInt("video_decode") : 0;
        int audioRender = jsonObject.has("audio_render") ? jsonObject.getInt("audio_render") : 0;
        int videoRender = jsonObject.has("video_render") ? jsonObject.getInt("video_render") : 0;
        int keyLoaded = jsonObject.has("key_load") ? jsonObject.getInt("key_load") : 0;
        StreamInfoMgr.StreamInfo info = StreamInfoMgr.getStreamInfo(channelId);
        int engineTotal = 0;
        int playerTotal = 0;
        int allTotal = 0;
        if (info.type != StreamInfoMgr.INVALID) {
            engineTotal = (int) ((info.engineReadyTime - info.startupTime) / 1000_000);
            int sdkTotal = (int) ((info.playerStartupTime - info.engineReadyTime) / 1000_000);
            playerTotal = videoRender > 0 ? videoRender : audioRender;
            allTotal = engineTotal + sdkTotal + playerTotal;
        }
        if (engineTotal <= 0 || playerTotal <= 0) {
            throw new IllegalStateException("invalid startup info: " + engineTotal + " " + playerTotal);
        }
        boolean completed = false;
        Map<String, Integer> map = null;
        int channelHash = info.hashCode();
        if (startupMap.containsKey(channelHash)) {
            completed = true;
            map = startupMap.get(channelHash);
        }
        if (map == null) {
            completed = false;
            map = new LinkedHashMap<>();
        }
        map.put("player_type", playerType);
        map.put("first_ts", engineTotal);
        map.put("total", allTotal);
        if (streamInfo > 0) {
            map.put("stream_info", streamInfo);
        }
        if (decOpen > 0) {
            map.put("dec_open", decOpen);
        }
        if (keyLoaded > 0) {
            map.put("key_load", keyLoaded);
        }
        if (audioDec > 0) {
            map.put("audec", audioDec);
        } else if (videoDec > 0) {
            map.put("audec", videoDec);
        }
        if (videoDec > 0) {
            map.put("videc", videoDec);
        }
        if (audioRender > 0) {
            map.put("aurend", audioRender);
        }
        map.put("virend", playerTotal);
        if (completed) {
            notifyStartupTime(channelId, map);
            startupMap.remove(channelHash);
        } else {
            startupMap.put(channelHash, map);
        }
        if (videoRender >= CommonConfig.getPlayerStartupSlow()) {
            RapidAgentSDK.triggerLogcat(DebugConfig.DEBUG_WHAT_COMMON, DebugConfig.DEBUG_EXTRA_COMMON_STARTUP_LONG);
        }
    }

    public synchronized void notifyEngineStartupTime(int channelId, Map<String, String> params) {
        String trackRTT = params.get("tRRtt");
        String firstTrack = params.get("firstAddrTrMs");
        String prtRTT = params.get("prtRtt");
        String firstIndex = params.get("firstM3u8Ms");
        String firstGOP = params.get("flowGopMs");
        String firstTs = params.get("flowTsMs");
        String tsUse = params.get("tsUseMs");
        boolean completed = false;
        Map<String, Integer> map = null;
        StreamInfoMgr.StreamInfo info = StreamInfoMgr.getStreamInfo(channelId);
        int channelHash = info.hashCode();
        if (startupMap.containsKey(channelHash)) {
            completed = true;
            map = startupMap.get(channelHash);
        }
        if (map == null) {
            completed = false;
            map = new LinkedHashMap<>();
        }
        if (info.playerStartupTime == 0) {
            LogUtil.w("RapidAgentStatistic", "only notify engine startup info when rapid play is not used");
            completed = true;
            int engineTotal = (int) ((info.engineReadyTime - info.startupTime) / 1000_000);
            if (engineTotal <= 0) {
                throw new IllegalStateException("invalid startup info: " + engineTotal);
            }
            map.put("first_ts", engineTotal);
            map.put("player_type", 9999);
        }
        map.put("tr_rtt", parseInt(trackRTT, 0));
        map.put("first_tr", parseInt(firstTrack, 0));
        map.put("prt_rtt", parseInt(prtRTT, 0));
        map.put("first_index", parseInt(firstIndex, 0));
        int firstGOPValue = parseInt(firstGOP, 0);
        if (firstGOPValue == 0) {
            firstGOPValue = parseInt(firstTs, 0);
        }
        map.put("first_gop", firstGOPValue);
        map.put("ts_use", parseInt(tsUse, 0));
        if (completed) {
            notifyStartupTime(channelId, map);
            startupMap.remove(channelHash);
        } else {
            startupMap.put(channelHash, map);
        }
    }

    private void notifyStartupTime(int channelId, Map<String, Integer> map) {
        notifyEvent(channelId, METRIC_EVENT_NAME_STARTUP, new LinkedHashMap<>(map));
    }

    private @NonNull
    Map<String, Object> findStatusMap(int channelId) {
        Map<String, Object> map = null;
        StreamInfoMgr.StreamInfo info = StreamInfoMgr.getStreamInfo(channelId);
        int channelHash = info.hashCode();
        if (statusMap.containsKey(channelHash)) {
            map = statusMap.get(channelHash);
        }
        if (map == null) {
            for (Map.Entry<Integer, Map<String, Object>> entry : statusMap.entrySet()) {
                if (!StreamInfoMgr.checkStreamInfoByHashCode(entry.getKey())) {
                    statusMap.remove(entry.getKey());
                    LogUtil.d("RapidAgentStatistic", "remove status map " + entry.getKey());
                    break;
                }
            }
            map = new LinkedHashMap<>();
            statusMap.put(channelHash, map);
            LogUtil.d("RapidAgentStatistic", "add status map " + channelHash);
        }
        return map;
    }

    public void notifyPlayerStatus(int channelId, Map<String, String> params) {
        StreamInfoMgr.StreamInfo streamInfo = StreamInfoMgr.getStreamInfo(channelId);
        if (streamInfo.isLive()) {
            Map<String, Object> map = findStatusMap(channelId);
            float outputFPS = parseFloat(params.get("outputFPS"), 0);
            float decFPS = parseFloat(params.get("decFPS"), 0);
            map.put("dec_rate", decFPS);
            map.put("ren_rate", outputFPS);
            if (params.containsKey("playDelayMs")) {
                int playDelayMs = parseInt(params.get("playDelayMs"), 0);
                if (playDelayMs > 0) {
                    map.put("play_delay", playDelayMs);
                }
            }
            if (streamInfo.isMicroBlock() && CommonConfig.getStatisticLevel() >= CommonConfig.STATISTIC_LEVEL_DEBUG) {
                if (params.containsKey("maxCostMs")) {
                    int maxCostMs = parseInt(params.get("maxCostMs"), 0);
                    if (maxCostMs > 0) {
                        map.put("max_cost", maxCostMs);
                    }
                }
                if (params.containsKey("dropRate")) {
                    float dropRate = parseFloat(params.get("dropRate"), -1);
                    if (dropRate >= 0) {
                        map.put("los_rate", dropRate);
                    }
                }
                if (params.containsKey("minVideoBuf") && params.containsKey("maxVideoBuf")) {
                    int minVideoBuf = parseInt(params.get("minVideoBuf"), -1);
                    int maxVideoBuf = parseInt(params.get("maxVideoBuf"), -1);
                    int idealVideoBuf = parseInt(params.get("idealVideoBuf"), -1);
                    if (minVideoBuf >= 0 && maxVideoBuf >= 0) {
                        map.put("min_buf", minVideoBuf);
                        map.put("max_buf", maxVideoBuf);
                        map.put("idl_buf", idealVideoBuf);
                    }
                }
            }
        }
    }

    public void notifyEngineStatus(int channelId, Map<String, Object> params) {
        Map<String, Object> map = findStatusMap(channelId);
        map.putAll(params);
        notifyEvent(channelId, METRIC_EVENT_NAME_STATUS, map);
    }

    public void notifySDKInitError(String desc) {
        Map<String, Object> map = new LinkedHashMap<>();
        map.put("exp_type", formatEventType(MODULE_SDK, CATEGORY_LOGIN));
        map.put("exp_subtype", EXCEPTION_SUBTYPE_SDK_INIT_FAIL);
        Map<String, Object> descMap = new LinkedHashMap<>();
        descMap.put("desc", desc);
        map.put("exp_desc", formatDesc(descMap));
        notifyEvent(IGNORE_CHANNEL_ID, METRIC_EVENT_NAME_EXCEPTION, map);
    }

    public void notifySDKUpdateError(String desc) {
        Map<String, Object> map = new LinkedHashMap<>();
        map.put("exp_type", formatEventType(MODULE_SDK, CATEGORY_UNSPECIFIED));
        map.put("exp_subtype", EXCEPTION_SUBTYPE_SDK_UPDATE_FAIL);
        Map<String, Object> descMap = new LinkedHashMap<>();
        descMap.put("desc", desc);
        map.put("exp_desc", formatDesc(descMap));
        notifyEvent(IGNORE_CHANNEL_ID, METRIC_EVENT_NAME_EXCEPTION, map);
    }

    public void notifySDKNativeCrash(String desc) {
        Map<String, Object> map = new LinkedHashMap<>();
        map.put("exp_type", formatEventType(MODULE_SDK, CATEGORY_UNSPECIFIED));
        map.put("exp_subtype", EXCEPTION_SUBTYPE_SDK_NATIVE_CRASH);
        Map<String, Object> descMap = new LinkedHashMap<>();
        descMap.put("desc", desc);
        map.put("exp_desc", formatDesc(descMap));
        notifyEvent(IGNORE_CHANNEL_ID, METRIC_EVENT_NAME_EXCEPTION, map);
    }

    public void notifyPlayerError(int channelId, int playerType, int what, int extra, String desc, int category) {
        Map<String, Object> map = new LinkedHashMap<>();
        map.put("exp_type", formatEventType(MODULE_PLAYER, category));
        map.put("exp_subtype", extra);
        Map<String, Object> descMap = new LinkedHashMap<>();
        descMap.put("player_type", playerType);
        descMap.put("what", what);
        descMap.put("extra", extra);
        descMap.put("desc", desc);
        map.put("exp_desc", formatDesc(descMap));
        notifyEvent(channelId, METRIC_EVENT_NAME_EXCEPTION, map);
        RapidAgentSDK.triggerLogcat(what, extra);
    }

    public void notifyPlayerBuffering(int channelId, int playerType, int extra, long duration, String startReason, String endReason) {
        StreamInfoMgr.StreamInfo streamInfo = StreamInfoMgr.getStreamInfo(channelId);
        if (streamInfo.playerReadyTime == 0) {
            return;
        }
        long playDuration = (System.nanoTime() - streamInfo.playerReadyTime) / 1000_000_000L;
        Map<String, Object> map = new LinkedHashMap<>();
        map.put("player_type", playerType);
        map.put("extra", extra);
        map.put("play_dur", playDuration);
        map.put("buf_dur", duration);
        map.put("start_reason", startReason);
        map.put("end_reason", endReason);
        notifyEvent(channelId, METRIC_EVENT_NAME_BUFFERING, map);
    }

    public void notifyEngineError(int channelId, int what, int extra, String desc, int category) {
        Map<String, Object> map = new LinkedHashMap<>();
        map.put("exp_type", formatEventType(MODULE_ENGINE, category));
        map.put("exp_subtype", what);
        Map<String, Object> descMap = new LinkedHashMap<>();
        descMap.put("what", what);
        descMap.put("extra", extra);
        descMap.put("desc", desc);
        map.put("exp_desc", formatDesc(descMap));
        notifyEvent(channelId, METRIC_EVENT_NAME_EXCEPTION, map);
        RapidAgentSDK.triggerLogcat(what, extra);
    }

    public void notifySeekPerformance(int channelId, int playerType, long duration) {
        Map<String, Object> map = new LinkedHashMap<>();
        map.put("type", formatEventType(MODULE_PLAYER, CATEGORY_UNSPECIFIED));
        map.put("subtype", COMMON_SUBTYPE_PLAYER_SEEK);
        Map<String, Object> descMap = new LinkedHashMap<>();
        descMap.put("player_type", playerType);
        descMap.put("dur", duration);
        map.put("desc", formatDesc(descMap));
        notifyEvent(channelId, METRIC_EVENT_NAME_COMMON, map);
    }

    public void notifyReleasePerformance(int channelId, int playerType, long stopDuration, long releaseDuration, long lifecycleDuration) {
        Map<String, Object> map = new LinkedHashMap<>();
        map.put("type", formatEventType(MODULE_PLAYER, CATEGORY_UNSPECIFIED));
        map.put("subtype", COMMON_SUBTYPE_PLAYER_RELEASE);
        Map<String, Object> descMap = new LinkedHashMap<>();
        descMap.put("player_type", playerType);
        descMap.put("stop", stopDuration);
        descMap.put("release", releaseDuration);
        descMap.put("life", lifecycleDuration);
        map.put("desc", formatDesc(descMap));
        notifyEvent(channelId, METRIC_EVENT_NAME_COMMON, map);
    }

    public void notifyLiveCompleted(int channelId, int playerType) {
        Map<String, Object> map = new LinkedHashMap<>();
        map.put("type", formatEventType(MODULE_PLAYER, CATEGORY_UNSPECIFIED));
        map.put("subtype", COMMON_SUBTYPE_LIVE_COMPLETED);
        Map<String, Object> descMap = new LinkedHashMap<>();
        descMap.put("player_type", playerType);
        map.put("desc", formatDesc(descMap));
        notifyEvent(channelId, METRIC_EVENT_NAME_COMMON, map);
    }

    public void notifyLowMediaCodec(boolean avcAvailable, boolean hevcAvailable) {
        Map<String, Object> map = new LinkedHashMap<>();
        map.put("type", formatEventType(MODULE_PLAYER, CATEGORY_UNSPECIFIED));
        map.put("subtype", COMMON_SUBTYPE_LOW_MEDIACODEC);
        map.put("desc", (avcAvailable ? (1 << 1) : 0) | (hevcAvailable ? 1 : 0));
        notifyEvent(IGNORE_CHANNEL_ID, METRIC_EVENT_NAME_COMMON, map);
    }

    public void notifyAutoSwitchTrack(int channelId, int bitrate) {
        Map<String, Object> map = new LinkedHashMap<>();
        map.put("type", formatEventType(MODULE_PLAYER, CATEGORY_UNSPECIFIED));
        map.put("subtype", COMMON_SUBTYPE_AUTO_SWITCH_TRACK);
        map.put("desc", bitrate);
        notifyEvent(channelId, METRIC_EVENT_NAME_COMMON, map);
    }

    private static String formatDesc(Map<String, Object> map) {
        return map.toString();
    }

    private static int formatEventType(int module, int category) {
        return category << 3 | module;
    }
}
