package com.bitkernel.stream.rapid.prt;

import static com.bitkernel.stream.rapid.prt.PrtEngine.setTokenSecretKey;

import android.text.TextUtils;

import androidx.annotation.NonNull;

import com.bitkernel.stream.rapid.RapidAgentConstant;
import com.bitkernel.stream.rapid.RapidAgentUri;
import com.bitkernel.stream.rapid.config.ServerAddress;
import com.bitkernel.stream.rapid.utils.LogUtil;
import com.bitkernel.stream.rapid.utils.NetworkManager;
import com.bitkernel.stream.rapid.utils.SHA;

import java.io.IOException;
import java.util.LinkedHashMap;
import java.util.Map;

public final class PrtConfig {
    private static final ServerAddress LIVE_SERVER_ADDRESS = new ServerAddress();
    private static final ServerAddress VOD_SERVER_ADDRESS = new ServerAddress();
    private static final ServerAddress CATCHUP_SERVER_ADDRESS = new ServerAddress();
    private static final Map<String, Object> DEFAULT_OPTIONS = new LinkedHashMap<>();
    private static final String DEFAULT_LIVE_LOG_HOST = "51.81.245.92:40000";
    private static final String DEFAULT_VOD_LOG_HOST = "158.69.246.214:39000";
    private static final int sMaxFlashPer;
    private static final int sMinFreeFlash;
    private static String sSecretKey = null;

    static {
        int storageSize = RapidAgentConstant.getStorageSizeGb();
        if (storageSize >= 64) {
            sMaxFlashPer = 50;
            sMinFreeFlash = 100;
        } else if (storageSize >= 16) {
            sMaxFlashPer = 50;
            sMinFreeFlash = 100;
        } else {
            sMaxFlashPer = 50;
            sMinFreeFlash = 100;
        }
    }

    public static void setDefaultOptions(Map<String, Object> map) {
        DEFAULT_OPTIONS.putAll(map);
        Object tokenSecretKey = DEFAULT_OPTIONS.remove("secret_key");
        if (tokenSecretKey instanceof String) {
            LogUtil.i("PrtConfig", "found a token secret key");
            sSecretKey = (String) tokenSecretKey;
            setTokenSecretKey(sSecretKey);
        } else {
            LogUtil.e("PrtConfig", "cant find a token secret key");
        }
    }

    private static String createConnectId() {
        return String.valueOf(RapidAgentConstant.getCID());
    }

    private static Object getDefaultOption(String key, Object defaultValue) {
        if (DEFAULT_OPTIONS.containsKey(key)) {
            return DEFAULT_OPTIONS.get(key);
        }
        return defaultValue;
    }

    public static void setLiveServerAddress(ServerAddress address) {
        LIVE_SERVER_ADDRESS.copy(address);
    }

    static ServerAddress getLiveServerAddress() {
        return LIVE_SERVER_ADDRESS;
    }

    public static void setVodServerAddress(ServerAddress address) {
        VOD_SERVER_ADDRESS.copy(address);
    }

    static ServerAddress getVodServerAddress() {
        return VOD_SERVER_ADDRESS;
    }

    public static void setCatchupServerAddress(ServerAddress address) {
        CATCHUP_SERVER_ADDRESS.copy(address);
    }

    static ServerAddress getCatchupServerAddress() {
        return CATCHUP_SERVER_ADDRESS;
    }

    public static String getTrackerServerId(String ip) {
        if (ip == null) {
            return null;
        }
        String serverId = LIVE_SERVER_ADDRESS.getTrackerServerId(ip);
        if (serverId != null) {
            return serverId;
        }
        serverId = VOD_SERVER_ADDRESS.getTrackerServerId(ip);
        if (serverId != null) {
            return serverId;
        }
        serverId = CATCHUP_SERVER_ADDRESS.getTrackerServerId(ip);
        if (serverId != null) {
            return serverId;
        }
        return "unknown";
    }

    public static boolean hasValidSecretKey() {
        return !TextUtils.isEmpty(sSecretKey);
    }

    static Map<String, Object> getInitLiveOptions(String liveTracker, @NonNull String userToken) {
        Map<String, Object> initLiveOptions = new LinkedHashMap<>();
        initLiveOptions.put("wkm", 1);
        initLiveOptions.put("aid", RapidAgentConstant.getAID());
        initLiveOptions.put("cid", createConnectId());
        initLiveOptions.put("tkn", userToken);
        initLiveOptions.put("prt", 1);
        initLiveOptions.put("localip", NetworkManager.getIpAddress());
        if (!TextUtils.isEmpty(liveTracker)) {
            initLiveOptions.put("th", liveTracker);
        }
        initLiveOptions.put("plfm", RapidAgentConstant.getPlatformType());
        return initLiveOptions;
    }

    static Map<String, Object> getStartLiveSysOptions() {
        Map<String, Object> liveSysOptions = new LinkedHashMap<>();
        liveSysOptions.put("aid", RapidAgentConstant.getAID());
        liveSysOptions.put("cid", createConnectId());
        liveSysOptions.put("prt", 1);
        liveSysOptions.put("localip", NetworkManager.getIpAddress());
        liveSysOptions.put("tacker_group", getDefaultOption("tacker_group", "s"));
        liveSysOptions.put("ssctype", getDefaultOption("ssctype", 0));
        liveSysOptions.put("tkbt", getDefaultOption("tkbt", 10000));
        liveSysOptions.put("rid", getDefaultOption("rid", ""));
        liveSysOptions.put("loghost", getDefaultOption("loghost", DEFAULT_LIVE_LOG_HOST));
        liveSysOptions.put("rname", RapidAgentConstant.getApplicationName());
        liveSysOptions.put("appver", RapidAgentConstant.getSdkVersion());
        return liveSysOptions;
    }

    static Map<String, Object> getStartLiveChannelOptions(@NonNull String playToken, String tracker, String defaultPrt, String turboPrt, String channelId, String channelName, int solutionType, int solutionBrand) {
        Map<String, Object> liveChannelOptions = new LinkedHashMap<>();
        liveChannelOptions.put("ptkn", playToken);
        if (!TextUtils.isEmpty(tracker)) {
            liveChannelOptions.put("th", tracker);
        }
        liveChannelOptions.put("ph", defaultPrt);
        liveChannelOptions.put("tph", turboPrt);
        liveChannelOptions.put("stream", channelId);
        liveChannelOptions.put("chan", "/" + channelId);
        liveChannelOptions.put("chid", channelId);
        liveChannelOptions.put("port-b", 41000);
        liveChannelOptions.put("port-r", 1000);
        liveChannelOptions.put("mask", getDefaultOption("mask", "0"));
        liveChannelOptions.put("chname", channelName);
        liveChannelOptions.put("soluty", solutionType);
        liveChannelOptions.put("soluba", solutionBrand);
        liveChannelOptions.put("eproxy", 1);
        return liveChannelOptions;
    }

    static Map<String, Object> getInitCatchupOptions(String catchUpTracker, @NonNull String userToken) {
        Map<String, Object> initCatchupOptions = new LinkedHashMap<>();
        initCatchupOptions.put("aid", RapidAgentConstant.getAID());
        initCatchupOptions.put("cid", createConnectId());
        initCatchupOptions.put("cuth", catchUpTracker);
        initCatchupOptions.put("fdir", RapidAgentConstant.getPrtPmCacheDir());
        initCatchupOptions.put("prt", 1);
        initCatchupOptions.put("localip", NetworkManager.getIpAddress());
        initCatchupOptions.put("tkn", userToken);
        initCatchupOptions.put("maxFlashPer", getDefaultOption("maxFlashPer", sMaxFlashPer));
        initCatchupOptions.put("minFreeFlash", getDefaultOption("minFreeFlash", sMinFreeFlash));
        initCatchupOptions.put("fpbn", getDefaultOption("fpbn", 20));//pre buffer block size before playing
        initCatchupOptions.put("loghost", getDefaultOption("vodloghost", DEFAULT_VOD_LOG_HOST));
        initCatchupOptions.put("tkbt", getDefaultOption("tkbt", 10000));
        initCatchupOptions.put("plfm", RapidAgentConstant.getPlatformType());
        initCatchupOptions.put("pmak", sSecretKey);
        return initCatchupOptions;
    }

    static Map<String, Object> getRequestCatchupInfoOptions(@NonNull String playToken, String channelId, String epgId, long start, long duration, String tracker, String pmHost, int solutionType, int solutionBrand) {
        Map<String, Object> catchupChannelOptions = new LinkedHashMap<>();
        catchupChannelOptions.put("ptkn", playToken);
        catchupChannelOptions.put("epgId", epgId);
        catchupChannelOptions.put("start", start);
        catchupChannelOptions.put("duration", duration);
        catchupChannelOptions.put("crt", 1);
        catchupChannelOptions.put("pt", getProtocol(solutionType));
        catchupChannelOptions.put("stream", channelId);
        catchupChannelOptions.put("chan", SHA.sha1(String.format("epgId=%s&start=%s&duration=%s&rand=%s", epgId, start, duration, System.nanoTime())));
        catchupChannelOptions.put("chid", channelId);
        catchupChannelOptions.put("mask", getDefaultOption("mask", "0"));
        catchupChannelOptions.put("rqxinfo", 1);
        if (!TextUtils.isEmpty(tracker)) {
            catchupChannelOptions.put("th", tracker);
        }
        if (!TextUtils.isEmpty(pmHost)) {
            catchupChannelOptions.put("ph", pmHost);
        }
        catchupChannelOptions.put("soluty", solutionType);
        catchupChannelOptions.put("soluba", solutionBrand);
        catchupChannelOptions.put("eproxy", 1);
        return catchupChannelOptions;
    }

    static Map<String, Object> getStartCatchupChannelOptions(@NonNull String playToken, String tracker, int solutionType, int solutionBrand) {
        Map<String, Object> catchupChannelOptions = new LinkedHashMap<>();
        catchupChannelOptions.put("ptkn", playToken);
        catchupChannelOptions.put("cuth", tracker);
        catchupChannelOptions.put("crt", 1);
        catchupChannelOptions.put("useBlockIdMap", 1);
        catchupChannelOptions.put("soluty", solutionType);
        catchupChannelOptions.put("soluba", solutionBrand);
        catchupChannelOptions.put("eproxy", 1);
        catchupChannelOptions.put("pt", getProtocol(solutionType));
        return catchupChannelOptions;
    }

    static Map<String, Object> getInitVodOptions(String tracker, @NonNull String userToken) {
        Map<String, Object> initVodOptions = new LinkedHashMap<>();
        initVodOptions.put("aid", RapidAgentConstant.getAID());
        initVodOptions.put("cid", createConnectId());
        if (!TextUtils.isEmpty(tracker)) {
            initVodOptions.put("th", tracker);
        }
        initVodOptions.put("fdir", RapidAgentConstant.getPrtPmCacheDir());
        initVodOptions.put("prt", 1);
        initVodOptions.put("localip", NetworkManager.getIpAddress());
        initVodOptions.put("tkn", userToken);
        initVodOptions.put("maxFlashPer", getDefaultOption("maxFlashPer", 50));
        initVodOptions.put("minFreeFlash", getDefaultOption("minFreeFlash", 100));
        initVodOptions.put("fpbn", getDefaultOption("fpbn", 20));//pre buffer block size before playing
        initVodOptions.put("loghost", getDefaultOption("vodloghost", DEFAULT_VOD_LOG_HOST));
        initVodOptions.put("tkbt", getDefaultOption("tkbt", 10000));
        initVodOptions.put("plfm", RapidAgentConstant.getPlatformType());
        initVodOptions.put("pmak", sSecretKey);
        return initVodOptions;
    }

    static Map<String, Object> getRequestVodInfoOptions(@NonNull String playToken, String channelId, String tracker, String pmHost, int solutionType, int solutionBrand) throws IOException {
        Map<String, Object> vodChannelOptions = new LinkedHashMap<>();
        vodChannelOptions.put("ptkn", playToken);
        vodChannelOptions.put("fsize", 524288 * 100);
        vodChannelOptions.put("usize", 524288);
        vodChannelOptions.put("vnoId", RapidAgentConstant.getVnoId());
        vodChannelOptions.put("pt", getProtocol(solutionType));
        vodChannelOptions.put("stream", channelId);
        vodChannelOptions.put("chan", SHA.sha1(String.format("stream=%s&rand=%s", channelId, System.nanoTime())));
        vodChannelOptions.put("chid", channelId);
        vodChannelOptions.put("mask", getDefaultOption("mask", "0"));
        vodChannelOptions.put("rqxinfo", 1);
        if (!TextUtils.isEmpty(tracker)) {
            vodChannelOptions.put("th", tracker);
        }
        if (!TextUtils.isEmpty(pmHost)) {
            vodChannelOptions.put("ph", pmHost);
        }
        vodChannelOptions.put("soluty", solutionType);
        vodChannelOptions.put("soluba", solutionBrand);
        vodChannelOptions.put("eproxy", 1);
        return vodChannelOptions;
    }

    static Map<String, Object> getStartVodChannelOptions(@NonNull String playToken, String tracker, int solutionType, int solutionBrand) {
        Map<String, Object> vodChannelOptions = new LinkedHashMap<>();
        vodChannelOptions.put("ptkn", playToken);
        vodChannelOptions.put("cuth", tracker);
        vodChannelOptions.put("useBlockIdMap", 1);
        vodChannelOptions.put("soluty", solutionType);
        vodChannelOptions.put("soluba", solutionBrand);
        vodChannelOptions.put("eproxy", 1);
        vodChannelOptions.put("pt", getProtocol(solutionType));
        return vodChannelOptions;
    }

    private static String getProtocol(int solutionType) {
        switch (solutionType) {
            case PrtEngine.SOLUTION_TYPE_NONE:
            case PrtEngine.SOLUTION_TYPE_LIVE_VALO_HLS:
            case PrtEngine.SOLUTION_TYPE_LIVE_PLAIN_RPD:
                return "PRPD";
            case PrtEngine.SOLUTION_TYPE_VOD_PLAIN_RPD:
                return "RPD";
            case PrtEngine.SOLUTION_TYPE_VOD_MPT:
                return "MPT";
            case PrtEngine.SOLUTION_TYPE_VOD_MPQ:
                return "MPQ";
            case PrtEngine.SOLUTION_TYPE_LIVE_DRM_HLS:
            case PrtEngine.SOLUTION_TYPE_LIVE_DRM_DASH:
            case PrtEngine.SOLUTION_TYPE_VOD_DRM_HLS:
            case PrtEngine.SOLUTION_TYPE_VOD_DRM_DASH:
                return "DRM";
            case PrtEngine.SOLUTION_TYPE_VOD_TIME_SHIFT_PLAIN_RPD:
            case PrtEngine.SOLUTION_TYPE_VOD_TIME_SHIFT_DRM_HLS:
            case PrtEngine.SOLUTION_TYPE_VOD_TIME_SHIFT_DRM_DASH:
                return "TIMESHIFT";
        }

        throw new IllegalStateException("getProtocol: invalid solutionType " + solutionType);
    }

    public static void printConfig() {
        LogUtil.i("PrtConfig", "engine config : \nliveServerUrl=" + LIVE_SERVER_ADDRESS +
                ",\nvodServerUrl=" + VOD_SERVER_ADDRESS +
                ",\ncatchupServerUrl=" + CATCHUP_SERVER_ADDRESS);
        LogUtil.sensitive("engine config : " + DEFAULT_OPTIONS);
    }
}
