package com.bitkernel.stream.rapid.prt;

import static com.bitkernel.stream.rapid.utils.CommonUtil.parseLong;
import static com.stream.prt.PrtCallBack.DEFAULT_REQUEST_ID;

import android.text.TextUtils;
import android.util.Pair;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.bitkernel.stream.rapid.RapidAgentCatchupUri;
import com.bitkernel.stream.rapid.RapidAgentConstant;
import com.bitkernel.stream.rapid.RapidAgentLiveUri;
import com.bitkernel.stream.rapid.RapidAgentSDK;
import com.bitkernel.stream.rapid.RapidAgentUri;
import com.bitkernel.stream.rapid.RapidAgentVodUri;
import com.bitkernel.stream.rapid.config.CommonConfig;
import com.bitkernel.stream.rapid.config.ServerAddress;
import com.bitkernel.stream.rapid.utils.LogUtil;
import com.bitkernel.stream.rapid.utils.SPUtil;
import com.stream.prt.NetworkState;
import com.stream.prt.PrtListenerManager;
import com.stream.prt.utils.AESSecurityPrt;
import com.stream.prt.utils.JniApiImpl;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.RejectedExecutionHandler;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;

public final class PrtEngine {
    private static final String TAG = "PrtEngine";

    /* 当前使用的是那种方案 */
    public static final int SOLUTION_TYPE_NONE = 0;
    /* 点播自有方案 */
    public static final int SOLUTION_TYPE_VOD_MPT = 1;
    public static final int SOLUTION_TYPE_VOD_MPQ = 2;
    /* 直播自有方案，单个m3u8索引文件, 区分drm hls */
    public static final int SOLUTION_TYPE_LIVE_VALO_HLS = 3;
    /* DRM方案下的 hls 和 dash */
    public static final int SOLUTION_TYPE_LIVE_DRM_HLS = 4;
    public static final int SOLUTION_TYPE_LIVE_DRM_DASH = 5;
    /* 点播清流, 清流不加解密 */
    public static final int SOLUTION_TYPE_VOD_PLAIN_RPD = 6;
    /* 点播drm方案, 点播drm方案用索引中的key解密 */
    public static final int SOLUTION_TYPE_VOD_DRM_HLS = 7;
    public static final int SOLUTION_TYPE_VOD_DRM_DASH = 8;

    public static final int SOLUTION_TYPE_LIVE_PLAIN_RPD = 9;

    /* 点播时移清流, 时移是用点播逻辑实现，但索引像直播一样不断更新 */
    public static final int SOLUTION_TYPE_VOD_TIME_SHIFT_PLAIN_RPD = 100;
    /* 点播时移drm方案, 点播drm方案用索引中的key解密 */
    public static final int SOLUTION_TYPE_VOD_TIME_SHIFT_DRM_HLS = 101;
    public static final int SOLUTION_TYPE_VOD_TIME_SHIFT_DRM_DASH = 102;

    /* 方案品牌商，同种方案下不同品牌索引上有些微小差别 */
    public static final int SOLUTION_BRAND_NONE = 0;
    public static final int SOLUTION_BRAND_GOOSE = 1;
    public static final int SOLUTION_BRAND_IRDETO = 2;

    private static final int PLAY_TYPE_LIVE = 0;
    private static final int PLAY_TYPE_CATCHUP = 1;
    private static final int PLAY_TYPE_VOD = 2;
    private static final int PLAY_TYPE_TIME_SHIFT = 3;

    private static final TokenGenerator sTokenGenerator = new TokenGenerator();
    private static int sGlobalPlayType = -1;
    private static int sGlobalInstanceCount = 0;

    private static final ThreadPoolExecutor sThreadPoolExecutor;
    private static final BlockingQueue<Runnable> sPoolWorkQueue = new LinkedBlockingQueue<>(8);
    private static final ThreadFactory sThreadFactory = new ThreadFactory() {
        private final AtomicInteger count = new AtomicInteger(1);

        public Thread newThread(Runnable r) {
            Thread thread = new Thread(r, "PrtThread#" + count.getAndIncrement());
            LogUtil.e("PrtThreadPool", "newThread " + thread.getName());
            return thread;
        }
    };
    private static final RejectedExecutionHandler sRejectedHandler = new ThreadPoolExecutor.AbortPolicy() {
        @Override
        public void rejectedExecution(Runnable r, ThreadPoolExecutor e) {
            super.rejectedExecution(r, e);
            LogUtil.e("PrtThreadPool", "rejectedExecution");
        }
    };

    private DefaultJniPrtListener jniPrtListener;
    private RapidAgentUri rapidAgentUri;
    private String currentStreamId;
    private int solutionType = SOLUTION_TYPE_NONE;
    private int solutionBrand = SOLUTION_BRAND_NONE;
    private int currChannelIndex = 0;
    private int currentPlayType = -1;
    private int bufferingTimes = 0;

    static {
        sThreadPoolExecutor = new ThreadPoolExecutor(1, 1, 1, TimeUnit.SECONDS,
                sPoolWorkQueue, sThreadFactory, sRejectedHandler);
        sThreadPoolExecutor.allowCoreThreadTimeOut(false);
    }

    public static void runOnWorkThread(Runnable work) {
        sThreadPoolExecutor.execute(work);
    }

    public static void init(final String workPath, final String cfgFile) {
        runOnWorkThread(() -> {
            LogUtil.e("PrtConnect", "start init prt engine");
            JniApiImpl.getInstance().init(workPath, cfgFile, null);
            LogUtil.e("PrtConnect", "finish init prt engine");
        });
    }

    public static void setNetworkState(NetworkState state, String localIp) {
        JniApiImpl.getInstance().setNetworkState(state, localIp);
    }

    public static void setRunningMode(int mode) {
        //foreground 0 background 1
        JniApiImpl.getInstance().setRunningMode(mode);
    }

    public static void setLog(boolean enable, int period) {
        JniApiImpl.getInstance().setLog(enable, period);
    }

    public void notifyBuffering(int ms) {
        JniApiImpl.getInstance().notifyBuffering(currChannelIndex, ++bufferingTimes, ms);
    }

    public void sendFrontLog(String log) {
        JniApiImpl.getInstance().sendFrontLog(currChannelIndex, log);
    }

    public int startup(RapidAgentUri rapidAgentUri, @NonNull DefaultJniPrtListener jniPrtListener) throws IOException {
        try {
            return startupInternal(rapidAgentUri, jniPrtListener);
        } catch (IOException e) {
            jniPrtListener.onEvent(currChannelIndex, RapidAgentSDK.EVENT_CREATE_CHANNEL_FAIL, "", "create channel exception:" + e.getMessage());
            throw new IOException(e);
        }
    }

    private int startupInternal(RapidAgentUri rapidAgentUri, @NonNull DefaultJniPrtListener jniPrtListener) throws IOException {
        this.rapidAgentUri = rapidAgentUri;
        int drmType = rapidAgentUri.getDrmType();
        int protocol = rapidAgentUri.getProtocol();
        String drmLicenseUrl = rapidAgentUri.getDrmLicenseUrl();
        String vnoKey = rapidAgentUri.getVnoKey();
        long startupTime = System.nanoTime();
        currentStreamId = rapidAgentUri.getStreamId();
        String userToken;
        String playToken;
        int channelType;
        int useBlockIdMap;
        int blockSize = 0;
        long fileSize = 0;

        if (sGlobalInstanceCount > 0) {
            if (sGlobalPlayType == PLAY_TYPE_LIVE) {
                if (!(rapidAgentUri instanceof RapidAgentLiveUri)) {
                    jniPrtListener.onEvent(currChannelIndex, RapidAgentSDK.EVENT_CREATE_CHANNEL_FAIL, "", "current is vod work mode");
                    return -1;
                }
            } else {
                if (rapidAgentUri instanceof RapidAgentLiveUri) {
                    jniPrtListener.onEvent(currChannelIndex, RapidAgentSDK.EVENT_CREATE_CHANNEL_FAIL, "", "current is live work mode");
                    return -1;
                }
            }
        }

        if (rapidAgentUri instanceof RapidAgentCatchupUri) {
            RapidAgentCatchupUri catchupUri = (RapidAgentCatchupUri) rapidAgentUri;
            Pair<Integer, Integer> typeAndBrand = getCatchupSolutionAndBrand(drmType, catchupUri.isTimeShift());
            solutionType = typeAndBrand.first;
            solutionBrand = typeAndBrand.second;
            userToken = generatePlayToken(currentStreamId, solutionType, solutionBrand, RapidAgentConstant.getAccountToken());
            playToken = generatePlayToken(currentStreamId, solutionType, solutionBrand, rapidAgentUri.getStreamToken());
            if (catchupUri.isTimeShift()) {
                setParams(PLAY_TYPE_TIME_SHIFT, userToken, true);
            } else {
                setParams(PLAY_TYPE_CATCHUP, userToken, true);
            }
            currChannelIndex = startupCatchup(catchupUri, playToken, solutionType, solutionBrand);
            channelType = StreamInfoMgr.CATCHUP;
            useBlockIdMap = 1;
        } else if (rapidAgentUri instanceof RapidAgentVodUri) {
            Pair<Integer, Integer> typeAndBrand = getVodSolutionAndBrand(drmType, protocol);
            solutionType = typeAndBrand.first;
            solutionBrand = typeAndBrand.second;
            userToken = generatePlayToken(currentStreamId, solutionType, solutionBrand, RapidAgentConstant.getAccountToken());
            playToken = generatePlayToken(currentStreamId, solutionType, solutionBrand, rapidAgentUri.getStreamToken());
            setParams(PLAY_TYPE_VOD, userToken, true);
            currChannelIndex = startupVod((RapidAgentVodUri) rapidAgentUri, playToken, solutionType, solutionBrand);
            channelType = StreamInfoMgr.VOD;
            useBlockIdMap = 1;
            blockSize = 512 * 1024;
            if (rapidAgentUri.getFrontInfo() != null) {
                fileSize = parseLong(rapidAgentUri.getFrontInfo().get("trfsize"), 0);
            }
        } else if (rapidAgentUri instanceof RapidAgentLiveUri) {
            Pair<Integer, Integer> typeAndBrand = getLiveSolutionAndBrand(drmType);
            solutionType = typeAndBrand.first;
            solutionBrand = typeAndBrand.second;
            userToken = generatePlayToken(currentStreamId, solutionType, solutionBrand, RapidAgentConstant.getAccountToken());
            playToken = generatePlayToken(currentStreamId, solutionType, solutionBrand, rapidAgentUri.getStreamToken());
            setParams(PLAY_TYPE_LIVE, userToken, true);
            currChannelIndex = startupLive((RapidAgentLiveUri) rapidAgentUri, playToken, solutionType, solutionBrand);
            channelType = rapidAgentUri.isMicroBlock() ? getMicroBlockChannelType(rapidAgentUri) : StreamInfoMgr.LIVE;
            useBlockIdMap = 0;
        } else {
            throw new IOException("unknown RapidAgentUri instance: " + rapidAgentUri.getClass().getCanonicalName());
        }

        jniPrtListener.setChannelId(currChannelIndex);
        if (currChannelIndex > 0) {
            int isAbr = rapidAgentUri.isAdaptiveBitrate() ? 1 : 0;
            int proxyPushMode = rapidAgentUri.isAdaptiveBitrate() ? 0 : 1;
            int isMicroBlock = rapidAgentUri.isMicroBlock() ? 1 : 0;
            int externalDecrypt = rapidAgentUri.isNeedExternalDecrypt() ? 1 : 0;
            int flags = 0;
            flags |= externalDecrypt;
            JniApiImpl.getInstance().initEngineProxyBuffer(currChannelIndex, blockSize, fileSize,
                    solutionType, solutionBrand, drmLicenseUrl, vnoKey, useBlockIdMap, proxyPushMode, isAbr, isMicroBlock, flags);
            this.jniPrtListener = jniPrtListener;
            PrtListenerManager.getInstance().registerListener(currChannelIndex, new PrtListenerManager.PrtManagerCallBack() {
                public int getRequestId() {
                    return DEFAULT_REQUEST_ID;
                }

                public int getChannelId() {
                    return currChannelIndex;
                }
            }, jniPrtListener);
            jniPrtListener.setProxyUrl(getProxyUrl());
            sGlobalInstanceCount++;
        } else {
            jniPrtListener.onEvent(currChannelIndex, RapidAgentSDK.EVENT_CREATE_CHANNEL_FAIL, "", "create channel fail");
        }
        StreamInfoMgr.add(currChannelIndex, channelType, currentStreamId, startupTime);
        LogUtil.i("PrtEngine", "startup new engine " + this + ", current size is " + sGlobalInstanceCount);
        return currChannelIndex;
    }

    public void updateTokens(String newUserToken, String newStreamToken) throws IOException {
        RapidAgentConstant.setAccountToken(newUserToken);
        if (currentPlayType == -1) {
            return;
        }
        if (!TextUtils.equals(newStreamToken, rapidAgentUri.getStreamToken())) {
            rapidAgentUri.setStreamToken(newStreamToken);
        }
        String userToken = generatePlayToken(currentStreamId, solutionType, solutionBrand, newUserToken);
        String playToken = generatePlayToken(currentStreamId, solutionType, solutionBrand, newStreamToken);
        setParams(currentPlayType, userToken, true);
        JniApiImpl.getInstance().updateChannel(currChannelIndex, "ptkn=" + playToken);
    }

    private void setParams(int newPlayType, String userToken, boolean forceUpdate) throws IOException {
        if (currentPlayType == newPlayType && !forceUpdate) {
            return;
        }
        int reconnectTrack = 0;
        if ((sGlobalPlayType == PLAY_TYPE_CATCHUP && newPlayType == PLAY_TYPE_TIME_SHIFT) ||
                (sGlobalPlayType == PLAY_TYPE_TIME_SHIFT && newPlayType == PLAY_TYPE_CATCHUP)) {
            reconnectTrack = 1;
        }
        currentPlayType = newPlayType;
        sGlobalPlayType = newPlayType;
        if (currentPlayType == PLAY_TYPE_LIVE) {
            String tracker = PrtConfig.getLiveServerAddress().getTracker();
            Map<String, Object> initParams = PrtConfig.getInitLiveOptions(tracker, userToken);
            String liveInitParams = paramsMap2String(initParams, JniApiImpl.seperator_and);
            LogUtil.sensitive("setParams for live : " + liveInitParams);
            JniApiImpl.getInstance().setParam(liveInitParams, null);
        } else if (currentPlayType == PLAY_TYPE_CATCHUP || currentPlayType == PLAY_TYPE_TIME_SHIFT) {
            String tracker = PrtConfig.getCatchupServerAddress().getTracker();
            Map<String, Object> initParams = PrtConfig.getInitCatchupOptions(tracker, userToken);
            if (reconnectTrack == 1) {
                initParams.put("rsth", reconnectTrack);
            }
            mergeParamsMap(initParams, rapidAgentUri != null ? rapidAgentUri.getCustomInfo() : null);
            String catchupInitParams = "wkm=2&vodInit=" + paramsMap2String(initParams, "#");
            LogUtil.sensitive("setParams for catchup : " + catchupInitParams);
            JniApiImpl.getInstance().setParam(catchupInitParams, null);
        } else if (currentPlayType == PLAY_TYPE_VOD) {
            String tracker = PrtConfig.getVodServerAddress().getTracker();
            Map<String, Object> initParams = PrtConfig.getInitVodOptions(tracker, userToken);
            mergeParamsMap(initParams, rapidAgentUri != null ? rapidAgentUri.getCustomInfo() : null);
            String vodInitParams = "wkm=2&vodInit=" + paramsMap2String(initParams, "#");
            LogUtil.sensitive("setParams for vod : " + vodInitParams);
            JniApiImpl.getInstance().setParam(vodInitParams, null);
        }
    }

    public static Pair<Integer, Integer> getLiveSolutionAndBrand(int drmType) {
        int type;
        int brand;
        if (drmType == RapidAgentUri.DRM_TYPE_WIDEVINE) {
            type = SOLUTION_TYPE_LIVE_DRM_DASH;
            brand = SOLUTION_BRAND_IRDETO;
        } else if (drmType == RapidAgentUri.DRM_TYPE_GOOSE) {
            type = SOLUTION_TYPE_LIVE_DRM_HLS;
            brand = SOLUTION_BRAND_GOOSE;
        } else {
            int prtVersionCode = getVersionCode();
            type = prtVersionCode >= 21000 ? SOLUTION_TYPE_LIVE_PLAIN_RPD : SOLUTION_TYPE_LIVE_VALO_HLS;
            brand = SOLUTION_BRAND_NONE;
        }
        return new Pair<>(type, brand);
    }

    public static Pair<Integer, Integer> getCatchupSolutionAndBrand(int drmType, boolean isTimeShift) {
        int type;
        int brand;
        if (drmType == RapidAgentUri.DRM_TYPE_WIDEVINE) {
            type = isTimeShift ? SOLUTION_TYPE_VOD_TIME_SHIFT_DRM_DASH : SOLUTION_TYPE_VOD_DRM_DASH;
            brand = SOLUTION_BRAND_IRDETO;
        } else if (drmType == RapidAgentUri.DRM_TYPE_GOOSE) {
            type = isTimeShift ? SOLUTION_TYPE_VOD_TIME_SHIFT_DRM_HLS : SOLUTION_TYPE_VOD_DRM_HLS;
            brand = SOLUTION_BRAND_GOOSE;
        } else {
            type = isTimeShift ? SOLUTION_TYPE_VOD_TIME_SHIFT_PLAIN_RPD : SOLUTION_TYPE_VOD_PLAIN_RPD;
            brand = SOLUTION_BRAND_NONE;
        }
        return new Pair<>(type, brand);
    }

    public static Pair<Integer, Integer> getVodSolutionAndBrand(int drmType, int protocol) {
        int type;
        int brand;
        if (drmType == RapidAgentUri.DRM_TYPE_WIDEVINE) {
            type = SOLUTION_TYPE_VOD_DRM_DASH;
            brand = SOLUTION_BRAND_IRDETO;
        } else if (drmType == RapidAgentUri.DRM_TYPE_GOOSE) {
            type = SOLUTION_TYPE_VOD_DRM_HLS;
            brand = SOLUTION_BRAND_GOOSE;
        } else if (protocol == RapidAgentUri.PROTOCOL_MPT) {
            type = SOLUTION_TYPE_VOD_MPT;
            brand = SOLUTION_BRAND_NONE;
        } else if (protocol == RapidAgentUri.PROTOCOL_RPD) {
            type = SOLUTION_TYPE_VOD_PLAIN_RPD;
            brand = SOLUTION_BRAND_NONE;
        } else {
            type = SOLUTION_TYPE_VOD_MPQ;
            brand = SOLUTION_BRAND_NONE;
        }
        return new Pair<>(type, brand);
    }

    private int startupLive(RapidAgentLiveUri prtUri, String playToken, int solutionType, int solutionBrand) throws IOException {
        String channelId = prtUri.getStreamId();
        String channelName = prtUri.getStreamName();
        ServerAddress address = PrtConfig.getLiveServerAddress();
        String tracker = address.getTracker();
        String defaultPrt = address.getDefaultPrt();
        String turboPrt = address.getTurboPrt();
        Map<String, Object> sysParamsMap = PrtConfig.getStartLiveSysOptions();
        Map<String, Object> channelParamsMap = new LinkedHashMap<>();

        if (TextUtils.isEmpty(tracker) && TextUtils.isEmpty(defaultPrt)) {
            LogUtil.e(TAG, "startupLive error : there is not a valid live tracker or default prt");
            return -1;
        }

        if (prtUri.isAdaptiveBitrate()) {
            long bandwidth = SPUtil.getMBBandwidth(0);
            if (bandwidth > 0) {
                channelParamsMap.put("mbbw", bandwidth);
            }
        }
        if (prtUri.isMicroBlock()) {
            int idealPackets = SPUtil.getMBIdealPacket(prtUri.getMicroBlockLevel(), 0);
            switch (rapidAgentUri.getMicroBlockLevel()) {
                case RapidAgentUri.MICRO_BLOCK_ULTRA_LOW:
                    channelParamsMap.put("fdbo", idealPackets > 30 ? Math.min(idealPackets / 30, 3) : 1);
                    break;
                case RapidAgentUri.MICRO_BLOCK_LOW:
                    channelParamsMap.put("fdbo", idealPackets > 30 ? Math.min(idealPackets / 30, 3) : 2);
                    break;
                case RapidAgentUri.MICRO_BLOCK_NORMAL:
                    channelParamsMap.put("fdbo", 3);
                    break;
                case RapidAgentUri.MICRO_BLOCK_GOP3:
                    channelParamsMap.put("fdbo", idealPackets > 90 ? Math.min(idealPackets / 90, 3) : 2);
                    break;
            }
        }
        Map<String, Object> dflChannelParams = PrtConfig.getStartLiveChannelOptions(
                playToken, tracker, defaultPrt, turboPrt, channelId, channelName, solutionType, solutionBrand);
        channelParamsMap.putAll(dflChannelParams);
        channelParamsMap.putAll(prtUri.getCustomInfo());
        if (prtUri.isAdaptiveBitrate()) {
            channelParamsMap.put("abr", 1);
        }
        if (prtUri.isMicroBlock()) {
            channelParamsMap.put("lld", 1);
            switch (prtUri.getMicroBlockLevel()) {
                case RapidAgentUri.MICRO_BLOCK_ULTRA_LOW:
                    channelParamsMap.put("nnsr", 1);
                    channelParamsMap.put("mbgi", 33);
                    break;
                case RapidAgentUri.MICRO_BLOCK_LOW:
                    channelParamsMap.put("nnsr", 0);
                    channelParamsMap.put("mbgi", 1000);
                    break;
                case RapidAgentUri.MICRO_BLOCK_NORMAL:
                    channelParamsMap.put("nnsr", 0);
                    channelParamsMap.put("mbgi", 0);
                    break;
                case RapidAgentUri.MICRO_BLOCK_GOP3:
                    channelParamsMap.put("nnsr", 0);
                    channelParamsMap.put("mbgi", 3000);
                    break;
            }
        }
        String sysParamsString = paramsMap2String(sysParamsMap);
        String channelParamsString = paramsMap2String(channelParamsMap);
        LogUtil.sensitive("startupLive sysParamsString:" + sysParamsString);
        LogUtil.sensitive("startupLive channelParamsString:" + channelParamsString);

        return JniApiImpl.getInstance().startChannel(sysParamsString, channelParamsString, null);
    }

    private int startupCatchup(RapidAgentCatchupUri prtUri, String playToken, int solutionType, int solutionBrand) throws IOException {
        String channelId = prtUri.getStreamId();
        String epgStart = prtUri.getEpgStart();
        String epgDuration = prtUri.getEpgDuration();
        ServerAddress address = PrtConfig.getCatchupServerAddress();
        String tracker = address.getTracker();
        String defaultPrt = address.getDefaultPrt();
        Map<String, Object> customInfo = prtUri.getCustomInfo();
        String channelParamsString;
        Map<String, Object> channelParamsMap;

        if (TextUtils.isEmpty(tracker) && TextUtils.isEmpty(defaultPrt)) {
            LogUtil.e(TAG, "startupCatchup error: there is not a valid catchup tracker or default pm");
            return -1;
        }

        if (prtUri.getFrontInfo() == null) {
            channelParamsMap = PrtConfig.getRequestCatchupInfoOptions(playToken, channelId, channelId, parseLong(epgStart, 0),
                    parseLong(epgDuration, 0), tracker, defaultPrt, solutionType, solutionBrand);
            channelParamsMap.putAll(customInfo);
        } else {
            channelParamsMap = PrtConfig.getStartCatchupChannelOptions(playToken, tracker, solutionType, solutionBrand);
            mergeParamsMap(channelParamsMap, customInfo);
            channelParamsMap.putAll(prtUri.getFrontInfo());
        }
        channelParamsString = paramsMap2String(channelParamsMap);
        LogUtil.sensitive("startupCatchup channelParamsString:" + channelParamsString);

        int result = JniApiImpl.getInstance().startVodChannel(channelParamsString, null);
        if (result > 0 && prtUri.getFrontInfo() != null && prtUri.isTimeShift()) {
            JniApiImpl.getInstance().requestTimeShiftData(result, prtUri.getTimeShiftDelay());
        }
        return result;
    }

    private int startupVod(RapidAgentVodUri prtUri, String playToken, int solutionType, int solutionBrand) throws IOException {
        String channelId = prtUri.getStreamId();
        ServerAddress address = PrtConfig.getVodServerAddress();
        String tracker = address.getTracker();
        String defaultPrt = address.getDefaultPrt();
        Map<String, Object> customInfo = prtUri.getCustomInfo();
        String channelParamsString;
        Map<String, Object> channelParamsMap;

        if (TextUtils.isEmpty(tracker) && TextUtils.isEmpty(defaultPrt)) {
            LogUtil.e(TAG, "startupVod error : there is not a valid vod tracker or default pm");
            return -1;
        }

        if (prtUri.getFrontInfo() == null) {
            channelParamsMap = PrtConfig.getRequestVodInfoOptions(playToken, channelId, tracker, defaultPrt, solutionType, solutionBrand);
            channelParamsMap.putAll(customInfo);
        } else {
            channelParamsMap = PrtConfig.getStartVodChannelOptions(playToken, tracker, solutionType, solutionBrand);
            mergeParamsMap(channelParamsMap, customInfo);
            channelParamsMap.putAll(prtUri.getFrontInfo());
        }
        channelParamsMap.put("lsid", prtUri.getStreamId());
        channelParamsString = paramsMap2String(channelParamsMap);
        LogUtil.sensitive("startupVod channelParamsString:" + channelParamsString);

        return JniApiImpl.getInstance().startVodChannel(channelParamsString, null);
    }

    public void shutdown() {
        if (currChannelIndex <= 0) {
            return;
        }
        JniApiImpl.getInstance().closeEngineProxyBuffer(currChannelIndex);
        JniApiImpl.getInstance().stopChannel(currChannelIndex);
        PrtListenerManager.getInstance().unRegisterListener(currChannelIndex, DEFAULT_REQUEST_ID);
        jniPrtListener.release();
        jniPrtListener.removeAllPlayListener();
        sGlobalInstanceCount--;
        LogUtil.i("PrtEngine", "shutdown engine " + this + ", current size is " + sGlobalInstanceCount);
    }

    public void requestTimeShiftData(long delayTime) {
        delayTime = Math.max(delayTime, CommonConfig.timeShiftMinDelay());
        JniApiImpl.getInstance().requestTimeShiftData(currChannelIndex, delayTime);
    }

    public void addListener(RapidAgentSDK.OnPlayListener listener) {
        jniPrtListener.addPlayListener(listener);
    }

    public void removeListener(RapidAgentSDK.OnPlayListener listener) {
        jniPrtListener.removePlayListener(listener);
    }

    private String getProxyUrl() {
        if (currChannelIndex <= 0) {
            return null;
        }
        if (rapidAgentUri.isMicroBlock()) {
            return String.format("mls://micro.segment.live.streaming/%s.mmd?channel=%s", rapidAgentUri.getStreamId(), currChannelIndex);
        }
        int port = JniApiImpl.getInstance().setAndGetPort(0);
        String urlSuffix = getProxyUrlSuffix(currentStreamId, solutionType, solutionBrand, rapidAgentUri.isAdaptiveBitrate());
        return String.format("http://127.0.0.1:%s/%s/%s", port, currChannelIndex, urlSuffix);
    }

    public static String getProxyUrlSuffix(String currentStreamId, int solutionType, int solutionBrand, boolean isAbr) {
        String fileName = TextUtils.isEmpty(currentStreamId) ? "stream" : currentStreamId;
        switch (solutionType) {
            case SOLUTION_TYPE_NONE:
            case SOLUTION_TYPE_LIVE_VALO_HLS:
                return fileName + "-1_0.ts";
            case SOLUTION_TYPE_VOD_MPT:
                return "master.mp4";
            case SOLUTION_TYPE_LIVE_DRM_HLS:
            case SOLUTION_TYPE_VOD_MPQ:
            case SOLUTION_TYPE_VOD_PLAIN_RPD:
            case SOLUTION_TYPE_VOD_DRM_HLS:
            case SOLUTION_TYPE_VOD_TIME_SHIFT_PLAIN_RPD:
            case SOLUTION_TYPE_VOD_TIME_SHIFT_DRM_HLS:
                return "master.m3u8";
            case SOLUTION_TYPE_LIVE_DRM_DASH:
            case SOLUTION_TYPE_VOD_DRM_DASH:
            case SOLUTION_TYPE_VOD_TIME_SHIFT_DRM_DASH:
                return "index.mpd";
            case SOLUTION_TYPE_LIVE_PLAIN_RPD:
                return isAbr ? "master.m3u8" : fileName + "-1_0.ts";
        }

        throw new IllegalStateException("invalid solutionType " + solutionType + " and solutionBrand " + solutionBrand);
    }

    public static String getProxyUrlSuffix(RapidAgentUri uri) {
        if (uri.isMicroBlock()) {
            return ".mmd";
        }
        Pair<Integer, Integer> typeAndBrand;
        if (uri instanceof RapidAgentLiveUri) {
            typeAndBrand = getLiveSolutionAndBrand(uri.getDrmType());
        } else if (uri instanceof RapidAgentCatchupUri) {
            typeAndBrand = getCatchupSolutionAndBrand(uri.getDrmType(), ((RapidAgentCatchupUri) uri).isTimeShift());
        } else if (uri instanceof RapidAgentVodUri) {
            typeAndBrand = getVodSolutionAndBrand(uri.getDrmType(), uri.getProtocol());
        } else {
            typeAndBrand = Pair.create(SOLUTION_TYPE_NONE, SOLUTION_BRAND_NONE);
        }
        return getProxyUrlSuffix(null, typeAndBrand.first, typeAndBrand.second, uri.isAdaptiveBitrate());
    }

    public int getChannelId() {
        return currChannelIndex;
    }

    static String getVersionName() {
        return JniApiImpl.getInstance().getVersion();
    }

    static int getVersionCode() {
        return JniApiImpl.getInstance().getDigitalVersion();
    }

    public static String encryptToBase64(String key, byte[] data) {
        return AESSecurityPrt.encryptToBase64(key, data);
    }

    public static String decryptFromBase64(String key, String data) {
        return AESSecurityPrt.decryptFromBase64(key, data);
    }

    static String translate(String old) {
        byte[] oldBytes = old.getBytes(StandardCharsets.UTF_8);
        int length = oldBytes.length;
        byte[] newBytes = new byte[length];
        for (int i = 0; i < length; i++) {
            if (oldBytes[i] >= 65 && oldBytes[i] <= 74) {
                oldBytes[i] -= 17;
            } else if (oldBytes[i] >= 48 && oldBytes[i] <= 57) {
                oldBytes[i] += 17;
            }
            newBytes[length - 1 - i] = oldBytes[i];
        }
        return new String(newBytes, StandardCharsets.UTF_8);
    }

    static void setTokenSecretKey(String tokenSecretKey) {
        sTokenGenerator.setTokenSecretKey(translate(tokenSecretKey));
    }

    private String generatePlayToken(String channelId, int solutionType, int solutionBrand, String token) throws IOException {
        return sTokenGenerator.generatePlayToken(channelId, solutionType, solutionBrand, rapidAgentUri, token);
    }

    private static void mergeParamsMap(@NonNull Map<String, Object> paramsMap, @Nullable Map<String, Object> customInfo) {
        if (customInfo != null && !customInfo.isEmpty()) {
            for (Map.Entry<String, Object> entry : paramsMap.entrySet()) {
                if (customInfo.containsKey(entry.getKey())) {
                    paramsMap.put(entry.getKey(), customInfo.get(entry.getKey()));
                }
            }
        }
    }

    private static String paramsMap2String(Map<String, Object> paramsMap) throws IOException {
        return paramsMap2String(paramsMap, JniApiImpl.seperator_and);
    }

    private static String paramsMap2String(Map<String, Object> paramsMap, String splitFlag) throws IOException {
        StringBuilder stringBuilder = new StringBuilder();
        boolean isFirstItem = true;
        for (Map.Entry<String, Object> entry : paramsMap.entrySet()) {
            if (!isFirstItem) {
                stringBuilder.append(splitFlag);
            }
            isFirstItem = false;
            String key = entry.getKey();
            Object value = entry.getValue();
            if (key == null) {
                throw new IOException("paramsMap2String error invalid key");
            }
            if (value == null) {
                throw new IOException("paramsMap2String error invalid value for " + key);
            }
            stringBuilder.append(key).append("=").append(value);
        }
        return stringBuilder.toString();
    }

    private static int getMicroBlockChannelType(@NonNull RapidAgentUri rapidAgentUri) {
        switch (rapidAgentUri.getMicroBlockLevel()) {
            case RapidAgentUri.MICRO_BLOCK_ULTRA_LOW:
                return StreamInfoMgr.MB_ULTRA_LOW;
            case RapidAgentUri.MICRO_BLOCK_LOW:
                return StreamInfoMgr.MB_LOW;
            case RapidAgentUri.MICRO_BLOCK_NORMAL:
                return StreamInfoMgr.MB_NORMAL;
            case RapidAgentUri.MICRO_BLOCK_GOP3:
            default:
                return StreamInfoMgr.MB_GOP3;
        }
    }
}
