package com.bitkernel.stream.rapid.player;

import android.content.Context;
import android.media.MediaCodecInfo;
import android.media.MediaPlayer;
import android.os.Handler;
import android.os.Looper;
import android.os.Parcel;
import android.util.Log;
import android.util.Pair;
import android.view.Surface;
import android.view.SurfaceHolder;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.bitkernel.stream.rapid.player.utils.CommonUtil;
import com.bitkernel.stream.rapid.player.utils.DatabaseHelper;
import com.bitkernel.stream.rapid.player.utils.MediaCodecUtil;

import java.io.IOException;
import java.lang.ref.WeakReference;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;

import tv.danmaku.ijk.media.player.AndroidMediaPlayer;
import tv.danmaku.ijk.media.player.IMediaPlayer;
import tv.danmaku.ijk.media.player.IjkMediaMeta;
import tv.danmaku.ijk.media.player.IjkMediaPlayer;
import tv.danmaku.ijk.media.player.IjkTimedText;
import tv.danmaku.ijk.media.player.misc.ITrackInfo;

public class RapidMediaPlayer extends AbsRapidMediaPlayer {
    private static final String TAG = "RapidMediaPlayer";

    private static final Set<Integer> SINGLE_INFO_EVENT_SET = new HashSet<>();

    static {
        SINGLE_INFO_EVENT_SET.add(IMediaPlayer.MEDIA_INFO_OPEN_INPUT);
        SINGLE_INFO_EVENT_SET.add(IMediaPlayer.MEDIA_INFO_READ_FIRST_VIDEO_FRAME);
        SINGLE_INFO_EVENT_SET.add(IMediaPlayer.MEDIA_INFO_READ_FIRST_AUDIO_FRAME);
        SINGLE_INFO_EVENT_SET.add(IMediaPlayer.MEDIA_INFO_FIND_STREAM_INFO);
        SINGLE_INFO_EVENT_SET.add(IMediaPlayer.MEDIA_INFO_COMPONENT_OPEN);
        SINGLE_INFO_EVENT_SET.add(IMediaPlayer.MEDIA_INFO_AUDIO_DECODED_START);
        SINGLE_INFO_EVENT_SET.add(IMediaPlayer.MEDIA_INFO_VIDEO_DECODED_START);
        SINGLE_INFO_EVENT_SET.add(IMediaPlayer.MEDIA_INFO_DRM_KEY_LOADED);
        SINGLE_INFO_EVENT_SET.add(IMediaPlayer.MEDIA_INFO_AUDIO_RENDERING_START);
        SINGLE_INFO_EVENT_SET.add(IMediaPlayer.MEDIA_INFO_VIDEO_RENDERING_START);
        SINGLE_INFO_EVENT_SET.add(IMediaPlayer.MEDIA_INFO_STREAM_FORMAT_SUMMARY);
        SINGLE_INFO_EVENT_SET.add(IMediaPlayer.MEDIA_INFO_STARTUP_INFO);
    }

    protected final Handler appHandler;
    protected final boolean isMicroBlock;
    private final AtomicBoolean startWhenInitPrepared = new AtomicBoolean();
    private final AtomicBoolean startWhenRebuildPrepared = new AtomicBoolean();
    private final WeakReference<Context> contextRef;
    private final Set<Integer> notifiedEventList = new HashSet<>();
    private boolean enableRebuildFeature = !CommonUtil.isVirtualMachine();
    private final Map<Pair<Integer, Integer>, Integer> customRebuildEvent = new HashMap<>();
    private long startupTimeoutMs = 0;

    private Surface surface;

    private int drmType = DRM_TYPE_NULL;
    private boolean drmMultiSession;
    private String drmLicenceServerUrl;
    private Map<String, String> drmHeaders;
    private String drmReqMethod;
    private byte[] drmOfflineLicenseKeySetId;
    private String offlineLicenseDrmInitInfo;

    public RapidMediaPlayer(@Nullable Context context, int playerType, Map<String, String> params) {
        this(context, playerType, params, new DefaultPlayerStatistic(), new DefaultPlayerBuilder());
    }

    public RapidMediaPlayer(@Nullable Context context, int playerType, Map<String, String> params,
                            PlayerStatistic playerStatistic, PlayerBuilder playerBuilder) {
        super(playerStatistic, playerBuilder);
        Log.i(TAG, "start init sdk version " + BuildConfig.GIT_HASH);
        Looper looper;
        if ((looper = Looper.myLooper()) != null) {
            appHandler = new Handler(looper);
        } else if ((looper = Looper.getMainLooper()) != null) {
            appHandler = new Handler(looper);
        } else {
            appHandler = null;
        }
        isMicroBlock = params.containsKey(PlayerBuilder.MLS);
        if (playerType == PLAYER_TYPE_UNKNOWN) {
            setInternalPlayer(playerBuilder.createMediaPlayer(context, params));
        } else {
            setInternalPlayer(playerBuilder.createMediaPlayer(context, playerType));
        }
        contextRef = new WeakReference<>(context);
        if (getPlayerType() == PLAYER_TYPE_AMP) {
            startWhenInitPrepared.set(true);
        }
    }

    public static void preload(Context context, int h264Rank, int h265Rank) {
        DatabaseHelper.init(context);
        DefaultPlayerBuilder.preload(h264Rank, h265Rank);
    }

    public static void setLogLevel(int level) {
        IjkMediaPlayer.native_setLogLevel(level);
    }

    public void enableRebuildFeature(boolean enabled) {
        enableRebuildFeature = enabled;
        Log.i(TAG, "enableRebuildFeature " + enableRebuildFeature);
    }

    public void setCustomRebuildEvent(@NonNull Map<Pair<Integer, Integer>, Integer> events) {
        customRebuildEvent.clear();
        customRebuildEvent.putAll(events);
        Log.i(TAG, "setCustomRebuildEvent " + customRebuildEvent);
    }

    public void setStartupTimeoutMs(long startupTimeoutMs) {
        this.startupTimeoutMs = startupTimeoutMs;
    }

    protected boolean isRebuilding() {
        return rebuilding.get();
    }

    protected void rebuildInternalPlayer(int playerType, int what, int extra) {
        rebuildInternalPlayer(playerType, null, what, extra);
    }

    protected void rebuildInternalPlayer(int playerType, String newStreamUrl, int what, int extra) {
        Log.i(TAG, "prepare to rebuildInternalPlayer");
        if (isMicroBlock && playerType != PLAYER_TYPE_IJK) {
            Log.e(TAG, "cant rebuild a mls to a none-ijkplayer");
            super.onError(this, what, extra);
            return;
        }
        if (rebuilding.get()) {
            Log.w(TAG, "rebuilding now and ignore this request");
            return;
        }
        internalHashCode = 0;
        rebuildWhat = what;
        rebuildExtra = extra;
        appHandler.post(() -> {
            if (!rebuilding.get()) {
                rebuildInternalPlayerLock(playerType, newStreamUrl);
            }
        });
    }

    private void rebuildInternalPlayerLock(int playerType, String newStreamUrl) {
        Log.i(TAG, "start to rebuildInternalPlayerLock");
        rebuilding.set(true);
        if (outsideInfoListener != null) {
            outsideInfoListener.onInfo(this, MEDIA_INFO_BUFFERING_START, 0, String.valueOf(FFP_BUFFERING_REASON_START_REBUILD));
        }
        synchronized (rebuilding) {
            IMediaPlayer oldPlayer = getInternalPlayer();
            String path = newStreamUrl != null ? newStreamUrl : oldPlayer.getDataSource();
            oldPlayer.stop();
            oldPlayer.release();
            Pair<IMediaPlayer, Integer> pair = playerBuilder.createMediaPlayer(contextRef.get(), playerType);
            setInternalPlayer(pair);
            IMediaPlayer newPlayer = pair.first;
            if (drmType != DRM_TYPE_NULL) {
                newPlayer.setDrmInfo(drmType, drmMultiSession, drmLicenceServerUrl, drmHeaders, drmReqMethod, drmOfflineLicenseKeySetId, offlineLicenseDrmInitInfo);
            }
            newPlayer.setSurface(surface);
            try {
                newPlayer.setDataSource(path);
            } catch (IOException e) {
                e.printStackTrace();
            }
            startWhenRebuildPrepared.set(true);
            newPlayer.prepareAsync();
        }
        rebuilding.set(false);
        Log.i(TAG, "finish to rebuildInternalPlayerLock");
    }

    private final Runnable prepareTimeoutWork = () -> {
        if (rebuildWhat != 0 || rebuildExtra != 0) {
            RapidMediaPlayer.super.onError(getInternalPlayer(), rebuildWhat, rebuildExtra);
            rebuildWhat = 0;
            rebuildExtra = 0;
        } else {
            RapidMediaPlayer.super.onError(getInternalPlayer(), RAPID_ERROR_CATEGORY, RAPID_ERROR_PREPARE_TIMEOUT);
        }
    };

    @Override
    public void setDisplay(SurfaceHolder sh) {
        setSurface(sh != null ? sh.getSurface() : null);
    }

    @Override
    public void setSurface(Surface surface) {
        super.setSurface(surface);
        this.surface = surface;
    }

    @Override
    public void prepareAsync() throws IllegalStateException {
        super.prepareAsync();
        if (startupTimeoutMs > 0) {
            appHandler.postDelayed(prepareTimeoutWork, startupTimeoutMs);
        }
    }

    @Override
    public void release() {
        appHandler.removeCallbacksAndMessages(null);
        super.release();
    }

    @Override
    public void setDrmInfo(int drmType, boolean multiSession, String licenceServerUrl, Map<String, String> headers, String reqMethod, byte[] offlineLicenseKeySetId, String offlineLicenseDrmInitInfo) {
        this.drmType = drmType;
        this.drmMultiSession = multiSession;
        this.drmLicenceServerUrl = licenceServerUrl;
        this.drmHeaders = headers != null ? new HashMap<>(headers) : new HashMap<>();
        this.drmReqMethod = reqMethod;
        this.drmOfflineLicenseKeySetId = offlineLicenseKeySetId;
        this.offlineLicenseDrmInitInfo = offlineLicenseDrmInitInfo;
        super.setDrmInfo(this.drmType, this.drmMultiSession, this.drmLicenceServerUrl, this.drmHeaders, this.drmReqMethod, this.drmOfflineLicenseKeySetId, this.offlineLicenseDrmInitInfo);
    }

    @Override
    protected void onPrepared(IMediaPlayer mp) {
        if (mp.hashCode() != internalHashCode) {
            Log.i(TAG, "ignore prepared msg for " + mp);
            return;
        }
        if (startWhenRebuildPrepared.compareAndSet(true, false)) {
            start();
            return;
        }
        if (startWhenInitPrepared.compareAndSet(true, false)) {
            start();
        }
        super.onPrepared(mp);
    }

    @Override
    protected void onCompletion(IMediaPlayer mp) {
        if (mp.hashCode() != internalHashCode) {
            Log.i(TAG, "ignore completed msg for " + mp);
            return;
        }
        super.onCompletion(mp);
    }

    @Override
    protected void onBufferingUpdate(IMediaPlayer mp, int percent) {
        if (mp.hashCode() != internalHashCode) {
            Log.i(TAG, "ignore buffering update msg for " + mp);
            return;
        }
        super.onBufferingUpdate(mp, percent);
    }

    @Override
    protected void onSeekComplete(IMediaPlayer mp) {
        if (mp.hashCode() != internalHashCode) {
            Log.i(TAG, "ignore seek completed msg for " + mp);
            return;
        }
        super.onSeekComplete(mp);
    }

    @Override
    protected void onVideoSizeChanged(IMediaPlayer mp, int width, int height, int sar_num, int sar_den) {
        if (mp.hashCode() != internalHashCode) {
            Log.i(TAG, "ignore video size changed msg for " + mp);
            return;
        }
        super.onVideoSizeChanged(mp, width, height, sar_num, sar_den);
    }

    @Override
    protected boolean onInfo(IMediaPlayer mp, int what, int extra, String info) {
        if (mp.hashCode() != internalHashCode) {
            Log.i(TAG, "ignore info msg " + what + "," + extra + " for " + mp);
            return true;
        }
        if (what == MEDIA_INFO_AUDIO_RENDERING_START) {
            int videoTrack = mp.getSelectedTrack(ITrackInfo.MEDIA_TRACK_TYPE_VIDEO);
            if (videoTrack < 0) {
                extra = 1;
                Log.w("AbsRapidMediaPlayer", "it is an audio only stream");
            }
        }
        if (what == MEDIA_INFO_VIDEO_RENDERING_START || (what == MEDIA_INFO_AUDIO_RENDERING_START && extra == 1)) {
            appHandler.removeCallbacks(prepareTimeoutWork);
        }
        if (SINGLE_INFO_EVENT_SET.contains(what)) {
            if (notifiedEventList.contains(what)) {
                Log.w(TAG, "ignore info event what:" + what);
                if (what == MEDIA_INFO_VIDEO_RENDERING_START || (what == MEDIA_INFO_AUDIO_RENDERING_START && extra == 1)) {
                    if (outsideInfoListener != null) {
                        outsideInfoListener.onInfo(this, MEDIA_INFO_BUFFERING_END, 0, String.valueOf(FFP_BUFFERING_REASON_END_REBUILD));
                    }
                }
                return true;
            } else {
                notifiedEventList.add(what);
            }
        }
        if (what == IMediaPlayer.MEDIA_INFO_VIDEO_RENDERING_START) {
            if (mp instanceof IjkMediaPlayer) {
                int codecType = ((IjkMediaPlayer) mp).getVideoDecoder();
                if (codecType == IjkMediaPlayer.FFP_PROPV_DECODER_MEDIACODEC) {
                    savePlayRecord(true);
                }
            } else if ("tv.danmaku.ijk.media.exo.IjkExoMediaPlayer".equals(mp.getClass().getCanonicalName())) {
                MediaCodecInfo mediaCodecInfo = MediaCodecUtil.findMediaCodecInfo(mp.getVideoCodecName());
                savePlayRecord(MediaCodecUtil.isHardwareAccelerated(mediaCodecInfo));
            } else {
                savePlayRecord(true);
            }
        }
        return super.onInfo(mp, what, extra, info);
    }

    @Override
    protected boolean onError(IMediaPlayer mp, int what, int extra) {
        if (mp.hashCode() != internalHashCode) {
            Log.i(TAG, "ignore error msg " + what + "," + extra + " for " + mp);
            return true;
        }
        if (!enableRebuildFeature) {
            return super.onError(mp, what, extra);
        }
        int rebuildType = -1;
        if (extra == IjkMediaPlayer.MEDIA_NO_AUDIO_RECV || extra == IjkMediaPlayer.MEDIA_NO_VIDEO_RECV || extra == IjkMediaPlayer.MEDIA_NO_VIDEO_BLOCK) {
            rebuildType = getPlayerType();
        } else if (extra == IjkMediaPlayer.MEDIA_VIDEO_DECODER_OPEN_ERROR) {
            savePlayRecord(false);
            rebuildType = PLAYER_TYPE_AMP;
        } else {
            for (Map.Entry<Pair<Integer, Integer>, Integer> entry : customRebuildEvent.entrySet()) {
                Pair<Integer, Integer> pair = entry.getKey();
                int playerType = entry.getValue();
                if (what == pair.first && extra == pair.second) {
                    savePlayRecord(false);
                    Log.i(TAG, "rebuild player due to custom request " + pair + ", " + playerType);
                    rebuildType = playerType;
                    break;
                }
            }
        }
        if (rebuildType > 0 && appHandler != null) {
            rebuildInternalPlayer(rebuildType, what, extra);
            return true;
        }
        return super.onError(mp, what, extra);
    }

    @Override
    protected void onTimedText(IMediaPlayer mp, IjkTimedText text) {
        if (mp.hashCode() != internalHashCode) {
            Log.i(TAG, "ignore time text msg for " + mp);
            return;
        }
        super.onTimedText(mp, text);
    }

    private void savePlayRecord(boolean success) {
        int playerType = getPlayerType();
        String codec = null;
        int profile = -1;
        int level = -1;
        int width = getVideoWidth();
        int height = getVideoHeight();
        if (playerType == PLAYER_TYPE_IJK) {
            ITrackInfo[] trackInfoArray = getTrackInfo();
            for (ITrackInfo trackInfo : trackInfoArray) {
                if (trackInfo.getTrackType() == ITrackInfo.MEDIA_TRACK_TYPE_VIDEO) {
                    codec = trackInfo.getFormat().getString(IjkMediaMeta.IJKM_KEY_CODEC_NAME);
                    profile = trackInfo.getFormat().getInteger(IjkMediaMeta.IJKM_KEY_CODEC_PROFILE_ID);
                    level = trackInfo.getFormat().getInteger(IjkMediaMeta.IJKM_KEY_CODEC_LEVEL);
                    if (width == 0 || height == 0) {
                        width = trackInfo.getFormat().getInteger(IjkMediaMeta.IJKM_KEY_WIDTH);
                        height = trackInfo.getFormat().getInteger(IjkMediaMeta.IJKM_KEY_HEIGHT);
                    }
                }
            }
        } else if (playerType == PLAYER_TYPE_EXO) {
            ITrackInfo[] trackInfoArray = getTrackInfo();
            for (ITrackInfo trackInfo : trackInfoArray) {
                if (trackInfo.getTrackType() == ITrackInfo.MEDIA_TRACK_TYPE_VIDEO) {
                    codec = trackInfo.getFormat().getString(IjkMediaMeta.IJKM_KEY_CODEC_NAME);
                    Pair<Integer, Integer> pair = MediaCodecUtil.getCodecProfileAndLevel(codec);
                    if (pair != null) {
                        profile = pair.first;
                        level = pair.second;
                    }
                }
            }
        } else if (playerType == PLAYER_TYPE_AMP) {
            AndroidMediaPlayer androidMediaPlayer = (AndroidMediaPlayer) getInternalPlayer();
            MediaPlayer.TrackInfo[] trackInfoArray = androidMediaPlayer.getInternalMediaPlayer().getTrackInfo();
            for (MediaPlayer.TrackInfo trackInfo : trackInfoArray) {
                if (trackInfo.getTrackType() == MediaPlayer.TrackInfo.MEDIA_TRACK_TYPE_VIDEO) {
                    Parcel parcel = Parcel.obtain();
                    trackInfo.writeToParcel(parcel, 0);
                    parcel.setDataPosition(0);
                    parcel.readInt();
                    codec = parcel.readString();
                    parcel.recycle();
                    profile = 0;
                    level = 0;
                }
            }
        }
        if (codec != null && width > 0 && height > 0 && profile >= 0 && level >= 0) {
            codec = formatCodec(codec);
            PlayRecordManager.getInstance().savePlayRecord(playerType, codec2mime(codec), codec, profile, level, width, height, success);
        } else {
            Log.e(TAG, "invalid record playerType:" + playerType + ",codec:" + codec + ",profile:" + profile + ",level:" + level +
                    ",width:" + width + ",height:" + height + ",success:" + success);
        }
    }

    private String formatCodec(String codec) {
        if (codec.contains("avc1")) {
            codec = "avc1";
        } else if (codec.contains("avc2")) {
            codec = "avc2";
        } else if (codec.contains("avc")) {
            codec = "avc";
        } else if (codec.contains("h264")) {
            codec = "h264";
        } else if (codec.contains("hevc")) {
            codec = "hevc";
        } else if (codec.contains("hvc1")) {
            codec = "hvc1";
        } else if (codec.contains("hev1")) {
            codec = "hev1";
        } else if (codec.contains("h265")) {
            codec = "h265";
        }
        return codec;
    }

    private String codec2mime(String codec) {
        if (codec.contains("avc") || codec.contains("h264")) {
            return "h264";
        } else if (codec.contains("hevc") || codec.contains("hvc1") || codec.contains("hev1") || codec.contains("h265")) {
            return "h265";
        }
        return codec;
    }
}
