package com.bitkernel.stream.rapid.player.egl;

import android.opengl.GLES20;
import android.os.Handler;
import android.os.HandlerThread;
import android.util.Log;
import android.view.Surface;

import androidx.annotation.NonNull;

public abstract class AbsTextureRenderer implements Renderer {
    private static final String TAG = "AbsTextureRenderer";

    private final EGL egl;
    private final MapsProgram glProgram;
    private OnStateChangedListener stateChangedListener;
    private HandlerThread eglThread;
    private Handler eglHandler;
    private Surface currentOutputSurface;

    public AbsTextureRenderer() {
        this.egl = new EGL();
        this.glProgram = new MapsProgram();
    }

    protected void runOnEglThread(Runnable work) {
        if (Thread.currentThread() == eglThread) {
            work.run();
        } else {
            eglHandler.post(work);
        }
    }

    @Override
    public void surfaceCreated(Surface surface) {
        eglThread = new HandlerThread(TAG) {
            @Override
            public void run() {
                Log.i(TAG, "egl thread start");
                super.run();
                Log.i(TAG, "egl thread end");
            }
        };
        eglThread.start();
        eglHandler = new Handler(eglThread.getLooper());
        runOnEglThread(() -> {
            Log.i(TAG, "surfaceCreated");
            currentOutputSurface = surface;
            egl.init();
            egl.setSurface(currentOutputSurface, true);
            glProgram.surfaceCreated();
            onPrepared();
        });
    }

    @Override
    public void surfaceChanged(int width, int height) {
        runOnEglThread(() -> {
            Log.i(TAG, "surfaceChanged " + width + "x" + height);
            GLES20.glViewport(0, 0, width, height);
        });
    }

    @Override
    public void surfaceDestroyed() {
        Log.i(TAG, "surfaceDestroyed");
        runOnEglThread(egl::release);
        eglThread.quitSafely();
        try {
            eglThread.join();
        } catch (InterruptedException e) {
            e.printStackTrace();
        }
        onReleased();
    }

    @Override
    public void setStateChangedListener(OnStateChangedListener listener) {
        this.stateChangedListener = listener;
    }

    @Override
    public void setOutputSurface(Surface surface) {
        runOnEglThread(() -> {
            if (currentOutputSurface == surface) {
                return;
            }
            currentOutputSurface = surface;
            egl.setSurface(surface);
        });
    }

    @Override
    public void requestRender(MapsTexture pipeline) {
        runOnEglThread(() -> {
            glProgram.drawFrame(pipeline);
            egl.swapBuffers();
        });
    }

    @Override
    public abstract Surface createInputSurface();

    @Override
    public abstract void releaseInputSurface(Surface surface);

    @Override
    public abstract void disableInputSurface(Surface surface);

    protected void onPrepared() {
        if (stateChangedListener != null) {
            stateChangedListener.onPrepared();
        }
    }

    protected void onReleased() {
        if (stateChangedListener != null) {
            stateChangedListener.onReleased();
        }
    }
}
