package com.bitkernel.stream.rapid.player.egl;

import android.opengl.GLES20;
import android.util.Log;

import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.FloatBuffer;

public class MapsProgram {

    private static final String TAG = "MapsProgram";

    private static final String TEXTURE_VERTEX =
            "attribute vec4 aPosition;\n" +
            "attribute vec4 aTextureCoord;\n" +
            "varying vec2 vTextureCoord;\n" +
            "uniform mat4 uTexTransform;\n" +
            "uniform mat4 uTransformationMatrix;\n" +
            "void main() {\n" +
            "  gl_Position = uTransformationMatrix * aPosition;\n" +
            "  vTextureCoord = (uTexTransform * aTextureCoord).xy;\n" +
            "}";
    private static final String TEXTURE_FRAGMENT =
            "#extension GL_OES_EGL_image_external : require\n" +
            "precision mediump float;\n" +
            "varying vec2 vTextureCoord;\n" +
            "uniform samplerExternalOES uTexture;\n" +
            "void main() {\n" +
            "  gl_FragColor = texture2D(uTexture, vTextureCoord);\n" +
            "}";

    private static final float[] TRANSFORM_MAT = {
            1, 0, 0, 0,
            0, 1, 0, 0,
            0, 0, 1, 0,
            0, 0, 0, 1
    };
    private static final float[] VERTICES_COORDINATE = {
            -1, -1, 0, 1,
            1, -1, 0, 1,
            -1, 1, 0, 1,
            1, 1, 0, 1
    };
    private static final float[] TEXTURE_COORDINATE = {
            0, 0, 0, 1,
            1, 0, 0, 1,
            0, 1, 0, 1,
            1, 1, 0, 1
    };

    private final FloatBuffer mVerticesBuffer;
    private final FloatBuffer mTextureBuffer;

    private int mProgram;
    private int aPositionHandle;
    private int aTextureHandle;
    private int uTexTransformHandle;
    private int uTransformationMatrixHandle;
    private int uSamplerOESHandle;

    public MapsProgram() {
        mVerticesBuffer = ByteBuffer
                .allocateDirect(VERTICES_COORDINATE.length * 4)
                .order(ByteOrder.nativeOrder())
                .asFloatBuffer()
                .put(VERTICES_COORDINATE);
        mVerticesBuffer.position(0);

        mTextureBuffer = ByteBuffer
                .allocateDirect(TEXTURE_COORDINATE.length * 4)
                .order(ByteOrder.nativeOrder())
                .asFloatBuffer()
                .put(TEXTURE_COORDINATE);
        mTextureBuffer.position(0);
    }

    /**
     * Initializes GL state.  Call this after the EGL surface has been created and made current.
     */
    public void surfaceCreated() {
        mProgram = createProgram(TEXTURE_VERTEX, TEXTURE_FRAGMENT);
        if (mProgram == 0) {
            throw new RuntimeException("failed creating program");
        }
        aPositionHandle = GLES20.glGetAttribLocation(mProgram, "aPosition");
        checkGlError("glGetAttribLocation aPosition");
        if (aPositionHandle == -1) {
            throw new RuntimeException("Could not get attrib location for aPosition");
        }
        aTextureHandle = GLES20.glGetAttribLocation(mProgram, "aTextureCoord");
        checkGlError("glGetAttribLocation aTextureCoord");
        if (aTextureHandle == -1) {
            throw new RuntimeException("Could not get attrib location for aTextureCoord");
        }
        uTransformationMatrixHandle = GLES20.glGetUniformLocation(mProgram, "uTransformationMatrix");
        checkGlError("glGetUniformLocation uTransformationMatrix");
        if (uTransformationMatrixHandle == -1) {
            throw new RuntimeException("Could not get uniform location for uTransformationMatrix");
        }
        uTexTransformHandle = GLES20.glGetUniformLocation(mProgram, "uTexTransform");
        checkGlError("glGetUniformLocation uTexTransform");
        if (uTexTransformHandle == -1) {
            throw new RuntimeException("Could not get uniform location for uTexTransform");
        }

        uSamplerOESHandle = GLES20.glGetUniformLocation(mProgram, "uTexture");
        checkGlError("glGetUniformLocation uTexture");
        if (uSamplerOESHandle == -1) {
            throw new RuntimeException("Could not get attrib location for uTexture");
        }
    }

    private int loadShader(int shaderType, String source) {
        int shader = GLES20.glCreateShader(shaderType);
        checkGlError("glCreateShader type=" + shaderType);
        GLES20.glShaderSource(shader, source);
        GLES20.glCompileShader(shader);
        int[] compiled = new int[1];
        GLES20.glGetShaderiv(shader, GLES20.GL_COMPILE_STATUS, compiled, 0);
        if (compiled[0] == 0) {
            Log.e(TAG, "Could not compile shader " + shaderType + ":");
            Log.e(TAG, " " + GLES20.glGetShaderInfoLog(shader));
            GLES20.glDeleteShader(shader);
            shader = 0;
        }
        return shader;
    }

    private int createProgram(String vertexSource, String fragmentSource) {
        int vertexShader = loadShader(GLES20.GL_VERTEX_SHADER, vertexSource);
        if (vertexShader == 0) {
            return 0;
        }
        int pixelShader = loadShader(GLES20.GL_FRAGMENT_SHADER, fragmentSource);
        if (pixelShader == 0) {
            return 0;
        }

        int program = GLES20.glCreateProgram();
        checkGlError("glCreateProgram");
        if (program == 0) {
            Log.e(TAG, "Could not create program");
        }
        GLES20.glAttachShader(program, vertexShader);
        checkGlError("glAttachShader");
        GLES20.glAttachShader(program, pixelShader);
        checkGlError("glAttachShader");
        GLES20.glLinkProgram(program);
        int[] linkStatus = new int[1];
        GLES20.glGetProgramiv(program, GLES20.GL_LINK_STATUS, linkStatus, 0);
        if (linkStatus[0] != GLES20.GL_TRUE) {
            Log.e(TAG, "Could not link program: ");
            Log.e(TAG, GLES20.glGetProgramInfoLog(program));
            GLES20.glDeleteProgram(program);
            program = 0;
        }
        return program;
    }

    public void drawFrame(MapsTexture pipeline) {
        GLES20.glUseProgram(mProgram);
        checkGlError("glUseProgram");

        pipeline.draw(uSamplerOESHandle, uTexTransformHandle);
        checkGlError("pipeline draw");

        GLES20.glUniformMatrix4fv(uTransformationMatrixHandle, 1, false, TRANSFORM_MAT, 0);

        GLES20.glVertexAttribPointer(
                aPositionHandle, 4, GLES20.GL_FLOAT, false, 0, mVerticesBuffer);
        checkGlError("glVertexAttribPointer aPositionHandle");
        GLES20.glEnableVertexAttribArray(aPositionHandle);
        checkGlError("glEnableVertexAttribArray aPositionHandle");

        GLES20.glVertexAttribPointer(
                aTextureHandle, 4, GLES20.GL_FLOAT, false, 0, mTextureBuffer);
        checkGlError("glVertexAttribPointer aTextureHandle");
        GLES20.glEnableVertexAttribArray(aTextureHandle);
        checkGlError("glEnableVertexAttribArray aTextureHandle");

        GLES20.glClear(GLES20.GL_COLOR_BUFFER_BIT);
        GLES20.glDrawArrays(GLES20.GL_TRIANGLE_STRIP, 0, 4);
        checkGlError("glDrawArrays");
    }

    private static void checkGlError(String op) {
        int error;
        while ((error = GLES20.glGetError()) != GLES20.GL_NO_ERROR) {
            Log.e(TAG, op + ": glError " + error);
            throw new RuntimeException(op + ": glError " + error);
        }
    }
}
