package com.bitkernel.stream.rapid.player;

import android.content.Context;
import android.net.Uri;
import android.os.Bundle;
import android.util.Log;
import android.util.Pair;
import android.view.Surface;
import android.view.SurfaceHolder;

import java.io.FileDescriptor;
import java.io.IOException;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;

import tv.danmaku.ijk.media.player.IMediaPlayer;
import tv.danmaku.ijk.media.player.IjkMediaPlayer;
import tv.danmaku.ijk.media.player.IjkTimedText;
import tv.danmaku.ijk.media.player.MediaInfo;
import tv.danmaku.ijk.media.player.misc.IMediaDataSource;
import tv.danmaku.ijk.media.player.misc.ITrackInfo;

public abstract class AbsRapidMediaPlayer implements IRapidMediaPlayer {
    protected final PlayerBuilder playerBuilder;
    protected final PlayerStatistic playerStatistic;
    protected final AtomicBoolean rebuilding = new AtomicBoolean(false);
    protected int rebuildWhat = 0;
    protected int rebuildExtra = 0;
    private IMediaPlayer internalPlayer;
    protected int internalHashCode;
    private int playerType;

    AbsRapidMediaPlayer(PlayerStatistic playerStatistic, PlayerBuilder playerBuilder) {
        this.playerStatistic = playerStatistic;
        this.playerBuilder = playerBuilder;
    }

    protected void setInternalPlayer(Pair<IMediaPlayer, Integer> pair) {
        Log.i("AbsRapidMediaPlayer", "setInternalPlayer " + pair.first + "," + pair.second);
        this.internalPlayer = pair.first;
        this.internalPlayer.setOnPreparedListener(internalPreparedListener);
        this.internalPlayer.setOnCompletionListener(internalCompletionListener);
        this.internalPlayer.setOnBufferingUpdateListener(internalOnBufferingUpdateListener);
        this.internalPlayer.setOnSeekCompleteListener(internalOnSeekCompleteListener);
        this.internalPlayer.setOnVideoSizeChangedListener(internalOnVideoSizeChangedListener);
        this.internalPlayer.setOnErrorListener(internalErrorListener);
        this.internalPlayer.setOnInfoListener(internalInfoListener);
        this.internalPlayer.setOnTimedTextListener(internalOnTimedTextListener);
        this.internalPlayer.setOnDrmInitInfoListener(internalDrmInitInfoListener);
        if (this.internalPlayer instanceof IjkMediaPlayer) {
            ((IjkMediaPlayer) this.internalPlayer).setOnNativeInvokeListener(internalNativeInvokeListener);
        }
        this.playerType = pair.second;
        this.internalHashCode = this.internalPlayer.hashCode();
        playerStatistic.setPlayerType(playerType, hashCode());
    }

    @Override
    public void setDisplay(SurfaceHolder sh) {
        internalPlayer.setDisplay(sh);
    }

    @Override
    public void setSurface(Surface surface) {
        internalPlayer.setSurface(surface);
    }

    @Override
    public void setDataSource(Context context, Uri uri) throws IOException, IllegalArgumentException, SecurityException, IllegalStateException {
        internalPlayer.setDataSource(context, uri);
        playerStatistic.setDataSource(uri);
    }

    @Override
    public void setDataSource(Context context, Uri uri, Map<String, String> headers) throws IOException, IllegalArgumentException, SecurityException, IllegalStateException {
        internalPlayer.setDataSource(context, uri, headers);
        playerStatistic.setDataSource(uri);
    }

    @Override
    public void setDataSource(FileDescriptor fd) throws IOException, IllegalArgumentException, IllegalStateException {
        internalPlayer.setDataSource(fd);
        playerStatistic.setDataSource(Uri.parse("http://FileDescriptor"));
    }

    @Override
    public void setDataSource(String path) throws IOException, IllegalArgumentException, SecurityException, IllegalStateException {
        internalPlayer.setDataSource(path);
        playerStatistic.setDataSource(Uri.parse(path));
    }

    @Override
    public void setDataSource(IMediaDataSource mediaDataSource) {
        internalPlayer.setDataSource(mediaDataSource);
        playerStatistic.setDataSource(Uri.parse("http://IMediaDataSource"));
    }

    @Override
    public String getDataSource() {
        return internalPlayer.getDataSource();
    }

    @Override
    public void prepareAsync() throws IllegalStateException {
        internalPlayer.prepareAsync();
        playerStatistic.prepareAsync();
    }

    @Override
    public void start() throws IllegalStateException {
        getInternalPlayer().start();
        playerStatistic.start();
    }

    @Override
    public void stop() throws IllegalStateException {
        playerStatistic.stopStart();
        getInternalPlayer().stop();
        playerStatistic.stopFinished();
    }

    @Override
    public void pause() throws IllegalStateException {
        getInternalPlayer().pause();
        playerStatistic.pause();
    }

    @Override
    public int getVideoWidth() {
        return getInternalPlayer().getVideoWidth();
    }

    @Override
    public int getVideoHeight() {
        return getInternalPlayer().getVideoHeight();
    }

    @Override
    public boolean isPlaying() {
        return getInternalPlayer().isPlaying();
    }

    @Override
    public void seekTo(long msec) throws IllegalStateException {
        getInternalPlayer().seekTo(msec);
        playerStatistic.seekTo(msec);
    }

    @Override
    public long getCurrentPosition() {
        return getInternalPlayer().getCurrentPosition();
    }

    @Override
    public long getDuration() {
        return getInternalPlayer().getDuration();
    }

    @Override
    public void release() {
        playerStatistic.releaseStart();
        IMediaPlayer player = getInternalPlayer();
        player.release();
        player.setOnPreparedListener(null);
        player.setOnCompletionListener(null);
        player.setOnBufferingUpdateListener(null);
        player.setOnSeekCompleteListener(null);
        player.setOnVideoSizeChangedListener(null);
        player.setOnErrorListener(null);
        player.setOnInfoListener(null);
        player.setOnTimedTextListener(null);
        player.setOnDrmInitInfoListener(null);
        if (player instanceof IjkMediaPlayer) {
            ((IjkMediaPlayer) player).setOnNativeInvokeListener(internalNativeInvokeListener);
        }
        playerStatistic.releaseFinished();
    }

    @Override
    public void reset() {
        getInternalPlayer().reset();
        playerStatistic.reset();
    }

    @Override
    public int getAudioSessionId() {
        return getInternalPlayer().getAudioSessionId();
    }

    @Override
    public MediaInfo getMediaInfo() {
        return getInternalPlayer().getMediaInfo();
    }

    @Override
    public boolean isPlayable() {
        return getInternalPlayer().isPlayable();
    }

    @Override
    public ITrackInfo[] getTrackInfo() {
        return getInternalPlayer().getTrackInfo();
    }

    @Override
    public int getVideoSarNum() {
        return getInternalPlayer().getVideoSarNum();
    }

    @Override
    public int getVideoSarDen() {
        return getInternalPlayer().getVideoSarDen();
    }

    @Override
    public void setLooping(boolean looping) {
        getInternalPlayer().setLooping(looping);
    }

    @Override
    public boolean isLooping() {
        return getInternalPlayer().isLooping();
    }

    @Override
    public void setScreenOnWhilePlaying(boolean screenOn) {
        getInternalPlayer().setScreenOnWhilePlaying(screenOn);
    }

    @Override
    public void setKeepInBackground(boolean keepInBackground) {
        getInternalPlayer().setKeepInBackground(keepInBackground);
    }

    @Override
    public void setLogEnabled(boolean enable) {
        getInternalPlayer().setLogEnabled(enable);
    }

    @Override
    public void setWakeMode(Context context, int mode) {
        getInternalPlayer().setWakeMode(context, mode);
    }

    @Override
    public void setAudioStreamType(int streamtype) {
        getInternalPlayer().setAudioStreamType(streamtype);
    }

    @Override
    public void setVolume(float leftVolume, float rightVolume) {
        getInternalPlayer().setVolume(leftVolume, rightVolume);
    }

    @Override
    public void setSpeed(float speed) {
        getInternalPlayer().setSpeed(speed);
    }

    @Override
    public String getVideoCodecName() {
        return getInternalPlayer().getVideoCodecName();
    }

    @Override
    public void setDrmInfo(int drmType, boolean multiSession, String licenceServerUrl, Map<String, String> headers, String reqMethod) {
        internalPlayer.setDrmInfo(drmType, multiSession, licenceServerUrl, headers, reqMethod);
    }

    @Override
    public void setDrmInfo(int drmType, boolean multiSession, String licenceServerUrl, Map<String, String> headers, String reqMethod, byte[] offlineLicenseKeySetId) {
        internalPlayer.setDrmInfo(drmType, multiSession, licenceServerUrl, headers, reqMethod, offlineLicenseKeySetId);
    }

    @Override
    public void setDrmInfo(int drmType, boolean multiSession, String licenceServerUrl, Map<String, String> headers, String reqMethod, byte[] offlineLicenseKeySetId, String offlineLicenseDrmInitInfo) {
        internalPlayer.setDrmInfo(drmType, multiSession, licenceServerUrl, headers, reqMethod, offlineLicenseKeySetId, offlineLicenseDrmInitInfo);
    }

    @Override
    public void updateDrmInitInfo(String stringObj) {
        internalPlayer.updateDrmInitInfo(stringObj);
    }

    protected OnPreparedListener outsidePreparedListener;
    protected final OnPreparedListener internalPreparedListener = AbsRapidMediaPlayer.this::onPrepared;

    protected void onPrepared(IMediaPlayer mp) {
        if (outsidePreparedListener != null) {
            outsidePreparedListener.onPrepared(mp);
        }
    }

    @Override
    public void setOnPreparedListener(OnPreparedListener listener) {
        outsidePreparedListener = listener;
    }

    protected OnCompletionListener outsideOnCompletionListener;
    protected final OnCompletionListener internalCompletionListener = AbsRapidMediaPlayer.this::onCompletion;

    protected void onCompletion(IMediaPlayer mp) {
        playerStatistic.onCompletion();
        if (outsideOnCompletionListener != null) {
            outsideOnCompletionListener.onCompletion(mp);
        }
    }

    @Override
    public void setOnCompletionListener(OnCompletionListener listener) {
        outsideOnCompletionListener = listener;
    }

    protected OnBufferingUpdateListener outsideOnBufferingUpdateListener;
    protected final OnBufferingUpdateListener internalOnBufferingUpdateListener = AbsRapidMediaPlayer.this::onBufferingUpdate;

    protected void onBufferingUpdate(IMediaPlayer mp, int percent) {
        if (outsideOnBufferingUpdateListener != null) {
            outsideOnBufferingUpdateListener.onBufferingUpdate(mp, percent);
        }
    }

    @Override
    public void setOnBufferingUpdateListener(OnBufferingUpdateListener listener) {
        outsideOnBufferingUpdateListener = listener;
    }

    protected OnSeekCompleteListener outsideOnSeekCompleteListener;
    protected final OnSeekCompleteListener internalOnSeekCompleteListener = AbsRapidMediaPlayer.this::onSeekComplete;

    protected void onSeekComplete(IMediaPlayer mp) {
        playerStatistic.onSeekComplete();
        if (outsideOnSeekCompleteListener != null) {
            outsideOnSeekCompleteListener.onSeekComplete(mp);
        }
    }

    @Override
    public void setOnSeekCompleteListener(OnSeekCompleteListener listener) {
        outsideOnSeekCompleteListener = listener;
    }

    protected OnVideoSizeChangedListener outsideOnVideoSizeChangedListener;
    protected final OnVideoSizeChangedListener internalOnVideoSizeChangedListener = AbsRapidMediaPlayer.this::onVideoSizeChanged;
    protected int currentVideoWidth = 0;
    protected int currentVideoHeight = 0;
    protected int currentSarNum = 0;
    protected int currentSarDen = 0;

    protected void onVideoSizeChanged(IMediaPlayer mp, int width, int height, int sar_num, int sar_den) {
        if (currentVideoWidth == width && currentVideoHeight == height && currentSarNum == sar_num && currentSarDen == sar_den) {
            Log.w("AbsRapidMediaPlayer", "ignore same video size message");
            return;
        }
        currentVideoWidth = width;
        currentVideoHeight = height;
        currentSarNum = sar_num;
        currentSarDen = sar_den;
        if (outsideOnVideoSizeChangedListener != null) {
            outsideOnVideoSizeChangedListener.onVideoSizeChanged(mp, width, height, sar_num, sar_den);
        }
    }

    @Override
    public void setOnVideoSizeChangedListener(OnVideoSizeChangedListener listener) {
        outsideOnVideoSizeChangedListener = listener;
    }

    protected OnErrorListener outsideErrorListener;
    protected final OnErrorListener internalErrorListener = AbsRapidMediaPlayer.this::onError;

    protected boolean onError(IMediaPlayer mp, int what, int extra) {
        playerStatistic.onError(what, extra);
        if (outsideErrorListener != null) {
            return outsideErrorListener.onError(mp, what, extra);
        }
        return false;
    }

    @Override
    public void setOnErrorListener(OnErrorListener listener) {
        outsideErrorListener = listener;
    }

    protected OnInfoListener outsideInfoListener;
    protected final OnInfoListener internalInfoListener = AbsRapidMediaPlayer.this::onInfo;

    protected boolean onInfo(IMediaPlayer mp, int what, int extra, String info) {
        playerStatistic.onInfo(what, extra, info);
        if (outsideInfoListener != null) {
            return outsideInfoListener.onInfo(mp, what, extra, info);
        }
        return false;
    }

    @Override
    public void setOnInfoListener(OnInfoListener listener) {
        outsideInfoListener = listener;
    }

    protected OnTimedTextListener outsideOnTimedTextListener;
    protected final OnTimedTextListener internalOnTimedTextListener = AbsRapidMediaPlayer.this::onTimedText;

    protected void onTimedText(IMediaPlayer mp, IjkTimedText text) {
        if (outsideOnTimedTextListener != null) {
            outsideOnTimedTextListener.onTimedText(mp, text);
        }
    }

    @Override
    public void setOnTimedTextListener(OnTimedTextListener listener) {
        outsideOnTimedTextListener = listener;
    }

    protected OnDrmInitInfoListener outsideDrmInitInfoListener;
    protected final OnDrmInitInfoListener internalDrmInitInfoListener = AbsRapidMediaPlayer.this::onDrmInitInfo;

    protected void onDrmInitInfo(IMediaPlayer mp, String drmInitInfo) {
        if (outsideDrmInitInfoListener != null) {
            outsideDrmInitInfoListener.onDrmInitInfo(mp, drmInitInfo);
        }
    }

    @Override
    public void setOnDrmInitInfoListener(OnDrmInitInfoListener listener) {
        outsideDrmInitInfoListener = listener;
    }

    protected IjkMediaPlayer.OnNativeInvokeListener outsideNativeInvokeListener;
    protected IjkMediaPlayer.OnNativeInvokeListener internalNativeInvokeListener = AbsRapidMediaPlayer.this::onNativeInvoke;

    protected boolean onNativeInvoke(int what, Bundle args) {
        if (outsideNativeInvokeListener != null) {
            return outsideNativeInvokeListener.onNativeInvoke(what, args);
        }
        return false;
    }

    @Override
    public void setOnNativeInvokeListener(IjkMediaPlayer.OnNativeInvokeListener listener) {
        outsideNativeInvokeListener = listener;
    }

    @Override
    public int getSelectedTrack(int trackType) {
        return getInternalPlayer().getSelectedTrack(trackType);
    }

    @Override
    public void selectTrack(int track) {
        IMediaPlayer internalPlayer = getInternalPlayer();
        ITrackInfo[] trackInfoArray = internalPlayer.getTrackInfo();
        if (trackInfoArray == null || trackInfoArray.length == 0) {
            Log.e("AbsRapidMediaPlayer", "invalid track array");
            return;
        }
        if (track < 0 || track >= trackInfoArray.length) {
            Log.e("AbsRapidMediaPlayer", "invalid track id " + track);
            return;
        }
        ITrackInfo selectTrackInfo = trackInfoArray[track];
        int currentTrack = internalPlayer.getSelectedTrack(selectTrackInfo.getTrackType());
        if (currentTrack == track) {
            Log.e("AbsRapidMediaPlayer", "selected same track and ignore");
            return;
        }
        if (internalPlayer instanceof IjkMediaPlayer) {
            IjkMediaPlayer ijkMediaPlayer = (IjkMediaPlayer) internalPlayer;
            if (ijkMediaPlayer.getAdaptivePlayback() != 0) {
                ijkMediaPlayer.setPreferVideoTrack(track);
                return;
            }
        }
        internalPlayer.selectTrack(track);
    }

    @Override
    public void deselectTrack(int track) {
        getInternalPlayer().deselectTrack(track);
    }

    @Override
    public void setPreferVideoTrack(int track) {
        IMediaPlayer internalPlayer = getInternalPlayer();
        if (internalPlayer instanceof IjkMediaPlayer) {
            IjkMediaPlayer ijkMediaPlayer = (IjkMediaPlayer) internalPlayer;
            if (ijkMediaPlayer.getAdaptivePlayback() != 0) {
                ijkMediaPlayer.setPreferVideoTrack(track);
            }
        }
    }

    @Override
    public int getPlayerType() {
        return playerType;
    }

    @Override
    public IMediaPlayer getInternalPlayer() {
        if (!rebuilding.get()) {
            return internalPlayer;
        }
        synchronized (rebuilding) {
            return internalPlayer;
        }
    }
}
