package com.bitkernel.stream.rapid.player;

import static com.bitkernel.stream.rapid.player.IRapidMediaPlayer.PLAYER_TYPE_AMP;
import static com.bitkernel.stream.rapid.player.IRapidMediaPlayer.PLAYER_TYPE_EXO;
import static com.bitkernel.stream.rapid.player.IRapidMediaPlayer.PLAYER_TYPE_IJK;
import static com.bitkernel.stream.rapid.player.IRapidMediaPlayer.PLAYER_TYPE_UNKNOWN;

import android.content.Context;
import android.util.Log;
import android.util.Pair;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.bitkernel.stream.rapid.player.utils.CommonUtil;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.util.HashMap;
import java.util.Map;

import tv.danmaku.ijk.media.player.AndroidMediaPlayer;
import tv.danmaku.ijk.media.player.IMediaPlayer;
import tv.danmaku.ijk.media.player.IjkMediaCodecInfo;
import tv.danmaku.ijk.media.player.IjkMediaPlayer;

public class DefaultPlayerBuilder implements PlayerBuilder {
    private static int DEFAULT_PLAYER_TYPE = PLAYER_TYPE_UNKNOWN;
    private static int AVC_MEDIACODEC_RANK = 0;
    private static int HEVC_MEDIACODEC_RANK = 0;
    private static boolean AVC_MEDIACODEC_AVAILABLE = true;
    private static boolean HEVC_MEDIACODEC_AVAILABLE = true;

    private static int findCodecRank(String mimeType, int profile, int level) {
        IjkMediaPlayer.DefaultMediaCodecSelector selector = IjkMediaPlayer.DefaultMediaCodecSelector.sInstance;
        IjkMediaCodecInfo codecInfo = selector.onIjkMediaCodecSelect(null, mimeType, profile, level, false);
        return codecInfo != null ? codecInfo.mRank : IjkMediaCodecInfo.RANK_SOFTWARE;
    }

    public static void preload(int h264Rank, int h265Rank) {
        AVC_MEDIACODEC_RANK = h264Rank == 0 ? findCodecRank("video/avc", 100, 40) : h264Rank;
        HEVC_MEDIACODEC_RANK = h265Rank == 0 ? findCodecRank("video/hevc", 1, 93) : h265Rank;
        AVC_MEDIACODEC_AVAILABLE = AVC_MEDIACODEC_RANK >= IjkMediaCodecInfo.RANK_ACCEPTABLE;
        HEVC_MEDIACODEC_AVAILABLE = HEVC_MEDIACODEC_RANK >= IjkMediaCodecInfo.RANK_ACCEPTABLE;
        Log.i("DefaultPlayerBuilder", "preload avc available:" + AVC_MEDIACODEC_AVAILABLE + ", hevc available:" + HEVC_MEDIACODEC_AVAILABLE);
    }

    public static int getAvcMediacodecRank() {
        return AVC_MEDIACODEC_RANK;
    }

    public static int getHevcMediacodecRank() {
        return HEVC_MEDIACODEC_RANK;
    }

    public static boolean isAvcMediacodecAvailable() {
        return AVC_MEDIACODEC_AVAILABLE;
    }

    public static boolean isHevcMediacodecAvailable() {
        return HEVC_MEDIACODEC_AVAILABLE;
    }

    @Override
    public void setDefaultPlayerType(int playerType) {
        DEFAULT_PLAYER_TYPE = playerType;
        Log.i("DefaultPlayerBuilder", "set default player type:" + DEFAULT_PLAYER_TYPE);
    }

    @Override
    public Pair<IMediaPlayer, Integer> createMediaPlayer(@Nullable Context context) {
        return createMediaPlayer(context, new HashMap<>());
    }

    @Override
    public Pair<IMediaPlayer, Integer> createMediaPlayer(@Nullable Context context, Map<String, String> params) {
        int playerType;

        if (params.containsKey(MLS)) {
            playerType = PLAYER_TYPE_IJK;
            return createMediaPlayer(context, playerType);
        }
        if (params.containsKey(ABR) || params.containsKey(DRM)) {
            playerType = (DEFAULT_PLAYER_TYPE == PLAYER_TYPE_IJK || DEFAULT_PLAYER_TYPE == PLAYER_TYPE_EXO) ? DEFAULT_PLAYER_TYPE : PLAYER_TYPE_IJK;
            return createMediaPlayer(context, playerType);
        }

        if (DEFAULT_PLAYER_TYPE == PLAYER_TYPE_IJK || DEFAULT_PLAYER_TYPE == PLAYER_TYPE_EXO || DEFAULT_PLAYER_TYPE == PLAYER_TYPE_AMP) {
            playerType = DEFAULT_PLAYER_TYPE;
            return createMediaPlayer(context, playerType);
        }

        if (!CommonUtil.isVirtualMachine() && (!AVC_MEDIACODEC_AVAILABLE || !HEVC_MEDIACODEC_AVAILABLE)) {
            playerType = PLAYER_TYPE_AMP;
        } else {
            playerType = PLAYER_TYPE_IJK;
        }
        if (PlayRecordManager.getInstance().checkPlayerType(playerType)) {
            return createMediaPlayer(context, playerType);
        }
        playerType = PlayRecordManager.getInstance().getBestPlayerType();
        return createMediaPlayer(context, playerType);
    }

    public Pair<IMediaPlayer, Integer> createMediaPlayer(@Nullable Context context, int playerType) {
        int targetPlayerType;
        if (playerType == PLAYER_TYPE_EXO && context != null) {
            targetPlayerType = PLAYER_TYPE_EXO;
        } else if (playerType == PLAYER_TYPE_AMP) {
            targetPlayerType = PLAYER_TYPE_AMP;
        } else {
            targetPlayerType = PLAYER_TYPE_IJK;
        }

        IMediaPlayer player = null;
        int realPlayerType = PLAYER_TYPE_UNKNOWN;
        switch (targetPlayerType) {
            case PLAYER_TYPE_IJK:
                player = createIjkMediaPlayer();
                realPlayerType = PLAYER_TYPE_IJK;
                break;
            case PLAYER_TYPE_EXO:
                player = createExoMediaPlayer(context);
                realPlayerType = PLAYER_TYPE_EXO;
                break;
            case PLAYER_TYPE_AMP:
                player = createAndroidMediaPlayer();
                realPlayerType = PLAYER_TYPE_AMP;
                break;
        }
        if (player == null) {
            player = createIjkMediaPlayer();
            realPlayerType = PLAYER_TYPE_IJK;
        }
        return Pair.create(player, realPlayerType);
    }

    @Override
    public IMediaPlayer createIjkMediaPlayer() {
        return new IjkMediaPlayer();
    }

    @Override
    public IMediaPlayer createExoMediaPlayer(@NonNull Context context) {
        try {
            Class<?> clazz = Class.forName("tv.danmaku.ijk.media.exo.IjkExoMediaPlayer");
            Constructor<?> constructor = clazz.getDeclaredConstructor(Context.class);
            Object obj = constructor.newInstance(context);
            return (IMediaPlayer) obj;
        } catch (ClassNotFoundException | NoSuchMethodException | IllegalAccessException |
                 InstantiationException | InvocationTargetException e) {
            Log.w("DefaultPlayerBuilder", "ExoPlayer is not implementation");
            return new IjkMediaPlayer();
        }
    }

    @Override
    public IMediaPlayer createAndroidMediaPlayer() {
        return new AndroidMediaPlayer();
    }
}
