package com.bitkernel.stream.rapid.player;

import static com.bitkernel.stream.rapid.player.IRapidMediaPlayer.PLAYER_TYPE_AMP;
import static com.bitkernel.stream.rapid.player.IRapidMediaPlayer.PLAYER_TYPE_EXO;
import static com.bitkernel.stream.rapid.player.IRapidMediaPlayer.PLAYER_TYPE_IJK;
import static com.bitkernel.stream.rapid.player.utils.DatabaseHelper.CODEC;
import static com.bitkernel.stream.rapid.player.utils.DatabaseHelper.FAIL_COUNT;
import static com.bitkernel.stream.rapid.player.utils.DatabaseHelper.HEIGHT;
import static com.bitkernel.stream.rapid.player.utils.DatabaseHelper.ID;
import static com.bitkernel.stream.rapid.player.utils.DatabaseHelper.LEVEL;
import static com.bitkernel.stream.rapid.player.utils.DatabaseHelper.MIME;
import static com.bitkernel.stream.rapid.player.utils.DatabaseHelper.PLAYER_TYPE;
import static com.bitkernel.stream.rapid.player.utils.DatabaseHelper.PLAY_RECORD_TABLE_NAME;
import static com.bitkernel.stream.rapid.player.utils.DatabaseHelper.PROFILE;
import static com.bitkernel.stream.rapid.player.utils.DatabaseHelper.SUCCESS_COUNT;
import static com.bitkernel.stream.rapid.player.utils.DatabaseHelper.WIDTH;

import android.annotation.SuppressLint;
import android.content.ContentValues;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.util.Log;

import androidx.annotation.NonNull;

import com.bitkernel.stream.rapid.player.utils.DatabaseHelper;

import java.util.ArrayList;
import java.util.List;

public final class PlayRecordManager {
    public static class Record {
        public int id;
        public int playerType;
        @NonNull
        public String mime;
        public String codec;
        public int profile;
        public int level;
        public int width;
        public int height;
        public int successCount;
        public int failCount;

        public Record(int playerType, @NonNull String mime, @NonNull String codec, int profile, int level, int width, int height,
                      int successCount, int failCount) {
            this(-1, playerType, mime, codec, profile, level, width, height, successCount, failCount);
        }

        public Record(int id, int playerType, @NonNull String mime, @NonNull String codec, int profile, int level, int width, int height,
                      int successCount, int failCount) {
            this.id = id;
            this.playerType = playerType;
            this.mime = mime;
            this.codec = codec;
            this.profile = profile;
            this.level = level;
            this.width = width;
            this.height = height;
            this.successCount = successCount;
            this.failCount = failCount;
        }

        public ContentValues toContentValues() {
            ContentValues cv = new ContentValues();
            if (id > 0) {
                cv.put(ID, id);
            }
            cv.put(PLAYER_TYPE, playerType);
            cv.put(MIME, mime);
            cv.put(CODEC, codec);
            cv.put(PROFILE, profile);
            cv.put(LEVEL, level);
            cv.put(WIDTH, width);
            cv.put(HEIGHT, height);
            cv.put(SUCCESS_COUNT, successCount);
            cv.put(FAIL_COUNT, failCount);
            return cv;
        }

        @NonNull
        @Override
        public String toString() {
            return "Record{" +
                    "id=" + id +
                    ", playerType=" + playerType +
                    ", mime='" + mime + '\'' +
                    ", codec='" + codec + '\'' +
                    ", profile=" + profile +
                    ", level=" + level +
                    ", width=" + width +
                    ", height=" + height +
                    ", successCount=" + successCount +
                    ", failCount=" + failCount +
                    '}';
        }
    }

    private static final String TAG = "PlayRecordManager";

    private static PlayRecordManager sInstance;

    private int forceUsingIjkInterval = 5;
    private int checkIjkCount = 0;

    private PlayRecordManager() {
    }

    public static PlayRecordManager getInstance() {
        if (sInstance == null) {
            synchronized (PlayRecordManager.class) {
                if (sInstance == null) {
                    sInstance = new PlayRecordManager();
                }
            }
        }
        return sInstance;
    }

    public void setForceUsingIjkInterval(int forceUsingIjkInterval) {
        this.forceUsingIjkInterval = forceUsingIjkInterval;
        Log.i(TAG, "setForceUsingIjkInterval " + forceUsingIjkInterval);
    }

    public void savePlayRecord(int playerType, @NonNull String mime, @NonNull String codec,
                               int profile, int level, int width, int height, boolean success) {
        List<Record> list = query(playerType, mime, codec, profile, level, width, height);
        Record record;
        if (!list.isEmpty()) {
            record = list.get(0);
        } else {
            record = new Record(playerType, mime, codec, profile, level, width, height, 0, 0);
        }
        if (success) {
            record.successCount++;
        } else {
            record.failCount++;
        }
        if (record.id > 0) {
            update(record);
            Log.i(TAG, "updatePlayRecord " + record);
        } else {
            insert(record);
            Log.i(TAG, "insertPlayRecord " + record);
        }
    }

    public List<Record> getAllRecord() {
        return query();
    }

    public List<Record> getAllRecord(int playerType) {
        return query(playerType);
    }

    public boolean checkPlayerType(int playerType) {
        if (playerType == PLAYER_TYPE_IJK) {
            if (checkIjkCount++ % forceUsingIjkInterval == 0) {
                return true;
            }
        }
        List<Record> all = getAllRecord(playerType);
        float score = getPlayerTypeScore(all);
        return score >= 1.0f;
    }

    public int getBestPlayerType() {
        List<Record> all = getAllRecord();
        List<Record> ijk = new ArrayList<>();
        List<Record> exo = new ArrayList<>();
        List<Record> amp = new ArrayList<>();
        for (Record record : all) {
            switch (record.playerType) {
                case PLAYER_TYPE_IJK:
                    ijk.add(record);
                    break;
                case PLAYER_TYPE_EXO:
                    exo.add(record);
                    break;
                case PLAYER_TYPE_AMP:
                    amp.add(record);
                    break;
            }
        }
        float ijkScore = getPlayerTypeScore(ijk);
        float exoScore = getPlayerTypeScore(exo);
        float ampScore = getPlayerTypeScore(amp);
        float maxScore = 0f;
        int bestPlayerType = PLAYER_TYPE_IJK;
        if (maxScore < ijkScore) {
            maxScore = ijkScore;
            bestPlayerType = PLAYER_TYPE_IJK;
        }
        if (maxScore < ampScore) {
            maxScore = ampScore;
            bestPlayerType = PLAYER_TYPE_AMP;
        }
        if (maxScore < exoScore) {
            bestPlayerType = PLAYER_TYPE_EXO;
        }
        return bestPlayerType;
    }

    private float getPlayerTypeScore(List<Record> recordList) {
        int successCount = 0;
        int failCount = 0;
        for (Record record : recordList) {
            // all fail
            if (record.successCount == 0 && record.failCount > 0) {
                failCount++;
            } else {
                successCount++;
            }
        }
        if (failCount + successCount == 0) {
            return 1.0f;
        }
        return successCount * 1.0f / (successCount + failCount);
    }

    private synchronized void update(Record record) {
        try (SQLiteDatabase db = DatabaseHelper.getInstance().getWritableDatabase()) {
            ContentValues cv = record.toContentValues();
            db.update(PLAY_RECORD_TABLE_NAME, cv, "id=?", new String[]{cv.getAsString(ID)});
        } catch (DatabaseHelper.InvalidContextException e) {
            Log.i(TAG, "update error: " + e.getMessage());
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private synchronized void insert(Record record) {
        try (SQLiteDatabase db = DatabaseHelper.getInstance().getWritableDatabase()) {
            ContentValues cv = record.toContentValues();
            db.insert(PLAY_RECORD_TABLE_NAME, null, cv);
        } catch (DatabaseHelper.InvalidContextException e) {
            Log.i(TAG, "insert error: " + e.getMessage());
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private synchronized List<Record> query() {
        String querySql = "select * from " + PLAY_RECORD_TABLE_NAME;
        return queryBySql(querySql);
    }

    private synchronized List<Record> query(int playerType) {
        String querySql = "select * from " + PLAY_RECORD_TABLE_NAME + " WHERE "
                + PLAYER_TYPE + " = " + playerType;
        return queryBySql(querySql);
    }

    private synchronized List<Record> query(int playerType, String mime, String codec, int profile, int level, int width, int height) {
        String querySql = "select * from " + PLAY_RECORD_TABLE_NAME + " WHERE "
                + PLAYER_TYPE + " = " + playerType + " AND "
                + MIME + " = '" + mime + "' AND "
                + CODEC + " = '" + codec + "' AND "
                + PROFILE + " = " + profile + " AND "
                + LEVEL + " = " + level + " AND "
                + WIDTH + " = " + width + " AND "
                + HEIGHT + " = " + height;
        return queryBySql(querySql);
    }

    @SuppressLint("Range")
    private synchronized List<Record> queryBySql(String querySql) {
        List<Record> list = new ArrayList<>();
        try (SQLiteDatabase db = DatabaseHelper.getInstance().getWritableDatabase(); Cursor cursor = db.rawQuery(querySql, null)) {
            while (cursor.moveToNext()) {
                int id = cursor.getInt(cursor.getColumnIndex(ID));
                int playerType = cursor.getInt(cursor.getColumnIndex(PLAYER_TYPE));
                String mime = cursor.getString(cursor.getColumnIndex(MIME));
                String codec = cursor.getString(cursor.getColumnIndex(CODEC));
                int profile = cursor.getInt(cursor.getColumnIndex(PROFILE));
                int level = cursor.getInt(cursor.getColumnIndex(LEVEL));
                int width = cursor.getInt(cursor.getColumnIndex(WIDTH));
                int height = cursor.getInt(cursor.getColumnIndex(HEIGHT));
                int successCount = cursor.getInt(cursor.getColumnIndex(SUCCESS_COUNT));
                int failCount = cursor.getInt(cursor.getColumnIndex(FAIL_COUNT));
                list.add(new Record(id, playerType, mime, codec, profile, level, width, height, successCount, failCount));
            }
        } catch (DatabaseHelper.InvalidContextException e) {
            Log.i(TAG, "queryBySql error: " + e.getMessage());
        } catch (Exception e) {
            e.printStackTrace();
        }
        return list;
    }
}
