package com.metric.client;

import com.metric.common.CryptTools;
import com.metric.common.escape.Escaper;
import com.metric.common.escape.Escapers;
import com.metric.client.utils.ExecutorThreadPool;

import java.io.BufferedWriter;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.UnsupportedEncodingException;
import java.math.BigDecimal;
import java.net.HttpURLConnection;
import java.net.URI;
import java.net.URL;
import java.net.URLEncoder;
import java.text.NumberFormat;
import java.util.*;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;

/**
 */
public class MetricAgent {
    private static class MetricEvent {
        public String name;
        public String tags;
        public String fields;
    }

    private static final Escaper FIELD_ESCAPER = Escapers.builder().addEscape('"', "\\\"").build();
    private static final Escaper KEY_ESCAPER = Escapers.builder()
            .addEscape(' ', "\\ ")
            .addEscape(',', "\\,")
            .addEscape('=', "\\=")
            .build();
    private static final int MAX_FRACTION_DIGITS = 340;

    private static ExecutorThreadPool threadPool;
    private static String appId;
    private static String appKey;
    private static String serverUrl;
    private static boolean debug = false;
    private static BlockingQueue<MetricEvent> eventQueue;
    private static int eventQueueLimit;
    private static int postBatchSize;

    private static boolean isInit() {
        return threadPool != null;
    }

    public static void setDebug(boolean d) {
        debug = d;
    }

    public static void init(String appId, String appKey, String serverUrl) {
        if (isInit()) {
            return;
        }
        MetricConfig metricConfig = new MetricConfig();
        metricConfig.setAppId(appId);
        metricConfig.setAppKey(appKey);
        metricConfig.setServerUrl(serverUrl);
        init(metricConfig);
    }

    public static void init(MetricConfig metricConfig) {
        if (isInit()) {
            return;
        }
        MetricAgent.appId = metricConfig.getAppId();
        MetricAgent.appKey = metricConfig.getAppKey();
        String appendServerUrl = metricConfig.getServerUrl();
        if (!appendServerUrl.endsWith("/")) {
            appendServerUrl += "/";
        }
        MetricAgent.serverUrl = appendServerUrl;
        eventQueueLimit = metricConfig.getBufferQueueSize();
        eventQueue = new LinkedBlockingQueue<>(eventQueueLimit);
        postBatchSize = metricConfig.getPostBatchSize();

        threadPool = new ExecutorThreadPool();
        threadPool.setInitialSize(metricConfig.getThreadPoolSize() > 1 ? metricConfig.getThreadPoolSize() : 1);
        threadPool.start();
        threadPool.background(new PostRun());
    }

    public static void postEvent(String eventName, Map<String, String> tagMap, Map<String, Object> fieldMap) {
        try {
            NumberFormat numberFormat = NumberFormat.getInstance(Locale.ENGLISH);
            numberFormat.setMaximumFractionDigits(MAX_FRACTION_DIGITS);
            numberFormat.setGroupingUsed(false);
            numberFormat.setMinimumFractionDigits(1);

            String tags = "";
            for (Map.Entry<String, String> tagEntry : tagMap.entrySet()) {
                //忽略空tag
                if(tagEntry.getValue() == null || tagEntry.getValue().trim().equals("")){
                    continue;
                }
                String tagValue = tagEntry.getValue();
                tags += KEY_ESCAPER.escape(tagEntry.getKey()) + "=" + KEY_ESCAPER.escape(tagValue);
                tags += ",";
            }
            //remove last ","
            tags = tags.substring(0, tags.length() - 1);
            String fields = "";
            for (Map.Entry<String, Object> fieldEntry : fieldMap.entrySet()) {
                fields += KEY_ESCAPER.escape(fieldEntry.getKey()) + "=";
                Object value = fieldEntry.getValue();
                if (value == null) {
                    if (debug) {
                        System.out.println("MetricAgent: Ignore empty field value of key " + fieldEntry.getKey());
                    }
                    continue;
                }

                if (value instanceof String) {
                    String stringValue = (String) value;
                    fields += "\"" + FIELD_ESCAPER.escape(stringValue) + ("\"");
                } else if (value instanceof Number) {
                    if (value instanceof Double || value instanceof Float || value instanceof BigDecimal) {
                        fields += numberFormat.format(value);
                    } else {
                        fields += value + "i";
                    }
                } else {
                    fields += value;
                }
                fields += ",";
            }
            //remove last ","
            fields = fields.substring(0, fields.length() - 1);
            postEvent(eventName, tags, fields);
        } catch (Exception e) {
            if (debug) {
                System.out.println("MetricAgent: Error postEvent: " + e.getMessage());
                e.printStackTrace();
            }
        }
    }

    private static void postEvent(String eventName, String tags, String fields) {
        if (!isInit()) {
            System.out.println("MetricAgent is not initialized");
            return;
        }
        int eventQueueSize = eventQueue.size();
        if (eventQueueSize >= eventQueueLimit) {
            System.out.println(String.format("MetricAgent: Reached buffer limit[%s], dropped new event[%s]", eventQueueSize, eventName));
            return;
        }
        MetricEvent metricEvent = new MetricEvent();
        metricEvent.name = eventName;
        metricEvent.tags = tags;
        metricEvent.fields = fields;
        try {
            boolean offered = eventQueue.offer(metricEvent, 0, TimeUnit.MILLISECONDS);
            if (!offered) {
                System.out.println(String.format("MetricAgent: Failed push event to buffer, dropped new event[%s]", eventName));
            }
        } catch (Exception e) {
            if (debug) {
                System.out.println("MetricAgent: Error in push event to buffer: " + e.getMessage());
            }
        }
        //threadPool.background(new PostRun(eventName, tags, fields));
        if (debug) {
            System.out.println(String.format("MetricAgent: postEvent scheduled: eventName[%s] tags[%s] fields[%s]", eventName, tags, fields));
        }
    }

    private static class PostRun implements Runnable {
        @Override
        public void run() {
            while (true) {
                //build events param
                List<MetricEvent> eventList = new ArrayList<>();
                try {
                    //take to wait the event
                    MetricEvent srcMetricEvent = eventQueue.take();
                    eventList.add(srcMetricEvent);
                    
                    while ((srcMetricEvent = eventQueue.poll(1_000, TimeUnit.MILLISECONDS)) != null) {
                        eventList.add(srcMetricEvent);
                        //enough
                        if (eventList.size() >= postBatchSize) {
                            break;
                        }
                    }
                    if (!eventList.isEmpty()) {
                        String eventSplitStr = "\n";
                        String fieldSplitStr = "#$#";
                        String eventsBody = "";
                        for (MetricEvent metricEvent : eventList) {
                            eventsBody += metricEvent.name + fieldSplitStr + metricEvent.tags + fieldSplitStr + metricEvent.fields + eventSplitStr;
                        }
                        //remove last "\n"
                        eventsBody = eventsBody.substring(0, eventsBody.length() - 1);
                        postEvents(eventsBody);
                        if (debug) {
                            System.out.println("MetricAgent: posted event count: " + eventList.size());
                        }
                    }
                } catch (Throwable e) {
                    if (debug) {
                        System.out.println("MetricAgent: Error in PostRun: " + e.getMessage());
                        e.printStackTrace();
                    }
                }
            }
        }
    }

    private static void postEvents(String eventsBody) throws Exception {
        //post to server
        String postUrl = serverUrl + "api/metric/batchpostevent/v2";
        URL url = URI.create(postUrl).toURL();
        HttpURLConnection urlConnection = (HttpURLConnection) url.openConnection();
        urlConnection.setRequestMethod("POST");
        urlConnection.setConnectTimeout(10_000);
        urlConnection.setReadTimeout(5_1000);
        urlConnection.setRequestProperty("Content-Type", "application/x-www-form-urlencoded");
        urlConnection.setRequestProperty("charset", "utf-8");
        urlConnection.setRequestProperty("User-Agent", "MetricAgent 1.0");
        urlConnection.setUseCaches(false);
        urlConnection.setDoOutput(true);
        urlConnection.setDoInput(true);

        Map<String, String> postDataParams = new HashMap<>();
        postDataParams.put("app_id", appId);
        postDataParams.put("app_key", appKey);
        postDataParams.put("events", eventsBody);
        
        Map<String, String> postMsg = new HashMap<>();
        postMsg.put("v", "1");
        postMsg.put("m", CryptTools.encrypt(getPostDataString(postDataParams)));
        
        OutputStream os = urlConnection.getOutputStream();
        BufferedWriter writer = new BufferedWriter(new OutputStreamWriter(os, "UTF-8"));
        writer.write(getPostDataString(postMsg));

        writer.flush();
        writer.close();
        os.close();
        int responseCode = urlConnection.getResponseCode();

        if (responseCode == HttpURLConnection.HTTP_OK) {
            if (debug) {
                System.out.println(String.format("postEvent OK: %s", eventsBody));
            }
        } else {
            if (debug) {
                System.out.println(String.format("postEvent server failed[%s]: %s", responseCode, eventsBody));
            }
        }
    }

    private static String getPostDataString(Map<String, String> params) throws UnsupportedEncodingException {
        StringBuilder result = new StringBuilder();
        boolean first = true;
        for (Map.Entry<String, String> entry : params.entrySet()) {
            if (first)
                first = false;
            else
                result.append("&");

            result.append(URLEncoder.encode(entry.getKey(), "UTF-8"));
            result.append("=");
            result.append(URLEncoder.encode(entry.getValue(), "UTF-8"));
        }

        return result.toString();
    }
}
