package com.cv.media.lib.api;

import com.cv.media.lib.common_utils.callback.IAction;
import com.cv.media.lib.common_utils.obfuscation.KeepClassMember;
import com.cv.media.lib.common_utils.pool.ObjectPool;
import com.cv.media.lib.common_utils.pool.ObjectPoolItem;
import com.cv.media.lib.common_utils.utils.GsonUtil;

import java.io.IOException;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.Proxy;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import okhttp3.Call;
import okhttp3.Connection;
import okhttp3.Handshake;
import okhttp3.Protocol;
import okhttp3.Request;
import okhttp3.Response;

/**
 * 描述网络的一个API请求
 *
 * @author Damon
 */
public class NW_ApiRequestEvent extends NW_Base implements ObjectPoolItem, INetWorkEvent, KeepClassMember {
    private transient int tag;
    private String path;
    private String method;
    private List<NW_HostNode> hostRequests = new ArrayList<>();
    private transient NW_HostNode currentRequest;
    private String errorMsg;
    private long startTs;
    private long interceptorRequestStartTs;
    private long interceptorRequestDoneTs;

    private long interceptorResponseStartTs;
    private long interceptorResponseDoneTs;
    private long endTs;
    private volatile transient boolean isDone;
    private transient ArrayList<NetWorkEvent> events = new ArrayList<>();
    private transient String debugTip = "";

    public NW_ApiRequestEvent() {
        reset();
    }

    void reset() {
        tag = (int) INVALID;
        isDone = false;
        interceptorRequestStartTs = INVALID;
        interceptorRequestDoneTs = INVALID;
        interceptorResponseStartTs = INVALID;
        interceptorResponseDoneTs = INVALID;
        startTs = INVALID;
        endTs = INVALID;
        errorMsg = null;
        path = null;
        method = null;
        hostRequests.clear();
        currentRequest = null;
        events.clear();
    }

    public int getTag() {
        return tag;
    }

    public void callCreated(Request request) {
        tag = request.hashCode();
        path = request.url().encodedPath();
        method = request.method();
    }

    public void interceptorRequestChainStart(Request request) {
        interceptorRequestStartTs = cts();
        path = request.url().encodedPath();
        method = request.method();
    }

    /**
     * 异常结束点
     * {@link #callFailed(Call, IOException)}
     * {@link #interceptorChainFail(Call, Exception)}
     * 两者互斥
     */
    public void interceptorChainFail(Call call, Exception e) {
        debugTip = e.getMessage();
        errorMsg = e.getMessage();
        isDone = true;
        NetWorksMonitor.getInstance().pushRequestInfo(call);
    }

    public void interceptorRequestChainDone(Request request) {
        interceptorRequestDoneTs = cts();
        currentRequest = ObjectPool.obtain(NW_HostNode.class);
        hostRequests.add(currentRequest);
        currentRequest.setHost(request.url().scheme() + "://" + request.url().host());
    }

    public void interceptorResponseChainStart(Call call) {
        interceptorResponseStartTs = cts();
    }

    /**
     * 正常结束点
     * {@link #callEnd(Call)}
     * {@link #interceptorResponseChainDone(Call)} (Call)}
     * 两者调用顺序 在各种场景调用顺序不一
     */
    public void interceptorResponseChainDone(Call call) {
        interceptorResponseDoneTs = cts();
        if (endTs != INVALID) {
            isDone = true;
            NetWorksMonitor.getInstance().pushRequestInfo(call);
        }
    }

    @Override
    public void callStart(Call call) {
        startTs = cts();
    }

    public void dnsStart(Call call, String domainName) {
        if (currentRequest != null) currentRequest.dnsStart(call, domainName);
    }

    public void dnsEnd(Call call, String domainName, List<InetAddress> inetAddressList) {
        if (currentRequest != null) currentRequest.dnsEnd(call, domainName, inetAddressList);
    }

    public void connectStart(Call call, InetSocketAddress inetSocketAddress, Proxy proxy) {
        if (currentRequest != null) currentRequest.connectStart(call, inetSocketAddress, proxy);
    }

    public void secureConnectStart(Call call) {
        if (currentRequest != null) currentRequest.secureConnectStart(call);
    }

    public void secureConnectEnd(Call call, Handshake handshake) {
        if (currentRequest != null) currentRequest.secureConnectEnd(call, handshake);
    }

    public void connectEnd(Call call, InetSocketAddress inetSocketAddress, Proxy proxy,
                           Protocol protocol) {
        if (currentRequest != null)
            currentRequest.connectEnd(call, inetSocketAddress, proxy, protocol);
    }

    public void connectFailed(Call call, InetSocketAddress inetSocketAddress, Proxy proxy,
                              Protocol protocol, IOException ioe) {
        if (currentRequest != null)
            currentRequest.connectFailed(call, inetSocketAddress, proxy, protocol, ioe);
    }


    public void connectionAcquired(Call call, Connection connection) {
        if (currentRequest != null) currentRequest.connectionAcquired(call, connection);
    }


    public void connectionReleased(Call call, Connection connection) {
        if (currentRequest != null) currentRequest.connectionReleased(call, connection);
    }


    public void requestHeadersStart(Call call) {
        if (currentRequest != null) currentRequest.requestHeadersStart(call);
    }


    public void requestHeadersEnd(Call call, Request request) {
        if (currentRequest != null) currentRequest.requestHeadersEnd(call, request);
    }


    public void requestBodyStart(Call call) {
        if (currentRequest != null) currentRequest.requestBodyStart(call);
    }


    public void requestBodyEnd(Call call, long byteCount) {
        if (currentRequest != null) currentRequest.requestBodyEnd(call, byteCount);
    }


    public void responseHeadersStart(Call call) {
        if (currentRequest != null) currentRequest.responseHeadersStart(call);
    }


    public void responseHeadersEnd(Call call, Response response) {
        if (currentRequest != null) currentRequest.responseHeadersEnd(call, response);
    }


    public void responseBodyStart(Call call) {
        if (currentRequest != null) currentRequest.responseBodyStart(call);
    }


    public void responseBodyEnd(Call call, long byteCount) {
        if (currentRequest != null) currentRequest.responseBodyEnd(call, byteCount);
    }

    /**
     * 正常结束点
     * {@link #callEnd(Call)}
     * {@link #interceptorResponseChainDone(Call)} (Call)}
     * 两者调用顺序 在各种场景调用顺序不一
     */
    public void callEnd(Call call) {
        if (currentRequest != null) currentRequest.callEnd(call);
        if (interceptorResponseStartTs != INVALID && interceptorResponseDoneTs == INVALID) {
            endTs = cts();
        } else if (interceptorResponseDoneTs != INVALID) {
            endTs = cts();
            if (isDone) throw new RuntimeException("done before: " + debugTip);
            isDone = true;
            NetWorksMonitor.getInstance().pushRequestInfo(call);
        }
    }

    /**
     * 异常结束点
     * {@link #callFailed(Call, IOException)}
     * {@link #interceptorChainFail(Call, Exception)}
     * 两者互斥
     */
    public void callFailed(Call call, IOException ioe) {
        if (isDone) throw new RuntimeException("done before: " + debugTip);
        if (currentRequest != null) currentRequest.callFailed(call, ioe);
        errorMsg = ioe.getMessage();
        debugTip = ioe.getMessage();
        isDone = true;
        NetWorksMonitor.getInstance().pushRequestInfo(call);
    }


    @Override
    public void recycle() {
        for (NW_HostNode request : hostRequests) {
            request.recycle();
        }

        for (NetWorkEvent event : events) {
            event.setDesCb(null);
        }
        super.recycle();
    }

    @Override
    public long getTotalCost() {
        return Math.max(endTs, interceptorRequestDoneTs) - startTs;
    }

    @Override
    public String getDescription() {
        return isDone ? GsonUtil.getGson().toJson(this) : null;
    }

    @Override
    public String getPath() {
        return path;
    }

    @Override
    public String getMethod() {
        return method;
    }


    NetWorkEvent transformToEvent() {
        NetWorkEvent event = new NetWorkEvent(path, method, tag, getTotalCost(), this::getDescription);
        events.add(event);
        return event;
    }
}
