package com.cv.media.lib.api;

import com.cv.media.lib.common_utils.async.ThreadUtils;
import com.cv.media.lib.common_utils.async.ThreadsBox;
import com.cv.media.lib.common_utils.pool.ObjectPool;
import com.cv.media.lib.common_utils.utils.Singleton;

import java.io.IOException;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.Proxy;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;

import okhttp3.Call;
import okhttp3.Connection;
import okhttp3.EventListener;
import okhttp3.Handshake;
import okhttp3.Protocol;
import okhttp3.Request;
import okhttp3.Response;

/**
 * 网络请求监控
 * 职责: 监控每一个请求的详情
 *
 * @author Damon
 */
class NetWorksMonitor extends EventListener {
    private final ConcurrentHashMap<Call, NW_ApiRequestEvent> mRequestMaps = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<Integer, NW_ApiRequestEvent> mCreatedRequestMaps = new ConcurrentHashMap<>();
    private final CopyOnWriteArrayList<INetWorkEventListener> mListeners = new CopyOnWriteArrayList<>();
    private final NW_ApiRequestEvent mRequestStub = new NW_ApiRequestEventStub();

    private final static Singleton<NetWorksMonitor> monitorSingleton = new Singleton<NetWorksMonitor>() {
        @Override
        protected NetWorksMonitor create() {
            return new NetWorksMonitor();
        }
    };

    public static NetWorksMonitor getInstance() {
        return monitorSingleton.get();
    }

    private synchronized NW_ApiRequestEvent callToRequest(Call call) {
        NW_ApiRequestEvent request;
        if ((request = mRequestMaps.get(call)) == null) request = mRequestStub;
        return request;
    }

    public void callCreate(Request request) {
        NW_ApiRequestEvent netWorkRequest = ObjectPool.obtain(NW_ApiRequestEvent.class);
        netWorkRequest.callCreated(request);
        mCreatedRequestMaps.put(netWorkRequest.getTag(), netWorkRequest);
        for (INetWorkEventListener eventListener : mListeners) {
            eventListener.onEventCreated(netWorkRequest.transformToEvent());
        }
    }

    @Override
    public void callStart(Call call) {
        NW_ApiRequestEvent request = mCreatedRequestMaps.remove(call.request().hashCode());
        if (request != null) {
            mRequestMaps.put(call, request);
            request.callStart(call);
        }
    }

    public void requestIn(Request request, Call call) {
        callToRequest(call).interceptorRequestChainStart(request);
    }

    public void interceptorException(Call call, Exception e) {
        callToRequest(call).interceptorChainFail(call, e);
    }

    public void requestOut(Request request, Call call) {
        callToRequest(call).interceptorRequestChainDone(request);
    }

    public void responseIn(Response response, Call call) {
        callToRequest(call).interceptorResponseChainStart(call);
    }

    public void responseOut(Response response, Call call) {
        callToRequest(call).interceptorResponseChainDone(call);
    }


    public void dnsStart(Call call, String domainName) {
        callToRequest(call).dnsStart(call, domainName);
    }

    public void dnsEnd(Call call, String domainName, List<InetAddress> inetAddressList) {
        callToRequest(call).dnsEnd(call, domainName, inetAddressList);
    }

    public void connectStart(Call call, InetSocketAddress inetSocketAddress, Proxy proxy) {
        callToRequest(call).connectStart(call, inetSocketAddress, proxy);
    }

    public void secureConnectStart(Call call) {
        callToRequest(call).secureConnectStart(call);
    }

    public void secureConnectEnd(Call call, Handshake handshake) {
        callToRequest(call).secureConnectEnd(call, handshake);
    }

    public void connectEnd(Call call, InetSocketAddress inetSocketAddress, Proxy proxy,
                           Protocol protocol) {
        callToRequest(call).connectEnd(call, inetSocketAddress, proxy, protocol);
    }

    public void connectFailed(Call call, InetSocketAddress inetSocketAddress, Proxy proxy,
                              Protocol protocol, IOException ioe) {
        callToRequest(call).connectFailed(call, inetSocketAddress, proxy, protocol, ioe);
    }


    public void connectionAcquired(Call call, Connection connection) {
        callToRequest(call).connectionAcquired(call, connection);
    }


    public void connectionReleased(Call call, Connection connection) {
        callToRequest(call).connectionReleased(call, connection);
    }


    public void requestHeadersStart(Call call) {
        callToRequest(call).requestHeadersStart(call);
    }


    public void requestHeadersEnd(Call call, Request request) {
        callToRequest(call).requestHeadersEnd(call, request);
    }


    public void requestBodyStart(Call call) {
        callToRequest(call).requestBodyStart(call);
    }


    public void requestBodyEnd(Call call, long byteCount) {
        callToRequest(call).requestBodyEnd(call, byteCount);
    }


    public void responseHeadersStart(Call call) {
        callToRequest(call).responseHeadersStart(call);
    }


    public void responseHeadersEnd(Call call, Response response) {
        callToRequest(call).responseHeadersEnd(call, response);
    }


    public void responseBodyStart(Call call) {
        callToRequest(call).responseBodyStart(call);
    }


    public void responseBodyEnd(Call call, long byteCount) {
        callToRequest(call).responseBodyEnd(call, byteCount);
    }


    public void callEnd(Call call) {
        callToRequest(call).callEnd(call);
    }


    public void callFailed(Call call, IOException ioe) {
        callToRequest(call).callFailed(call, ioe);
    }

    synchronized void addListener(INetWorkEventListener listener) {
        if (listener != null) {
            mListeners.add(listener);
        }
    }

    synchronized void removeListener(INetWorkEventListener listener) {
        if (listener != null) {
            mListeners.remove(listener);
        }
    }

    void pushRequestInfo(Call call) {
        if (mListeners.isEmpty()) {
            NW_ApiRequestEvent apiRequestEvent;
            synchronized (this) {
                 apiRequestEvent = callToRequest(call);
                if (apiRequestEvent != mRequestStub) {
                    mRequestMaps.remove(call);
                }else
                    return;
            }
            apiRequestEvent.recycle();
        } else {
            NW_ApiRequestEvent apiRequestEvent;
            synchronized (this) {
                apiRequestEvent = callToRequest(call);
                if (apiRequestEvent != mRequestStub)
                    mRequestMaps.remove(call);
                else
                    return;
            }
            ThreadUtils.execute(() -> {
                List<INetWorkEventListener> lis;
                synchronized (this) {
                    lis = new ArrayList<>(mListeners);
                }
                for (INetWorkEventListener eventListener : lis) {
                    eventListener.onEventDone(apiRequestEvent.transformToEvent());
                }
                apiRequestEvent.recycle();
            });
        }

    }
}
