package com.cv.media.lib.api;

import com.cv.media.lib.api.okhttp.BaseCryptoInterceptor;
import com.cv.media.lib.api.pct.PctInitializer;
import com.cv.media.lib.api.pct.PctInterceptorWrapper;
import com.pct.ext.okhttp.PctInterceptor;

import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManager;
import javax.net.ssl.X509TrustManager;

import me.ew.rdns.lib.dns.EnhancedDns;
import okhttp3.ConnectionPool;
import okhttp3.Interceptor;
import okhttp3.OkHttpClient;
import retrofit2.Converter;
import retrofit2.Retrofit;
import retrofit2.converter.gson.GsonConverterFactory;

/**
 * 网络请求层 封装类
 * 只开放 createAPI接口
 *
 * @Author Damon
 */
public class NetworkService {
    public final static String KEY_API_CONNECT_TIMEOUT = "KEY_API_CONNECT_TIMEOUT"; //Option TimeOut
    public final static String KEY_API_READ_TIMEOUT = "KEY_API_READ_TIMEOUT"; //Option  TimeOut
    public final static String KEY_OKHTTP_INTERCEPTORS = "KEY_OKHTTP_INTERCEPTORS"; //Option  拦截器 可选
    public final static String KEY_OKHTTP_CONVERT_FACTORY = "KEY_OKHTTP_CONVERT_FACTORY"; //Option  ConverterFactory

    public final static int DEFAULT_API_CONNECT_TIMEOUT = 20;
    public final static int DEFAULT_API_READ_TIMEOUT = 20;

    public final static String STUB_BASEURL = "http://stub";

    public static <T> T createAPI(Class<T> api, Map<String, Object> configuration) {
        Retrofit.Builder builder = new Retrofit.Builder();
        builder.baseUrl(STUB_BASEURL);
        OkHttpClient client = createOkHttpClient(configuration);
        builder.client(client);

        if (configuration.containsKey(KEY_OKHTTP_CONVERT_FACTORY)) {
            Converter.Factory convertFactory = (Converter.Factory) configuration.get(KEY_OKHTTP_CONVERT_FACTORY);
            if (convertFactory != null) builder.addConverterFactory(convertFactory);
        } else {
            builder.addConverterFactory(GsonConverterFactory.create());
        }
        builder.addConverterFactory(new NetWorkApiConverterFactory());
        builder.addCallAdapterFactory(new NetWorkApiCallAdapterFactory());
        builder.addCallAdapterFactory(new NetWorkMonitorCallAdapterFactory());

        T apiImpl = builder.build().create(api);
        NetWorkApiWrapper<T> wrapper = new NetWorkApiWrapper<>(api, apiImpl);
        wrapper.setInterceptors((List<Interceptor>) configuration.get(KEY_OKHTTP_INTERCEPTORS));
        return wrapper.getWrap();
    }

    public static void addNetWorkEventListener(INetWorkEventListener listener) {
        NetWorksMonitor.getInstance().addListener(listener);
    }

    public static void removeNetWorkEventListener(INetWorkEventListener listener) {
        NetWorksMonitor.getInstance().removeListener(listener);
    }

    private final static ConnectionPool connectionPool = new ConnectionPool(8, 5, TimeUnit.MINUTES);

    //构造一个OkHttpClient供retrofit使用
    private static OkHttpClient createOkHttpClient(Map<String, Object> configuration) {
        OkHttpClient.Builder builder = new OkHttpClient.Builder();
        builder.connectionPool(connectionPool);
        int connectTimeOut = DEFAULT_API_CONNECT_TIMEOUT;
        int readTimeOut = DEFAULT_API_READ_TIMEOUT;
        //超时
        if (configuration.containsKey(KEY_API_CONNECT_TIMEOUT)) {
            connectTimeOut = (int) configuration.get(KEY_API_CONNECT_TIMEOUT);
        }
        if (configuration.containsKey(KEY_API_READ_TIMEOUT)) {
            readTimeOut = (int) configuration.get(KEY_API_READ_TIMEOUT);
        }

        //拦截器
        if (configuration.containsKey(KEY_OKHTTP_INTERCEPTORS)) {
            ArrayList<Interceptor> interceptors = new ArrayList<>((Collection<Interceptor>) configuration.get(KEY_OKHTTP_INTERCEPTORS));
            Collections.sort(interceptors, (o1, o2) -> {
                if (o1 instanceof BaseHostSwitchInterceptor) {
                    return 1;
                } else if (o2 instanceof BaseHostSwitchInterceptor) {
                    return -1;
                } else if (o1 instanceof BaseCryptoInterceptor) {
                    return 1;
                } else if (o2 instanceof BaseCryptoInterceptor) {
                    return -1;
                }
                return 0;
            });
            if (!(interceptors.get(interceptors.size() - 1) instanceof BaseHostSwitchInterceptor)) {
                throw new RuntimeException("NetWorkService 最后一个Interceptor必须是HostSwitch");
            }

            interceptors.add(0, new InterceptorHead());
            interceptors.add(new InterceptorTail());
            try {
                //pct init may error
                interceptors.add(new PctInterceptorWrapper());
            }catch (Exception e){}

            builder.interceptors().addAll(interceptors);
        } else {
            throw new RuntimeException("NetWorkService 至少配置一个HostSwitchInterceptor");
        }

        //Event listener
        builder.eventListener(NetWorksMonitor.getInstance());

        SSLContext sslContext = getSslContext();
        if (sslContext != null) {
            // 设置忽略ssl证书
            builder.sslSocketFactory(sslContext.getSocketFactory(), xtm);
        }

        builder.hostnameVerifier((hostname, session) -> true);

        builder.connectTimeout(connectTimeOut, TimeUnit.SECONDS);
        builder.readTimeout(readTimeOut, TimeUnit.SECONDS);
        builder.dns(new EnhancedDns());
        return builder.build();
    }


    private static X509TrustManager xtm = new X509TrustManager() {
        @Override
        public void checkClientTrusted(X509Certificate[] chain, String authType) {
        }

        @Override
        public void checkServerTrusted(X509Certificate[] chain, String authType) {
        }

        @Override
        public X509Certificate[] getAcceptedIssuers() {
            //return null; okhttp 3.0
            return new X509Certificate[]{};
        }
    };

    private static SSLContext getSslContext() {
        SSLContext sslContext = null;
        try {
            sslContext = SSLContext.getInstance("SSL");
            sslContext.init(null, new TrustManager[]{xtm}, new SecureRandom());

        } catch (NoSuchAlgorithmException | KeyManagementException e) {
            e.printStackTrace();
        }
        return sslContext;
    }
}
