package com.cv.media.lib.common_utils.appinit;

import android.os.Handler;
import android.os.HandlerThread;
import android.os.Message;
import android.os.SystemClock;
import android.util.Log;

import androidx.annotation.NonNull;

import com.cv.media.lib.common_utils.utils.Singleton;

import org.apache.commons.lang3.concurrent.ConcurrentUtils;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;

import io.reactivex.Observable;
import io.reactivex.disposables.Disposable;
import io.reactivex.functions.Consumer;

/**
 * 实施 整个 App运行 必须依赖 的 基础功能 初始化 的 类
 * 提供一次性的Normal Tasks的执行流程 和 多次性的Situation Tasks的执行流程
 * 整体工作环境为 线程
 * 应用启动应该如下:
 * -----初始化流程 只做初始化, 不做任何业务请求------>
 * -------开始开展业务------------->
 *
 * @author Damon
 */
public class Initialization {
    private final LinkedHashMap<String, IInitializationTask> tasks = new LinkedHashMap<>();

    private Set<IInitializationSituationTask> readyTodoSituations = Collections.synchronizedSet(new HashSet<>());

    private final Handler handler;
    private final Handler threadHandler;
    private final ArrayList<InitializationListener> listeners = new ArrayList<>();
    private static final int MSG_RunTasks = 0x88a;
    private static final int MSG_DISPATCH_NORMAL_TASK_DONE = 0x89a;
    private final AtomicInteger normalTaskTotalCount = new AtomicInteger(0);
    private final AtomicInteger normalTaskDoneCount = new AtomicInteger(0);
    private final AtomicBoolean isNormalTaskProcessed = new AtomicBoolean(false);
    private boolean isInTestMode = false;
    private final AtomicBoolean isNetWorkReady = new AtomicBoolean(false);
    private final AtomicBoolean isAccountReady = new AtomicBoolean(false);

    private static final Singleton<Initialization> instance = new Singleton<Initialization>() {
        @Override
        protected Initialization create() {
            return new Initialization();
        }
    };

    public static Initialization getInstance() {
        return instance.get();
    }

    public static void reset() {
        instance.clear();
    }


    public void openTestMode() {
        isInTestMode = true;
    }

    /**
     * 添加 任务
     *
     * @param task 任务
     */
    public void addTask(IInitializationTask task) {
        if (((InitTask) task).getTag() == null) {
            if (isInTestMode) {
                throw new RuntimeException("任务 一定要指定 初始化类");
            } else {
                return;
            }
        }

        threadHandler.post(() -> {
            if (tasks.put(((InitTask) task).getTag(), task) == null) {
                if (task instanceof IInitializationSituationTask) {
                    logs("添加Situation任务 初始化类:" + ((InitTask) task).getTag());
                } else if (task instanceof IInitializationNormalTask) {
                    logs("添加Normal任务 初始化类:" + ((InitTask) task).getTag());
                    normalTaskTotalCount.incrementAndGet();
                }
            }
        });
    }

    /**
     * 开始执行 任务
     *
     * @param taskClass 任务类型 InitTask的子类
     */
    public <T extends InitTask> void startProceed(Class<? extends IInitializationTask> taskClass, String... reasons) {
        String reason = (reasons != null && reasons.length > 0) ? reasons[0] : "";
        if (IInitializationNormalTask.class.isAssignableFrom(taskClass)) {
            if (!isNormalTaskProcessed.get()) {
                isNormalTaskProcessed.set(true);
            } else {
                logs("重复执行Normal Task!!");
                return;
            }
        }
        logs("请求执行任务 任务类型:" + taskClass.getSimpleName() + "\treason: " + reason);
        //条件任务
        if (IInitializationSituationTask.class.isAssignableFrom(taskClass)) {
            if (normalTaskTotalCount.get() == 0 || normalTaskTotalCount.get() > normalTaskDoneCount.get()) {
                //正常任务未执行完 条件任务 下一轮
                logs("正常任务 未全部执行成功,  条件任务轮空 下一轮\treason: " + reason);
                threadHandler.postDelayed(() -> startProceed(taskClass, reasons), 500);
                return;
            } else if (!readyTodoSituations.isEmpty()) {
                //前面有一波条件任务 还没执行完 这次的条件任务申请 延续到下一轮
                logs("上一批条件任务 未全部执行成功,  条件任务轮空 下一轮\treason: " + reason);
                threadHandler.postDelayed(() -> startProceed(taskClass, reasons), 1000);
                return;
            }
        }
        threadHandler.post(() -> startProceedInternal(taskClass, reason));
    }

    Disposable networkDisposable;

    public void attachNetWorkObservable(Observable<Boolean> observable) {
        networkDisposable = observable.subscribe(new Consumer<Boolean>() {
            @Override
            public void accept(Boolean o) throws Exception {
                isNetWorkReady.set(o);
                if (o)
                    threadHandler.post(() -> startProceed(IInitializationSituationTask.class, "network event"));
            }
        });
    }

    Disposable accountDisposable;

    public void attachAccountObservable(Observable<Boolean> observable) {
        accountDisposable = observable.subscribe(new Consumer<Boolean>() {
            @Override
            public void accept(Boolean o) throws Exception {
                isAccountReady.set(o);
                if (o)
                    threadHandler.post(() -> startProceed(IInitializationSituationTask.class, "account event"));
            }
        });
    }

    public void addListener(InitializationListener listener) {
        if (listener == null) return;
        handler.post(() -> {
            if (!listeners.contains(listener)) listeners.add(listener);
            if (normalTaskTotalCount.get() == normalTaskDoneCount.get() && !handler.hasMessages(MSG_DISPATCH_NORMAL_TASK_DONE)) {
                listener.onAllNormalTaskFinish();
            }
        });
    }

    public void removeListener(InitializationListener listener) {
        if (listener == null) return;
        handler.post(new Runnable() {
            @Override
            public void run() {
                listeners.remove(listener);
            }
        });
    }

    private Initialization() {
        handler = new Handler(new RunTaskCallback());
        HandlerThread thread = new HandlerThread(Initialization.class.getSimpleName() + "I");
        thread.start();
        threadHandler = new Handler(thread.getLooper(), new RunTaskCallback());
    }

    private <T extends InitTask> void startProceedInternal(Class<? extends IInitializationTask> taskClass, String reason) {
        if (tasks.isEmpty()) {
            logs("提前结束 任务执行 因为 任务列表为空\treason:" + reason);
            return;
        }
        logs("开始 执行任务 任务类型:" + taskClass.getSimpleName() + "\treason: " + reason);
        Set<Map.Entry<String, IInitializationTask>> entries = tasks.entrySet();
        HashMap<String, IInitializationTask> needProceed = new HashMap<>();
        boolean isSituation = IInitializationSituationTask.class.isAssignableFrom(taskClass);
        for (Map.Entry<String, IInitializationTask> entry : entries) {
            if (taskClass.isAssignableFrom(entry.getValue().getClass())) {
                ((InitTask) entry.getValue()).setLevel(0);
                needProceed.put(entry.getKey(), entry.getValue());
                if (isSituation)
                    readyTodoSituations.add((IInitializationSituationTask) entry.getValue());
            }
        }
        logs("符合" + taskClass.getSimpleName() + "执行条件的 任务数量:" + needProceed.size());
        logs("开始 进行排序任务表 任务类型:" + taskClass.getSimpleName());
        List<InitTask> needProceedTasks = InitTaskSorter.sort(needProceed);
        logs("结束 进行排序任务表 任务类型:" + taskClass.getSimpleName());
        for (InitTask task : needProceedTasks) {
            tasks.remove(task.getTag());
            executeTask(task);
        }
    }

    private void executeTask(InitTask task) {
        long delayTime = 0;
        if (task instanceof SituationTask) delayTime = ((SituationTask) task).getDelayTime();
        if (task.isInThread()) {
            threadHandler.sendMessageDelayed(threadHandler.obtainMessage(MSG_RunTasks, task), delayTime);
        } else {
            handler.sendMessageDelayed(handler.obtainMessage(MSG_RunTasks, task), delayTime);
        }
    }

    private void dispatchAllNormalTaskFinish() {
//        assert accountDisposable != null: "请调用 attachAccountObservable 来注册账户条件触发时间点, 否则将不会触发账户依赖条件任务";
//        assert networkDisposable != null: "请调用 attachNetWorkObservable 来注册网络条件触发时间点, 否则将不会触发网络依赖条件任务";
        threadHandler.post(() -> startProceed(IInitializationSituationTask.class, "normalTasks done event"));
        for (InitializationListener listener : listeners) {
            listener.onAllNormalTaskFinish();
        }
    }

    private void dispatchTaskFinish(InitTask task) {
        handler.post(() -> {
            for (InitializationListener listener : listeners) {
                if (task instanceof IInitializationNormalTask)
                    listener.onNormalTaskFinish(task);
                if (task instanceof IInitializationSituationTask)
                    listener.onSituationTaskFinish(task);
            }
        });
    }

    private class RunTaskCallback implements Handler.Callback {
        @Override
        public boolean handleMessage(@NonNull Message msg) {
            if (msg.what == MSG_RunTasks) {
                InitTask task = (InitTask) msg.obj;
                boolean ref = false;
                boolean canRun = false;
                if (task instanceof IInitializationNormalTask) {
                    canRun = true;
                } else if (task instanceof IInitializationSituationTask) {
                    readyTodoSituations.remove(task);
                    SituationTask st = (SituationTask) task;
                    canRun = (!st.needNetWork || isNetWorkReady.get()) && (!st.needAccount || isAccountReady.get());
                }

                if (canRun) {
                    long startStamp = SystemClock.elapsedRealtime();
                    logs("Initialization 运行任务:" + task);
                    task.setStartTs(startStamp);
                    ref = task.run();
                    task.setEndTs(SystemClock.elapsedRealtime());
                    logs("Initialization 运行任务" + (ref ? "完成:" : "异常:") + task.toString() + " 使用时间: " + (task.endTime() - task.startTime()) + "毫秒 主线程: " + (task.isInThread() ? "否" : "是"));
                } else {
                    logs("Initialization 运行任务不成立:" + task.toString());
                }

                if (!ref) {
                    if (task instanceof IInitializationSituationTask) {
                        //条件任务就 安排在 下一次触发
                        threadHandler.post(() -> tasks.put(task.getTag(), (IInitializationTask) task));
                    } else if (IInitializationNormalTask.class.isAssignableFrom(task.getClass())) {
                        //普通任务就继续 触发
                        executeTask(task);
                    }
                } else {
                    handler.post(() -> dispatchTaskFinish(task));
                    if (task instanceof IInitializationNormalTask && normalTaskDoneCount.incrementAndGet() == normalTaskTotalCount.get()) {
                        handler.sendEmptyMessage(MSG_DISPATCH_NORMAL_TASK_DONE);
                    }
                }
            } else if (msg.what == MSG_DISPATCH_NORMAL_TASK_DONE) {
                dispatchAllNormalTaskFinish();
            }
            return false;
        }
    }

    private void logs(String content) {
        if (isInTestMode) {
            Log.v(Initialization.class.getSimpleName(), content);
        }
    }


    public interface InitializationListener {
        //正常任务全部完成
        void onNormalTaskFinish(ITask task);

        void onAllNormalTaskFinish();

        //正常任务 无法 全部完成 Todo
        void onNormalTasksFail();

        //某一个 场景任务 完成
        void onSituationTaskFinish(ITask task);
    }

}
