package com.cv.media.lib.common_utils.ntp;

import android.app.AlarmManager;
import android.content.Context;
import android.os.SystemClock;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.TimeUnit;


/**
 * Ntp 时间管理
 */
public class SntpManager {
    public final static int NETWORK_TIMEOUT_MILLIS = 5000;
    public final static long DEFAULT_ERROR_LIMIT = TimeUnit.MINUTES.toMillis(5);// 默认允许误差时间
    public final static long DEFAULT_SYNC_INTERVAL = TimeUnit.HOURS.toMillis(6);// 成功后，6小时校验一次
    private final static List<String> ntpServers = new ArrayList<>();

    private boolean isSuccess; // 最近一次同步是否成功
    private long lastSyncTime; // 上次同步完成时间

    static {
        ntpServers.add("0.pool.ntp.org");
        ntpServers.add("176.9.118.9");
        ntpServers.add("1.pool.ntp.org");
        ntpServers.add("61.216.153.106");
        ntpServers.add("2.pool.ntp.org");
        ntpServers.add("203.135.184.123");
        ntpServers.add("3.pool.ntp.org");
        ntpServers.add("108.59.2.24");
        ntpServers.add("0.south-america.pool.ntp.org");
        ntpServers.add("200.189.40.8");
        ntpServers.add("1.south-america.pool.ntp.org");
        ntpServers.add("164.73.227.4");
        ntpServers.add("2.south-america.pool.ntp.org");
        ntpServers.add("200.192.232.8");
        ntpServers.add("3.south-america.pool.ntp.org");
        ntpServers.add("201.217.3.86");
        ntpServers.add("time.google.com");
        ntpServers.add("216.239.35.12");
    }

    private static SntpManager instance;

    private SntpManager() {
        isSuccess = false;
        lastSyncTime = 0;
    }

    private SntpClient client = new SntpClient();

    public static SntpManager getInstance() {
        if (instance == null) {
            instance = new SntpManager();
        }
        return instance;
    }

    public long getNtpNow() {
        //循环尝试获取时间，只要任何一个服务器返回正确，则使用，否则尝试下一个服务器
        for (String ntpServer : ntpServers) {
            if (client.requestTime(ntpServer, NETWORK_TIMEOUT_MILLIS)) {
                long ntpNow = client.getNtpTime() + SystemClock.elapsedRealtime() - client.getNtpTimeReference();
                return ntpNow;
            }
        }
        //TimberUtils.e("fail to get NTP from: %s", ntpServers);
        return -1;
    }

    /**
     * 更新系统时间
     * permission SET_TIME
     * Signed with the system image
     * Installed to the /system/ folder
     *
     * @param errorLimitMillis 误差值。因为时间难以绝对准确，如果在误差之内则不更新时间
     * @return true 更新成功，或者在误差之内不需要更新/false 更新失败
     */
    public synchronized boolean syncNtpNow(Context context, long errorLimitMillis) {
        long ntpNow = getNtpNow();
        if (ntpNow < 0) {
            //TimberUtils.e("NTP: error in get ntp time from internet");
            return false;
        }
        long sysNow = System.currentTimeMillis();
        long errorMillis = Math.abs(ntpNow - sysNow);
        if (errorMillis <= errorLimitMillis) {
            //Timber.i("NTP: don't need to change time. ntp[%d], sys[%d], error range[%d]", ntpNow, sysNow, errorMillis);
            return true;
        }
        try {
            AlarmManager am = (AlarmManager) context.getSystemService(Context.ALARM_SERVICE);
            am.setTime(ntpNow);
            //Timber.i("NTP: changed sys time. ntp[%d], sys[%d], error range[%d]", ntpNow, sysNow, errorMillis);
            return true;
        } catch (Exception e) {
            //TimberUtils.e(e, "NTP: set time error");
            return false;
        }
    }


    public boolean syncNtp(Context context) {
        return syncNtp(context, DEFAULT_ERROR_LIMIT);
    }

    public boolean syncNtp(Context context, long errorLimitMillis) {
        isSuccess = syncNtpNow(context, errorLimitMillis);
        if (isSuccess) {
            lastSyncTime = System.currentTimeMillis();
        }
        return isSuccess;
    }

    public void syncNtpInterval(Context context) {
        long now = System.currentTimeMillis();
        if (isSuccess) {
            if (now - lastSyncTime >= DEFAULT_SYNC_INTERVAL) syncNtp(context);
        } else {
            syncNtp(context);
        }
    }
}
