package com.cv.media.lib.common_utils.clock;

import android.text.format.Time;

import com.cv.media.lib.common_utils.callback.ICallBack;
import com.cv.media.lib.common_utils.utils.Singleton;

import java.util.ArrayList;
import java.util.concurrent.TimeUnit;

import io.reactivex.Observable;
import io.reactivex.annotations.NonNull;
import io.reactivex.disposables.Disposable;
import io.reactivex.functions.Function;
import io.reactivex.schedulers.Schedulers;

/**
 * 时钟
 * 职责:
 *
 * @author Damon
 */
public class ClockManager {
    private final Time t = new Time();
    private final ArrayList<ICallBack<ClockTime>> mCbs = new ArrayList<>();
    private Disposable disposable;

    private static Singleton<ClockManager> singleton = new Singleton<ClockManager>() {
        @Override
        protected ClockManager create() {
            return new ClockManager();
        }
    };

    public static ClockManager getInstance() {
        return singleton.get();
    }

    private ClockManager() {
    }

    public void addCallback(ICallBack<ClockTime> cb) {
        synchronized (ClockManager.class) {
            mCbs.add(cb);
            startWork();
        }
    }

    public synchronized void removeCallback(ICallBack<ClockTime> cb) {
        synchronized (ClockManager.class) {
            mCbs.remove(cb);
            if(mCbs.isEmpty()) stopWork();
        }
    }

    private void startWork() {
        if (disposable == null) {
            disposable = Observable.interval(1, TimeUnit.SECONDS).subscribeOn(Schedulers.io()).map(new Function<Long, ClockTime>() {
                @Override
                public ClockTime apply(@NonNull Long aLong) {
                    t.setToNow(); // 取得系统时间。
                    String hour = t.hour < 10 ? "0" + (t.hour) : t.hour + ""; // 默认24小时制
                    String minute = t.minute < 10 ? "0" + (t.minute) : t.minute + "";
                    return new ClockTime(hour, minute);
                }
            }).subscribe(time -> {
                for (ICallBack<ClockTime> cb : mCbs) {
                    cb.call(time);
                }
            });
        }
    }

    private void stopWork() {
        if (disposable != null) {
            disposable.dispose();
            disposable = null;
        }
    }

    public static class ClockTime {
        public String hour;
        public String min;

        public ClockTime(String hour, String min) {
            this.hour = hour;
            this.min = min;
        }
    }
}
