package com.cv.media.lib.common_utils.pool;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;

/**
 * An object pool that can provide reused objects if available.
 *
 * @hide
 */
public class ObjectPool {

    private static final Object sPoolSync = new Object();
    private static final Map<Class, ArrayList<? extends ObjectPoolItem>> sPoolMap =
            new HashMap<>();

    private static final int MAX_POOL_SIZE = 50;

    /**
     * Obtain an instance of a specific class from the pool
     *
     * @param itemClass The class of the object we're looking for.
     * @return An instance or null if there is none.
     */
    public static <T extends ObjectPoolItem> T obtain(Class<T> itemClass) {
        synchronized (sPoolSync) {
            @SuppressWarnings("unchecked") final ArrayList<T> itemPool = (ArrayList<T>) sPoolMap.get(itemClass);
            if (itemPool != null && !itemPool.isEmpty()) {
                return itemPool.remove(itemPool.size() - 1);
            }
            try {
                T item = itemClass.newInstance();
                return item;
            } catch (Exception e) {
                throw new RuntimeException(e);
            }
        }
    }

    /**
     * Recycle the object to the pool. The object should be properly cleared before this.
     *
     * @param item The object to recycle.
     * @see ObjectPoolItem#recycle()
     */
    public static <T extends ObjectPoolItem> void recycle(T item) {
        synchronized (sPoolSync) {
            @SuppressWarnings("unchecked")
            ArrayList<T> itemPool = (ArrayList<T>) sPoolMap.get(item.getClass());
            if (itemPool == null) {
                itemPool = new ArrayList<>();
                sPoolMap.put(item.getClass(), itemPool);
            }
            // Check if the item is already in the pool
            final int size = itemPool.size();
            for (int i = 0; i < size; i++) {
                if (itemPool.get(i) == item) {
                    throw new IllegalStateException("Trying to recycle already recycled item :" + item.toString());
                }
            }

            if (size < MAX_POOL_SIZE) {
                itemPool.add(item);
            }
        }
    }
}
