package com.cv.media.lib.common_utils.keyevent;

import android.app.Activity;
import android.app.Application;
import android.os.Bundle;
import android.util.SparseArray;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.cv.media.lib.common_utils.utils.Singleton;

import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

class TriggerManager implements Application.ActivityLifecycleCallbacks {
    final HashMap<Class<? extends Activity>, HashMap<String, Method>> mMethodsCache = new HashMap<>();
    SparseArray<CallBackWrapper> mWrapperCache = new SparseArray<>();
    List<Class<? extends Activity>> activityClassesUnNeeded = new ArrayList<>();

    static Singleton<TriggerManager> mInstance = new Singleton<TriggerManager>() {
        @Override
        protected TriggerManager create() {
            return new TriggerManager();
        }
    };

    private TriggerManager() {

    }

    static TriggerManager getInstance() {
        return mInstance.get();
    }

    void create(Activity activity) {
        if(activityClassesUnNeeded.contains(activity.getClass())) return;
        Set<Map.Entry<String, Method>> entries = getEntry(activity);
        if(entries != null) {
            CallBackWrapper callBackWrapper = new CallBackWrapper(activity);
            for (Map.Entry<String, Method> entry : entries) {
                callBackWrapper.add(entry.getKey(), entry.getValue());
            }
            mWrapperCache.put(activity.hashCode(), callBackWrapper);
        }else {
            activityClassesUnNeeded.add(activity.getClass());
        }
    }

    void destroy(Activity activity) {
        mWrapperCache.remove(activity.hashCode());
    }

    void on(Activity activity) {
        CallBackWrapper callBackWrapper =  mWrapperCache.get(activity.hashCode());
        if(callBackWrapper != null) callBackWrapper.on();
    }

    void off(Activity activity) {
        CallBackWrapper callBackWrapper =  mWrapperCache.get(activity.hashCode());
        if(callBackWrapper != null) callBackWrapper.off();
    }

    Set<Map.Entry<String, Method>> getEntry(Activity activity) {
        HashMap<String, Method> keyCodesToMethods = mMethodsCache.get(activity.getClass());
        if (keyCodesToMethods == null) {
            keyCodesToMethods = new HashMap<>();
            ArrayList<Method> methods = new ArrayList<>();
            Class ref = activity.getClass();
            while (ref != null && !ref.getName().startsWith("android.app") && !ref.getName().startsWith("androidx")) {
                Method[] ms = ref.getDeclaredMethods();
                for (Method method:ms){
                    methods.add(method);
                }
                ref = ref.getSuperclass();
            }
            for (Method method : methods) {
                OnNumKeyCombination keyEvents = method.getAnnotation(OnNumKeyCombination.class);
                if (keyEvents != null) {
                    method.setAccessible(true);
                    String keyCodeString = keyEvents.value();
                    keyCodesToMethods.put(keyCodeString, method);
                }
            }
        }
        if(!keyCodesToMethods.isEmpty()) {
            mMethodsCache.put(activity.getClass(), keyCodesToMethods);
            return keyCodesToMethods.entrySet();
        }else
            return null;
    }


    @Override
    public void onActivityCreated(@NonNull Activity activity, @Nullable Bundle savedInstanceState) {
        create(activity);
    }

    @Override
    public void onActivityStarted(@NonNull Activity activity) {
        on(activity);
    }

    @Override
    public void onActivityResumed(@NonNull Activity activity) {
    }

    @Override
    public void onActivityPaused(@NonNull Activity activity) {
    }

    @Override
    public void onActivityStopped(@NonNull Activity activity) {
        off(activity);
    }

    @Override
    public void onActivitySaveInstanceState(@NonNull Activity activity, @NonNull Bundle outState) {
    }

    @Override
    public void onActivityDestroyed(@NonNull Activity activity) {
        destroy(activity);
    }
}
