package com.cv.media.lib.common_utils.image;

import android.graphics.Bitmap;
import android.graphics.Canvas;
import android.graphics.Matrix;
import android.graphics.Paint;
import android.os.Build;

import androidx.annotation.NonNull;

import com.bumptech.glide.load.engine.bitmap_recycle.BitmapPool;
import com.bumptech.glide.load.resource.bitmap.BitmapTransformation;
import com.bumptech.glide.util.Synthetic;

import java.math.BigDecimal;
import java.security.MessageDigest;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

/**
 * 因显示设备可能会比例有变化, 导致控件比例有变化
 * 此处处理 只是裁剪原始位图, 不应该增大位图 长宽
 * 会损失比例之外的像素
 * 职责:
 *
 * @author Damon
 */
class MyTransform extends BitmapTransformation {
    public static MyTransform TRANSFORM = new MyTransform();
    private static float SAFE_PERCENT_DIFF = 0.2f;  //比例差 宽容度
    private static final Set<String> MODELS_REQUIRING_BITMAP_LOCK =
            new HashSet<>(
                    Arrays.asList(
                            // Moto X gen 2
                            "XT1085",
                            "XT1092",
                            "XT1093",
                            "XT1094",
                            "XT1095",
                            "XT1096",
                            "XT1097",
                            "XT1098",
                            // Moto G gen 1
                            "XT1031",
                            "XT1028",
                            "XT937C",
                            "XT1032",
                            "XT1008",
                            "XT1033",
                            "XT1035",
                            "XT1034",
                            "XT939G",
                            "XT1039",
                            "XT1040",
                            "XT1042",
                            "XT1045",
                            // Moto G gen 2
                            "XT1063",
                            "XT1064",
                            "XT1068",
                            "XT1069",
                            "XT1072",
                            "XT1077",
                            "XT1078",
                            "XT1079"
                    )
            );


    private static final Lock BITMAP_DRAWABLE_LOCK =
            MODELS_REQUIRING_BITMAP_LOCK.contains(Build.MODEL)
                    ? new ReentrantLock() : new NoLock();

    private static final Paint DEFAULT_PAINT = new Paint(Paint.DITHER_FLAG | Paint.FILTER_BITMAP_FLAG);

    private static final String ID = "com.mm.android.baseclz.glide.MyTransform";
    private static final byte[] ID_BYTES = ID.getBytes(CHARSET);


    @Override
    protected Bitmap transform(@NonNull BitmapPool pool, @NonNull Bitmap toTransform, int outWidth, int outHeight) {
        Bitmap inBitmap = toTransform;

        if (inBitmap.getWidth() == outWidth && inBitmap.getHeight() == outHeight) {
            return inBitmap;
        }
        float bitmapWH = inBitmap.getWidth() / (float) inBitmap.getHeight();
        float outWH = outWidth / (float) outHeight;
        float safeDiff = outWH * SAFE_PERCENT_DIFF;
        BigDecimal bSafe = new BigDecimal(safeDiff);
        BigDecimal bBitmap = new BigDecimal(bitmapWH);
        BigDecimal bOut = new BigDecimal(outWH);
        if (bSafe.compareTo(bBitmap.subtract(bOut).abs()) >= 0) {
            return inBitmap;
        }

        final float widthPercentage = outWidth / (float) inBitmap.getWidth();
        final float heightPercentage = outHeight / (float) inBitmap.getHeight();
        final float maxPercentage = Math.max(widthPercentage, heightPercentage);
        int outputBitmapWidth;
        int outputBitmapHeight;

        Matrix matrix = new Matrix();

        if (maxPercentage == widthPercentage) {
            outputBitmapWidth = inBitmap.getWidth();
            outputBitmapHeight = Math.round(outHeight * 1 / widthPercentage);
            matrix.postTranslate(0, -(inBitmap.getHeight() - outputBitmapHeight) / 2);
        } else {
            outputBitmapHeight = inBitmap.getHeight();
            outputBitmapWidth = Math.round(outWidth * 1 / heightPercentage);
            matrix.postTranslate(-(inBitmap.getWidth() - outputBitmapWidth) / 2, 0);
        }

        if (inBitmap.getWidth() == outputBitmapWidth && inBitmap.getHeight() == outputBitmapHeight) {
            return inBitmap;
        }

        Bitmap.Config config = inBitmap.getConfig() != null ? inBitmap.getConfig() : Bitmap.Config.ARGB_8888;
        Bitmap toReuse = pool.get(outputBitmapWidth, outputBitmapHeight, config);

        toReuse.setHasAlpha(inBitmap.hasAlpha());

        applyMatrix(inBitmap, toReuse, matrix);

        return toReuse;
    }

    private static void applyMatrix(@NonNull Bitmap inBitmap, @NonNull Bitmap targetBitmap,
                                    Matrix matrix) {
        BITMAP_DRAWABLE_LOCK.lock();
        try {
            Canvas canvas = new Canvas(targetBitmap);
            canvas.drawBitmap(inBitmap, matrix, DEFAULT_PAINT);
            canvas.setBitmap(null);
        } finally {
            BITMAP_DRAWABLE_LOCK.unlock();
        }
    }

    @Override
    public boolean equals(Object o) {
        return o instanceof MyTransform;
    }

    @Override
    public int hashCode() {
        return ID.hashCode();
    }

    @Override
    public void updateDiskCacheKey(@NonNull MessageDigest messageDigest) {
        messageDigest.update(ID_BYTES);
    }


    private static final class NoLock implements Lock {

        @Synthetic
        NoLock() {
        }

        @Override
        public void lock() {
            // do nothing
        }

        @Override
        public void lockInterruptibly() throws InterruptedException {
            // do nothing
        }

        @Override
        public boolean tryLock() {
            return true;
        }

        @Override
        public boolean tryLock(long time, @NonNull TimeUnit unit) throws InterruptedException {
            return true;
        }

        @Override
        public void unlock() {
            // do nothing
        }

        @NonNull
        @Override
        public Condition newCondition() {
            throw new UnsupportedOperationException("Should not be called");
        }
    }
}
