package com.cv.media.lib.hardware.network;

import android.annotation.SuppressLint;
import android.bluetooth.BluetoothAdapter;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;
import android.net.wifi.WifiManager;
import android.telephony.TelephonyManager;
import android.text.TextUtils;

import com.cv.media.lib.common_utils.utils.MACUtils;

import java.net.Inet4Address;
import java.net.InetAddress;
import java.net.NetworkInterface;
import java.net.SocketException;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.List;

import static com.cv.media.lib.common_utils.provider.ContextProvider.getContext;

/**
 * Created by zhangyaoa on 2021/1/26.
 */
public class NetworkManager {
    private static NetworkManager sInstance;

    public static final String ETHERNET_SERVICE_NAME = "ethernet";
    public static final String WIFI_INTERFACE = "wlan0";
    public static final String ETHERNET_INTERFACE = "eth0";
    public static int ETHERNET_ENABLED = 2;

    private String wifiMac;
    private String ethernetMac;
    private String bluetoothMac;
    private String ipAddress = "";

    private ConnectivityManager connectivityManager;
    private WifiManager wifiManager;
    private TelephonyManager telephonyManager;
    private BluetoothAdapter ba;
    private Context mContext;

    private List<NetWorkListener> mNetWorkListeners = new ArrayList<>();

    @SuppressLint("WrongConstant")
    private NetworkManager(Context context) {
        mContext = context;
        initNetwork();
    }

    public static synchronized NetworkManager getInstance() {
        if (sInstance == null) {
            sInstance = new NetworkManager(getContext());
        }
        return sInstance;
    }

    public static void init(Context context) {
        sInstance = new NetworkManager(context);
    }

    private void initNetwork() {
        wifiManager = (WifiManager) mContext.getSystemService(Context.WIFI_SERVICE);
        connectivityManager = (ConnectivityManager) mContext.getSystemService(Context.CONNECTIVITY_SERVICE);
        telephonyManager = (TelephonyManager) mContext.getSystemService(Context.TELEPHONY_SERVICE);

        ethernetMac = MACUtils.getEthMac();
        wifiMac = MACUtils.getWifiMac(mContext);

        try {
            bluetoothMac = MACUtils.getBluetoothMac(mContext);
        } catch (Exception e) {
            e.printStackTrace();
        }

        mContext.registerReceiver(mNetworkReceiver, new IntentFilter(ConnectivityManager.CONNECTIVITY_ACTION));

        refreshNetworkInfo(getNetworkState());
    }

    //刷新网络信息
    public void refreshNetworkInfo(ENetworkState networkState) {
        if (networkState != ENetworkState.WIFI_CONNECTED
                && networkState != ENetworkState.ETHERNET_CONNECTED
                && networkState != ENetworkState.MOBILE_CONNECTED) {
            return;
        }

        try {
            NetworkInterface nif = null;
            if (isWifiConnected()) {
                nif = NetworkInterface.getByName(WIFI_INTERFACE);
                wifiMac = MACUtils.getWifiMac(mContext);
            } else if (isEthernetConnected()) {
                nif = NetworkInterface.getByName(ETHERNET_INTERFACE);
            } else {
                //手机网络
                ipAddress = getMobileNetIpAddress();
//                nif = NetworkInterface.getByName(ETHERNET_INTERFACE);
            }

            if (null != nif) {
                Enumeration<InetAddress> inet = nif.getInetAddresses();
                // 遍历每一个接口绑定的所有ip
                while (inet.hasMoreElements()) {
                    InetAddress address = inet.nextElement();
                    if (address.isAnyLocalAddress() || address.isLoopbackAddress() || !(address instanceof Inet4Address) || !address.isSiteLocalAddress()) {
                        continue;
                    }
                    ipAddress = address.getHostAddress();
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        } finally {

        }
    }

    public String getMobileNetIpAddress() {
        try {
            for (Enumeration<NetworkInterface> en = NetworkInterface.getNetworkInterfaces(); en.hasMoreElements(); ) {
                NetworkInterface intf = en.nextElement();
                for (Enumeration<InetAddress> enumIpAddr = intf.getInetAddresses(); enumIpAddr.hasMoreElements(); ) {
                    InetAddress inetAddress = enumIpAddr.nextElement();
                    if (!inetAddress.isLoopbackAddress() && inetAddress instanceof Inet4Address) {
                        return inetAddress.getHostAddress();
                    }
                }
            }
        } catch (SocketException ex) {
            return "";
        }
        return "";
    }

    public String getWifiMac() {
        return wifiMac;
    }

    public String getEthernetMac() {
        return ethernetMac;
    }

    public String getBluetoothMac() {
        return bluetoothMac;
    }

    public String getIpAddress() {
        return ipAddress;
    }

    public String getMac() {
        String mac = "";
        if (!TextUtils.isEmpty(ethernetMac)) {
            mac = ethernetMac.trim();
        } else if (!TextUtils.isEmpty(wifiMac)) {
            mac = wifiMac.trim();
        }
        return mac.replace(":", "").toLowerCase();
    }

    /**
     * Wifi是否连接
     */
    public boolean isWifiConnected() {
        try {
            return getNetworkState() == ENetworkState.WIFI_CONNECTED;
        } catch (Exception e) {
            return false;
        }
    }

    /**
     * Ethernet是否连接
     */
    public boolean isEthernetConnected() {
        try {
            NetworkInfo networkInfo = connectivityManager.getActiveNetworkInfo();
            return networkInfo != null && networkInfo.isConnected() && networkInfo.getType() == ConnectivityManager.TYPE_ETHERNET;
        } catch (Exception e) {
            return false;
        }
    }

    public ENetworkState getNetworkState() {
        NetworkInfo networkInfo = connectivityManager.getActiveNetworkInfo();
        if (networkInfo != null && networkInfo.isConnected()) {
            switch (networkInfo.getType()) {
                case ConnectivityManager.TYPE_ETHERNET:
                    return ENetworkState.ETHERNET_CONNECTED;
                case ConnectivityManager.TYPE_WIFI:
                    return ENetworkState.WIFI_CONNECTED;
                // 目前缺省都归成WiFi
                case ConnectivityManager.TYPE_MOBILE:
                case ConnectivityManager.TYPE_MOBILE_DUN:
                case ConnectivityManager.TYPE_MOBILE_HIPRI:
                case ConnectivityManager.TYPE_MOBILE_MMS:
                case ConnectivityManager.TYPE_MOBILE_SUPL:
                    return ENetworkState.MOBILE_CONNECTED;
                default:
                    return ENetworkState.WIFI_CONNECTED;
            }
        } else if (wifiManager.isWifiEnabled()) {
            return ENetworkState.WIFI_DISCONNECTED;
        } else {
            return ENetworkState.ETHERNET_DISCONNECTED;
        }
    }

    public boolean isNetworkConnected() {
        try {
            NetworkInfo networkInfo = connectivityManager.getActiveNetworkInfo();
            return networkInfo != null && networkInfo.isConnected();
        }catch (Exception e){
            return false;
        }
    }


    /**
     * 注册网络变动的广播接收
     */
    private BroadcastReceiver mNetworkReceiver = new BroadcastReceiver() {
        @Override
        public void onReceive(Context context, Intent intent) {
            try {
                String action = intent.getAction();
                if (action.equals(ConnectivityManager.CONNECTIVITY_ACTION)) {
                    ConnectivityManager connectMgr = (ConnectivityManager) context.getSystemService(Context.CONNECTIVITY_SERVICE);
                    NetworkInfo networkInfo = connectMgr.getActiveNetworkInfo();
                    if (networkInfo == null || !networkInfo.isConnected()) {
                        notifyListeners(ENetworkState.SERVER_DISCONNECTED);
                        return;
                    }

                    ENetworkState networkState = getNetworkState();

                    refreshNetworkInfo(networkState);

                    notifyListeners(networkState);
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
    };

    private void notifyListeners(ENetworkState networkState) {
        for (NetWorkListener listener : mNetWorkListeners) {
            listener.onNetworkChanged(networkState);
        }
    }

    public interface NetWorkListener {
        void onNetworkChanged(ENetworkState networkState);
    }

    public void addNetworkListener(NetWorkListener listener) {
        try {
            mNetWorkListeners.add(listener);
        } catch (Exception e) {

        }
    }

    public void removeNetworkListener(NetWorkListener listener) {
        try {
            mNetWorkListeners.remove(listener);
        } catch (Exception e) {

        }
    }
}
