package com.cv.media.lib.common_utils.keyevent;

import android.app.Activity;
import android.app.Application;
import android.os.Bundle;
import android.util.SparseArray;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.cv.media.lib.common_utils.utils.Singleton;

import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import android.util.Log;

class TriggerManager implements Application.ActivityLifecycleCallbacks {
    final HashMap<Class<? extends Activity>, HashMap<String, Method>> mMethodsCache = new HashMap<>();
    SparseArray<CallBackWrapper> mWrapperCache = new SparseArray<>();
    List<Class<? extends Activity>> activityClassesUnNeeded = new ArrayList<>();

    static Singleton<TriggerManager> mInstance = new Singleton<TriggerManager>() {
        @Override
        protected TriggerManager create() {
            return new TriggerManager();
        }
    };

    private TriggerManager() {

    }

    /**
     * 安全地获取类的方法列表，处理各种兼容性问题
     * @param clazz 要扫描的类
     * @return 方法列表，如果遇到问题则返回空列表
     */
    private List<Method> getDeclaredMethodsSafely(Class<?> clazz) {
        List<Method> methods = new ArrayList<>();
        if (clazz == null) {
            return methods;
        }
        
        try {
            Method[] declaredMethods = clazz.getDeclaredMethods();
            for (Method method : declaredMethods) {
                try {
                    // 尝试访问方法属性，如果遇到类加载问题则跳过
                    method.getReturnType();
                    methods.add(method);
                } catch (NoClassDefFoundError e) {
                    // 跳过包含不可用类的方法
                    Log.w("TriggerManager", "跳过方法 " + method.getName() + ": " + e.getMessage());
                } catch (Exception e) {
                    // 跳过其他异常的方法
                    Log.w("TriggerManager", "跳过方法 " + method.getName() + ": " + e.getMessage());
                }
            }
        } catch (NoClassDefFoundError e) {
            // 如果整个类的getDeclaredMethods()都失败，记录并跳过
            Log.w("TriggerManager", "跳过类 " + clazz.getName() + ": " + e.getMessage());
        } catch (Exception e) {
            // 处理其他异常
            Log.w("TriggerManager", "跳过类 " + clazz.getName() + ": " + e.getMessage());
        }
        
        return methods;
    }

    static TriggerManager getInstance() {
        return mInstance.get();
    }

    void create(Activity activity) {
        if(activityClassesUnNeeded.contains(activity.getClass())) return;
        Set<Map.Entry<String, Method>> entries = getEntry(activity);
        if(entries != null) {
            CallBackWrapper callBackWrapper = new CallBackWrapper(activity);
            for (Map.Entry<String, Method> entry : entries) {
                callBackWrapper.add(entry.getKey(), entry.getValue());
            }
            mWrapperCache.put(activity.hashCode(), callBackWrapper);
        }else {
            activityClassesUnNeeded.add(activity.getClass());
        }
    }

    void destroy(Activity activity) {
        mWrapperCache.remove(activity.hashCode());
    }

    void on(Activity activity) {
        CallBackWrapper callBackWrapper =  mWrapperCache.get(activity.hashCode());
        if(callBackWrapper != null) callBackWrapper.on();
    }

    void off(Activity activity) {
        CallBackWrapper callBackWrapper =  mWrapperCache.get(activity.hashCode());
        if(callBackWrapper != null) callBackWrapper.off();
    }

    Set<Map.Entry<String, Method>> getEntry(Activity activity) {
        HashMap<String, Method> keyCodesToMethods = mMethodsCache.get(activity.getClass());
        if (keyCodesToMethods == null) {
            keyCodesToMethods = new HashMap<>();
            ArrayList<Method> methods = new ArrayList<>();
            Class ref = activity.getClass();
            while (ref != null && !ref.getName().startsWith("android.app") && !ref.getName().startsWith("androidx")) {
                // 使用安全的方法获取类的方法列表
                List<Method> classMethods = getDeclaredMethodsSafely(ref);
                methods.addAll(classMethods);
                ref = ref.getSuperclass();
            }
            for (Method method : methods) {
                try {
                    OnNumKeyCombination keyEvents = method.getAnnotation(OnNumKeyCombination.class);
                    if (keyEvents != null) {
                        method.setAccessible(true);
                        String keyCodeString = keyEvents.value();
                        keyCodesToMethods.put(keyCodeString, method);
                    }
                } catch (Exception e) {
                    // 跳过有问题的注解处理
                    Log.w("TriggerManager", "跳过注解处理: " + e.getMessage());
                }
            }
        }
        if(!keyCodesToMethods.isEmpty()) {
            mMethodsCache.put(activity.getClass(), keyCodesToMethods);
            return keyCodesToMethods.entrySet();
        }else
            return null;
    }


    @Override
    public void onActivityCreated(@NonNull Activity activity, @Nullable Bundle savedInstanceState) {
        create(activity);
    }

    @Override
    public void onActivityStarted(@NonNull Activity activity) {
        on(activity);
    }

    @Override
    public void onActivityResumed(@NonNull Activity activity) {
    }

    @Override
    public void onActivityPaused(@NonNull Activity activity) {
    }

    @Override
    public void onActivityStopped(@NonNull Activity activity) {
        off(activity);
    }

    @Override
    public void onActivitySaveInstanceState(@NonNull Activity activity, @NonNull Bundle outState) {
    }

    @Override
    public void onActivityDestroyed(@NonNull Activity activity) {
        destroy(activity);
    }
}
