package com.cv.media.lib.common_utils.async;

import android.util.Log;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.RejectedExecutionHandler;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.locks.LockSupport;

/**
 * 动态吞吐线程池
 *
 * @author Damon
 */
class ThreadPoolImpl extends ThreadPoolExecutor {
    private static final boolean debuggable = false;
    private static final String TAG = ThreadPoolImpl.class.getSimpleName();

    private ThreadPoolImpl(int corePoolSize, int maximumPoolSize, long keepAliveTime, TimeUnit unit, BlockingQueue<Runnable> workQueue, ThreadFactory threadFactory, RejectedExecutionHandler handler) {
        super(corePoolSize, maximumPoolSize, keepAliveTime, unit, workQueue, threadFactory, handler);
        ((MyThreadFactory) threadFactory).setThreadPool(this);
        new Thread(new StateWatcher(this), "ThreadPoolImpl Watcher").start();
    }

    //    private Set<MyThread> mThreads = new LinkedHashSet<>();
//    private Set<RunnableWrap> mTasks = new LinkedHashSet<>();
//    private Object lockThreads = new Object();
//    private Object lockTasks = new Object();
    private AtomicInteger countTaskIn = new AtomicInteger(0);
    private AtomicInteger countTaskHandle = new AtomicInteger(0);


    static ThreadPoolImpl newInstance() {
        int coreSize = Runtime.getRuntime().availableProcessors() + 1;
        ThreadPoolImpl threadPool = new ThreadPoolImpl(coreSize,
                coreSize * 2,
                0,
                TimeUnit.SECONDS,
                new MyTaskQueue(),
                new MyThreadFactory(),
                new MyRejectHandler());
        return threadPool;
    }

    @Override
    public void execute(Runnable command) {
        super.execute(new RunnableWrap(this, command));
    }

    static class MyThreadFactory implements ThreadFactory {
        ThreadPoolImpl threadPool;

        public MyThreadFactory() {
        }

        public void setThreadPool(ThreadPoolImpl threadPool) {
            this.threadPool = threadPool;
        }

        @Override
        public Thread newThread(Runnable r) {
            Logs("创建线程");
            MyThread thread = new MyThread(r, "ThreadPoolEx", threadPool);
//            synchronized (threadPool.lockThreads) {
//                threadPool.mThreads.add(thread);
//            }
            return thread;
        }
    }

    static class RunnableWrap implements Runnable {
        Runnable runnable;
        long timeCreated;
        long timeRun;
        long timeFinish;
        ThreadPoolImpl threadPool;

        public RunnableWrap(ThreadPoolImpl threadPool, Runnable runnable) {
            timeCreated = System.nanoTime();
            this.runnable = runnable;
            this.threadPool = threadPool;
            threadPool.countTaskIn.incrementAndGet();
        }

        @Override
        public void run() {
            threadPool.countTaskHandle.incrementAndGet();
            timeRun = System.nanoTime();
            ((MyThread) Thread.currentThread()).ref.getAndSet(this);
            ((MyThread) Thread.currentThread()).taskTimeLast.getAndSet(0);
            try {
                if (runnable != null) {
//                    synchronized (threadPool.lockTasks) {
//                        threadPool.mTasks.add(this);
//                    }
                    runnable.run();
                }
            } finally {
                Logs("处理完一个任务");
                ((MyThread) Thread.currentThread()).ref.getAndSet(null);
                timeFinish = System.nanoTime();
//                synchronized (threadPool.lockTasks) {
//                    threadPool.mTasks.remove(this);
//                }
            }
        }
    }

    static class MyThread extends Thread {
        ThreadPoolImpl threadPool;
        AtomicReference<Runnable> ref = new AtomicReference<>();
        AtomicLong taskTimeLast = new AtomicLong();

        public MyThread(@Nullable Runnable target, @NonNull String name, ThreadPoolImpl threadPool) {
            super(target, name);
            this.threadPool = threadPool;
        }

        @Override
        public void run() {
            super.run();
//            synchronized (threadPool.lockThreads) {
//                threadPool.mThreads.remove(this);
//            }
            Logs("线程执行结束");
        }
    }

    static class MyTaskQueue extends LinkedBlockingQueue<Runnable> {
        private AtomicInteger rejectTimes = new AtomicInteger();

        public MyTaskQueue() {
            super();
        }

        void reject(int times) {
            rejectTimes.addAndGet(times);
        }

        @Override
        public boolean offer(Runnable e) {
            if (rejectTimes.get() > 0) {
                //触发 创建一个非核心工作线程
                rejectTimes.decrementAndGet();
                return false;
            }
            return super.offer(e);
        }
    }

    static class StateWatcher implements Runnable {
        final long TIME_INTERVAL_RECORD = 500;
        final long TIME_INTERVAL_CHECK = 1000 * 15;
        final long TIME_INTERVAL_RECORD_NANO = TimeUnit.MILLISECONDS.toNanos(TIME_INTERVAL_RECORD);
        private long pastTime;
        ThreadPoolImpl threadPool;
//        Set<MyThread> threads = new LinkedHashSet<>();
//        Set<Runnable> tasks = new LinkedHashSet<>();

        public StateWatcher(ThreadPoolImpl threadPool) {
            this.threadPool = threadPool;
        }

        @Override
        public void run() {
            for (; ; ) {
                if (pastTime >= TIME_INTERVAL_CHECK) {
                    int tasksIn = threadPool.countTaskIn.getAndSet(0);
                    int tasksHandle = threadPool.countTaskHandle.getAndSet(0);
                    Logs("接收任务数:" + tasksIn + " 处理任务数:" + tasksHandle);
                    if (tasksHandle < (0.95f * tasksIn)) {
                        Logs("出现吞吐不够的场景, 触发添加非核心工作线程");
                        triggerAddWorker();
                    } else {
                        Logs("运行良好");
                    }
                    pastTime = 0;
                }

                LockSupport.parkNanos(TIME_INTERVAL_RECORD_NANO);
                if (Thread.currentThread().isInterrupted()) {
                    break;
                }

                pastTime += TIME_INTERVAL_RECORD;
                Logs("当前任务队列数量:" + threadPool.getQueue().size());
            }
        }

        private void triggerAddWorker() {
            ((MyTaskQueue) threadPool.getQueue()).reject(1);
            threadPool.execute(new StubTask());
        }
    }


    static class StubTask implements Runnable {
        @Override
        public void run() {

        }
    }

    static class MyRejectHandler implements RejectedExecutionHandler {
        @Override
        public void rejectedExecution(Runnable r, ThreadPoolExecutor executor) {
            if (r instanceof StubTask) {
                return;
            }

            if (!executor.isShutdown() && executor.getQueue().remainingCapacity() == 0) {
                executor.getQueue().poll();
            }

            executor.execute(r);
        }
    }

    private static void Logs(String content) {
        if (debuggable) Log.v(TAG, content);
    }
}
