package com.cv.media.lib.hardware.device;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.pm.PackageManager;
import android.os.Build;
import android.os.Environment;
import android.os.StatFs;
import android.provider.Settings;
import android.telephony.TelephonyManager;
import android.text.TextUtils;
import android.util.Base64;
import android.util.Log;

import androidx.core.content.ContextCompat;

import com.cv.media.lib.common_utils.crypto.Config;
import com.cv.media.lib.common_utils.utils.StringUtils;
import com.cv.media.lib.common_utils.utils.io.IOUtils;

import java.io.FileInputStream;
import java.io.IOException;
import java.io.LineNumberReader;
import java.io.RandomAccessFile;
import java.io.StringReader;
import java.util.HashMap;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import static com.cv.media.lib.common_utils.provider.ContextProvider.getContext;

/**
 * Created by zhangyaoa on 2021/1/25.
 * <p>
 * 以前代码中所有使用到deviceInfo buildInfo hardwareInfo的地方统一从这里获取deviceInfo并从中取值
 */
public class HardwareManager {
    public static final String KEY_CPUINFO = "cpuinfo";
    public static final String KEY_CPU_HARDWARE = "Hardware";
    public static final String KEY_CPU_SERIAL = "Serial";
    public static final String KEY_READ_ALLWINNER_CPU_HARDWARE = "sunxi_platform";
    public static final String KEY_READ_ALLWINNER_CPU_SERIAL = "sunxi_serial";

    private DeviceInfo mDeviceInfo;
    private Context mContext;

    private static HardwareManager sInstance;

    private HardwareManager(Context context) {
        mContext = context;
        mDeviceInfo = new DeviceInfo();
        updateDeviceInfo();
    }

    public static synchronized HardwareManager getInstance() {
        if (sInstance == null) {
            sInstance = new HardwareManager(getContext());
        }
        return sInstance;
    }

    public static void init(Context context) {
        sInstance = new HardwareManager(context);
    }


    public void updateDeviceInfo() {
        Map<String, String> cpuInfos = getCupInfo();
        mDeviceInfo.setCpuinfo(Base64.encodeToString(nvl(cpuInfos.get(KEY_CPUINFO), "").getBytes(), Config.BASE64_FLAG));
        mDeviceInfo.setCpuHardware(cpuInfos.get(KEY_CPU_HARDWARE));
        mDeviceInfo.setCpuSerial(cpuInfos.get(KEY_CPU_SERIAL));
        mDeviceInfo.setImei(readImei());
        mDeviceInfo.setAndroidId(readAndroidId());
        mDeviceInfo.setRamSize(getTotalRam());
        mDeviceInfo.setRomSize(getTotalRom());
    }

    public DeviceInfo getDeviceInfo() {
        return mDeviceInfo;
    }

    private String nvl(String value, String defaultValue) {
        if (value == null) return defaultValue;
        return value;
    }

    private Map<String, String> getCupInfo() {
        Map<String, String> cpuinfo = new HashMap<>();
        try {
            String cpuinfoStr = "";
            try {
                FileInputStream fis = new FileInputStream("/proc/cpuinfo");
                cpuinfoStr = IOUtils.toString(fis);
                IOUtils.closeQuietly(fis);
            } catch (Exception e) {
                //proc/cpuinfo可能不存在
                cpuinfoStr = "";
            }
            cpuinfo.put(KEY_CPUINFO, cpuinfoStr);

            parserCupSerial(cpuinfo, cpuinfoStr);

            if (TextUtils.isEmpty(cpuinfo.get(KEY_CPU_SERIAL))) {
                try {
                    //amlogic 的 s905x5 芯片的 cpu 系列号获取位置变更
                    FileInputStream fis = new FileInputStream("/proc/cpu_chipid");
                    cpuinfoStr = IOUtils.toString(fis);
                    IOUtils.closeQuietly(fis);
                } catch (Exception e) {
                    //proc/cpuinfo可能不存在
                    cpuinfoStr = "";
                }
                parserCupSerial(cpuinfo, cpuinfoStr);
            }

            if (TextUtils.isEmpty(cpuinfo.get(KEY_CPU_SERIAL))) {
                readAllWinnerCpuInfo(cpuinfo);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return cpuinfo;
    }

    //解析 hard和serial
    private void parserCupSerial(Map<String, String> cpuinfo, String cpuinfoStr) {
        try {
            LineNumberReader reader = new LineNumberReader(new StringReader(cpuinfoStr));
            String line;
            while ((line = reader.readLine()) != null) {
                String[] array = line.split(":");
                if (array.length == 2) {
                    String key = StringUtils.trimToEmpty(array[0]);
                    String value = StringUtils.trimToEmpty(array[1]);
                    if (KEY_CPU_HARDWARE.equalsIgnoreCase(key)) {
                        cpuinfo.put(KEY_CPU_HARDWARE, value);
                    } else if (KEY_CPU_SERIAL.equalsIgnoreCase(key)) {
                        cpuinfo.put(KEY_CPU_SERIAL, value);
                    }
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }


    //全智芯片读取方式不一样
    private void readAllWinnerCpuInfo(Map<String, String> cpuinfo) {
        try {
            FileInputStream fis = new FileInputStream("sys/class/sunxi_info/sys_info");
            String cpuinfoStr = IOUtils.toString(fis);
            IOUtils.closeQuietly(fis);
            cpuinfo.put(KEY_CPUINFO, cpuinfoStr);
            LineNumberReader reader = new LineNumberReader(new StringReader(cpuinfoStr));
            String line;
            while ((line = reader.readLine()) != null) {
                String[] array = line.split(":");
                if (array.length == 2) {
                    String key = StringUtils.trimToEmpty(array[0]);
                    String value = StringUtils.trimToEmpty(array[1]);
                    //读取值用全智值，存入用统一值
                    if (KEY_READ_ALLWINNER_CPU_HARDWARE.equalsIgnoreCase(key)) {
                        cpuinfo.put(KEY_CPU_HARDWARE, value);
                    } else if (KEY_READ_ALLWINNER_CPU_SERIAL.equalsIgnoreCase(key)) {
                        cpuinfo.put(KEY_CPU_SERIAL, value);
                    }
                }
            }
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    @SuppressLint("HardwareIds")
    private String readImei() {
        String imei = "";
        try {

            TelephonyManager telephonyManager = (TelephonyManager) mContext.getSystemService(Context.TELEPHONY_SERVICE);
            if (Build.VERSION.SDK_INT >= 23) {
                if (ContextCompat.checkSelfPermission(mContext, android.Manifest.permission.READ_PHONE_STATE) != PackageManager.PERMISSION_GRANTED) {
                    // 没有权限
                    imei = "";
                } else {
                    if (telephonyManager != null) imei = telephonyManager.getDeviceId();
                }
            } else {
                if (telephonyManager != null) imei = telephonyManager.getDeviceId();
            }
        } catch (Exception e) {

        }

        return imei;
    }

    private String readAndroidId() {
        try {
            @SuppressLint("HardwareIds") String androidId = Settings.Secure.getString(mContext.getContentResolver(), Settings.Secure.ANDROID_ID);
            return androidId;
        } catch (Exception e) {
            return "";
        }
    }

    private long getTotalRam() {
        RandomAccessFile reader = null;
        String load = null;
        long totRam = 0;
        try {
            reader = new RandomAccessFile("/proc/meminfo", "r");
            load = reader.readLine();

            Pattern p = Pattern.compile("(\\d+)");
            Matcher m = p.matcher(load);
            String value = "";
            while (m.find()) {
                value = m.group(1);
            }
            reader.close();

            totRam = Long.parseLong(value);

        } catch (IOException ex) {
            ex.printStackTrace();
        } finally {
            // Streams.close(reader);
        }

        return totRam;
    }

    private long getTotalRom() {
        long kbAvailable = 0;
        try {
            StatFs stat = new StatFs(Environment.getExternalStorageDirectory().getPath());
            long bytesAvailable = stat.getBlockSizeLong() * stat.getBlockCountLong();
            kbAvailable = bytesAvailable / 1024;
        } catch (Exception e) {
            e.printStackTrace();
        }
        return kbAvailable;
    }
}
