package com.cv.media.lib.common_utils.utils;

import android.bluetooth.BluetoothAdapter;
import android.content.Context;
import android.net.wifi.WifiInfo;
import android.net.wifi.WifiManager;
import android.os.Build;
import android.text.TextUtils;

import com.cv.media.lib.common_utils.crypto.CryptoUtils;

import org.apache.commons.lang3.StringUtils;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.net.NetworkInterface;
import java.util.Collections;
import java.util.List;
import java.util.Locale;

public class MACUtils {
    private static String MAC_DIGEST = null;

    /**
     * Android  6.0 之前（不包括6.0）
     * 必须的权限  <uses-permission android:name="android.permission.ACCESS_WIFI_STATE" />
     *
     * @param context
     * @return
     */
    private static String getMacDefault(Context context) {
        String mac = null;
        if (context == null) {
            return mac;
        }

        WifiManager wifi = (WifiManager) context.getApplicationContext()
                .getSystemService(Context.WIFI_SERVICE);
        if (wifi == null) {
            return mac;
        }
        WifiInfo info = null;
        try {
            info = wifi.getConnectionInfo();
        } catch (Exception e) {

        }
        if (info == null) {
            return null;
        }
        mac = info.getMacAddress();
        if (!TextUtils.isEmpty(mac)) {
            mac = mac.toLowerCase(Locale.ENGLISH);
        }
        return mac;
    }

    /**
     * Android 6.0（包括） - Android 7.0（不包括）
     *
     * @return
     */
    private static String getMacAddress() {
        String mac = "";
        try {
            File file = new File("/sys/class/net/wlan0/address");
            if (file.exists()) {
                BufferedReader bufferedReader = new BufferedReader(new FileReader(file));
                mac = bufferedReader.readLine();
                bufferedReader.close();
            }
            if (mac.contains("-"))
                mac = mac.replace("-", ":").trim();
        } catch (Exception e) {
            e.printStackTrace();
        }
        return mac;
    }

    /**
     * 遍历循环所有的网络接口，找到接口是 wlan0
     * 必须的权限 <uses-permission android:name="android.permission.INTERNET" />
     *
     * @return
     */
    private static String getMacFromHardware() {
        try {
            List<NetworkInterface> all = Collections.list(NetworkInterface.getNetworkInterfaces());
            for (NetworkInterface nif : all) {
                if (!nif.getName().equalsIgnoreCase("wlan0")) continue;
                byte[] macBytes = nif.getHardwareAddress();
                if (macBytes == null) {
                    return null;
                }

                StringBuilder res1 = new StringBuilder();
                for (byte b : macBytes) {
                    res1.append(String.format("%02x:", b));
                }

                if (res1.length() > 0) {
                    res1.deleteCharAt(res1.length() - 1);
                }
                return res1.toString();
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return null;
    }

    public static String getWifiMac(Context context) {
        String macAddress = null;
        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.M) {//5.0以下
            macAddress = getMacDefault(context);
        } else {
            macAddress = getMacFromHardware();
        }
        if (TextUtils.isEmpty(macAddress) || TextUtils.equals("020000000000", macAddress) || TextUtils.equals("02:00:00:00:00:00", macAddress)) {
            macAddress = getMacAddress();
        }
        return toShrinkMac(macAddress);
    }

    private static String getEthMacAddress() {
        String mac = null;
        try {
            File file = new File("/sys/class/net/eth0/address");
            if (file.exists()) {
                BufferedReader bufferedReader = new BufferedReader(new FileReader(file));
                mac = bufferedReader.readLine();
                bufferedReader.close();
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return mac;
    }

    private static String getEthMacFromHardware() {
        try {
            List<NetworkInterface> all = Collections.list(NetworkInterface.getNetworkInterfaces());
            for (NetworkInterface nif : all) {
                if (!nif.getName().equalsIgnoreCase("eth0")) continue;
                byte[] macBytes = nif.getHardwareAddress();
                if (macBytes == null) {
                    return null;
                }

                StringBuilder res1 = new StringBuilder();
                for (byte b : macBytes) {
                    res1.append(String.format("%02x:", b));
                }

                if (res1.length() > 0) {
                    res1.deleteCharAt(res1.length() - 1);
                }
                return res1.toString();
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return null;
    }

    public static String getEthMac() {
        String mac = null;
        mac = getEthMacAddress();
        if (TextUtils.isEmpty(mac)) {
            mac = getEthMacFromHardware();
        }
        return toShrinkMac(mac);
    }

    public static String getBluetoothMac(Context context) {
        String mac = null;
        BluetoothAdapter ba = BluetoothAdapter.getDefaultAdapter();
        if (ba != null) {
            mac = ba.getAddress();
        }
        return toShrinkMac(mac);
    }

    public static String toShrinkMac(String mac) {
        String out = "";
        if (!TextUtils.isEmpty(mac)) {
            out = mac.trim().replaceAll("[^0-9A-Fa-f]", "").toLowerCase();
        }
        return out;
    }

    public static String getMacDigest() {
        if (MAC_DIGEST != null)
            return MAC_DIGEST;

        String mac = getEthMac();
        if (StringUtils.isEmpty(mac))
            return "";

//        MAC_DIGEST = MD5Util.getStringMD5_16(mac);
        MAC_DIGEST = CryptoUtils.getDidAlias(mac);

        return MAC_DIGEST;
    }
}
