package com.kezong.fataar;

import com.android.build.api.transform.DirectoryInput;
import com.android.build.api.transform.Format;
import com.android.build.api.transform.JarInput;
import com.android.build.api.transform.QualifiedContent;
import com.android.build.api.transform.Status;
import com.android.build.api.transform.Transform;
import com.android.build.api.transform.TransformInput;
import com.android.build.api.transform.TransformInvocation;
import com.android.build.api.transform.TransformOutputProvider;
import com.android.build.gradle.internal.pipeline.TransformManager;
import com.google.common.collect.ImmutableSet;

import org.apache.commons.io.FileUtils;
import org.gradle.api.Project;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.function.Consumer;
import java.util.stream.Collectors;

import javassist.CannotCompileException;
import javassist.ClassPool;
import javassist.CtClass;
import javassist.NotFoundException;
import javassist.bytecode.ClassFile;
import javassist.bytecode.ConstPool;
import kotlin.io.FilesKt;

/**
 * com.sdk.R
 * |-- com.lib1.R
 * |-- com.lib2.R
 * <p>
 * rename com.lib1.R and com.lib2.R to com.sdk.R
 */
public class RClassesTransform extends Transform {

    private static final int CPU_COUNT = Runtime.getRuntime().availableProcessors();

    private final ExecutorService executor = Executors.newFixedThreadPool(CPU_COUNT + 1);

    private final List<Future<?>> futures = new ArrayList<>();

    private final Project project;

    private final Map<String, String> targetPackageMap = new HashMap<>();

    private final Map<String, Collection<String>> libraryPackageMap = new HashMap<>();

    public RClassesTransform(final Project project) {
        this.project = project;
    }

    /**
     * Different variants have different package names.
     * So targetPackageName must set after evaluate
     *
     * @param variantName   variant name
     * @param targetPackage main module's package name
     */
    public void putTargetPackage(String variantName, String targetPackage) {
        targetPackageMap.put(variantName, targetPackage);
    }

    /**
     * library packages name must set after exploded task perform
     *
     * @param variantName     variant name
     * @param libraryPackages sub module's package name, read from AndroidManifest.xml
     */
    public void putLibraryPackages(String variantName, Collection<String> libraryPackages) {
        libraryPackageMap.put(variantName, libraryPackages);
    }

    @Override
    public String getName() {
        return "transformR";
    }

    @Override
    public Set<QualifiedContent.ContentType> getInputTypes() {
        return TransformManager.CONTENT_CLASS;
    }

    @Override
    public Set<? super QualifiedContent.Scope> getScopes() {
        return ImmutableSet.of(QualifiedContent.Scope.PROJECT);
    }

    @Override
    public boolean isIncremental() {
        return true;
    }

    @Override
    public void transform(TransformInvocation transformInvocation) throws InterruptedException, IOException {
        long startTime = System.currentTimeMillis();
        Map<String, String> transformTable = buildTransformTable(transformInvocation.getContext().getVariantName());
        final boolean isIncremental = transformInvocation.isIncremental() && this.isIncremental();
        final TransformOutputProvider outputProvider = transformInvocation.getOutputProvider();

        if (!isIncremental) {
            outputProvider.deleteAll();
        }

        try {
            for (final TransformInput input : transformInvocation.getInputs()) {
                for (final JarInput jarInput : input.getJarInputs()) {
                    Status status = jarInput.getStatus();
                    File desDir = outputProvider.getContentLocation(jarInput.getName(), jarInput.getContentTypes(), jarInput.getScopes(), Format.DIRECTORY);
                    File srcDir = new File(transformInvocation.getContext().getTemporaryDir().getAbsolutePath() + File.separator + RClassesTransform.class.getSimpleName() + File.separator + jarInput.getName());
                    if (isIncremental && status != Status.NOTCHANGED) {
                        FileUtils.deleteQuietly(desDir);
                    }
                    if (!isIncremental || status == Status.ADDED || status == Status.CHANGED) {
                        FileUtils.deleteQuietly(srcDir);
                        project.copy(copySpec -> {
                            copySpec.from(project.zipTree(jarInput.getFile()));
                            copySpec.into(srcDir);
                        });
                        Map<File, Status> changedFiles = new HashMap<>();
                        Files.walk(srcDir.toPath()).filter(Files::isRegularFile).map(Path::toFile).forEach(file -> changedFiles.put(file, Status.CHANGED));
                        handleClassDir(transformTable, desDir, srcDir, changedFiles);
                    }
                }

                final File outputDir = outputProvider.getContentLocation("classes", getOutputTypes(), getScopes(), Format.DIRECTORY);
                for (final DirectoryInput directoryInput : input.getDirectoryInputs()) {
                    final File srcDirectoryFile = directoryInput.getFile();
                    Map<File, Status> changedFiles = directoryInput.getChangedFiles();
                    handleClassDir(transformTable, outputDir, srcDirectoryFile, changedFiles);
                }
            }
        } catch (NotFoundException e) {
            throw new RuntimeException(e);
        }


        for (Future<?> future : futures) {
            try {
                future.get();
            } catch (ExecutionException e) {
                e.printStackTrace();
            }
        }

        futures.clear();

        long endTime = System.currentTimeMillis();
        project.getLogger().info("the task cost "
                + (endTime - startTime)
                + "ms");
    }

    private void handleClassDir(Map<String, String> transformTable, File outputDir, File directoryFile, Map<File, Status> changedFiles) throws NotFoundException, IOException {
        final ClassPool classPool = new ClassPool();
        classPool.insertClassPath(directoryFile.getAbsolutePath());
        for (final File originalClassFile : getChangedClassesList(directoryFile, outputDir, changedFiles)) {
            if (!originalClassFile.getPath().endsWith(".class")) {
                FileUtils.copyFile(originalClassFile, new File(outputDir, FilesKt.relativeTo(originalClassFile, directoryFile).getPath()), false);
                continue; // ignore anything that is not class file
            }

            Future<?> submit = executor.submit(() -> {
                try {
                    File relative = FilesKt.relativeTo(originalClassFile, directoryFile);
                    String className = filePathToClassname(relative);
                    final CtClass ctClass = classPool.get(className);
                    if (transformTable != null) {
                        ClassFile classFile = ctClass.getClassFile();
                        ConstPool constPool = classFile.getConstPool();
                        constPool.renameClass(transformTable);
                    }
                    ctClass.writeFile(outputDir.getAbsolutePath());
                } catch (CannotCompileException | NotFoundException | IOException e) {
                    e.printStackTrace();
                }
            });

            futures.add(submit);
        }
    }

    private Map<String, String> buildTransformTable(String variantName) {
        String targetPackage = targetPackageMap.get(variantName);
        Collection<String> libraryPackages = libraryPackageMap.get(variantName);
        if (targetPackage == null || libraryPackages == null) {
            return null;
        }

        final List<String> resourceTypes = Arrays.asList("anim", "animator", "array", "attr", "bool", "color", "dimen",
                "drawable", "font", "fraction", "id", "integer", "interpolator", "layout", "menu", "mipmap", "navigation",
                "plurals", "raw", "string", "style", "styleable", "transition", "xml");

        HashMap<String, String> map = new HashMap<>();
        for (String resource : resourceTypes) {
            String targetClass = targetPackageMap.get(variantName).replace(".", "/") + "/R$" + resource;
            for (String libraryPackage : libraryPackageMap.get(variantName)) {
                String fromClass = libraryPackage.replace(".", "/") + "/R$" + resource;
                map.put(fromClass, targetClass);
            }
        }

        return map;
    }

    private List<File> getChangedClassesList(File sourceDir, File desDir, Map<File, Status> changedFiles) throws IOException {
        if (changedFiles.isEmpty()) {
            // we're in non incremental mode
            return Files.walk(sourceDir.toPath())
                    .filter(Files::isRegularFile)
                    .map(Path::toFile)
                    .collect(Collectors.toList());
        } else {
            changedFiles.entrySet().stream()
                    .filter(it -> it.getValue() == Status.REMOVED)
                    .forEach(it -> FileUtils.deleteQuietly(new File(desDir, FilesKt.relativeTo(it.getKey(), sourceDir).getPath())));

            return changedFiles.entrySet().stream()
                    .filter(it -> it.getValue() == Status.ADDED || it.getValue() == Status.CHANGED)
                    .map(Map.Entry::getKey)
                    .filter(File::isFile)
                    .collect(Collectors.toList());
        }
    }

    private String filePathToClassname(File file) {
        // com/classify/module/a.class -> com.classify.module.a.class -> comify.module.a is not expected
        // so must be replace .class first
        return file.getPath().replace(".class", "")
                .replace("/", ".")
                .replace("\\", ".");
    }
}
