package com.cv.media.lib.plugin.arouter

import com.android.SdkConstants
import com.android.build.api.transform.*
import com.android.build.gradle.AppPlugin
import com.android.build.gradle.internal.pipeline.TransformManager
import com.cv.media.lib.plugin.ZipUtils
import com.google.common.collect.ImmutableSet
import com.google.common.io.Files
import com.cv.media.lib.plugin.Logger
import org.apache.commons.io.FileUtils
import org.gradle.api.Project
import org.objectweb.asm.ClassReader
import org.objectweb.asm.ClassVisitor
import org.objectweb.asm.Opcodes

import java.util.jar.JarEntry
import java.util.jar.JarFile

/**
 Arouter 针对插件化框架
 @Author Damon
 */
class ArouterInitTransform extends Transform {
    Project project
    ArrayList<ScanSetting> registerList
    String nameInterface
    String nameProjectPackage
    static boolean leftSlash = File.separator == '/'


    ArouterInitTransform(Project project) {
        this.project = project
        registerList = [new ScanSetting('IRouteRoot'), new ScanSetting('IInterceptorGroup'), new ScanSetting('IProviderGroup'), new ScanSetting('IRouteGroup')]
        String prefix = project.getBuildFile().getParentFile().getAbsolutePath() + File.separator + SdkConstants.FD_SOURCES + File.separator + SdkConstants.FD_MAIN + File.separator + SdkConstants.FD_JAVA + File.separator;
        File javaDir = new File(prefix)
        List<File> files = []
        javaDir.eachFileRecurse() {
            files.add(it)
        }
        int fileIndex = -1
        files.eachWithIndex { File file, int index ->
            File[] subFileslist
            if (fileIndex < 0 && file.isDirectory() && (subFileslist = file.listFiles()).length >= 1) {
                if (subFileslist.length == 1 && subFileslist[0].isDirectory()) return
                fileIndex = index
                return
            }

            if (index == files.size() - 1 && fileIndex < 0) {
                fileIndex = index
            }
        }
        nameProjectPackage = files[fileIndex].getAbsolutePath().replace(prefix, '')
        nameProjectPackage = !leftSlash ? nameProjectPackage.replace('\\', '/') : nameProjectPackage
        nameInterface = ScanSetting.APPLICATION_INTTERFACE_PACKAGE_NAME + 'IARouterInitApplication'
    }

    @Override
    String getName() {
        ArouterInitTransform.simpleName
    }

    @Override
    Set<QualifiedContent.ContentType> getInputTypes() {
        TransformManager.CONTENT_CLASS
    }

    @Override
    Set<? super QualifiedContent.Scope> getScopes() {
        project.plugins.hasPlugin(AppPlugin) ? ImmutableSet.of(QualifiedContent.Scope.PROJECT, QualifiedContent.Scope.SUB_PROJECTS) : TransformManager.PROJECT_ONLY
    }

    @Override
    boolean isIncremental() {
        return false
    }


    @Override
    void transform(TransformInvocation v) throws TransformException, InterruptedException, IOException {
        super.transform(v)
        /**
         * 这里, 有个场景 前一个transform有可能将本工程的代码压缩成一个.jar
         * 所以jarInput也要去考虑匹配到初始化Application类
         */
        Logger.startCount()
        try {
            v.outputProvider.deleteAll()
        } catch (Exception e) {

        }
        String initClassFilePath
        String initClassFileEntryName
        v.inputs.each { TransformInput input ->
            input.jarInputs.each { JarInput jarInput ->
                File src = jarInput.file
                def jarFile = new JarFile(src)
                Enumeration enumeration = jarFile.entries()
                while (enumeration.hasMoreElements()) {
                    JarEntry jarEntry = (JarEntry) enumeration.nextElement()
                    if (!jarEntry.isDirectory() && jarEntry.name.endsWith(".class")) {
                        try {
                            collectARouterClass(jarFile.getInputStream(jarEntry), jarEntry.getName())
                        } catch (Exception e) {
                            //有的jar包, manifestVerification过不去
                            break
                        }
                        if (initClassFileEntryName == null && collectARouterInitClass(jarFile.getInputStream(jarEntry), jarEntry.getName())) {
                            initClassFileEntryName = jarEntry.name
                        }
                    }
                }

                File dest;
                if (initClassFileEntryName != null && initClassFilePath == null) {
                    dest = v.outputProvider.getContentLocation(jarInput.name, jarInput.contentTypes, jarInput.scopes, Format.DIRECTORY)
                    Files.createParentDirs(dest);
                    ZipUtils.unZip(src, dest.path)
                    initClassFilePath = dest.absolutePath + File.separator + initClassFileEntryName
                } else {
                    dest = v.outputProvider.getContentLocation(jarInput.name, jarInput.contentTypes, jarInput.scopes, Format.JAR)
                    FileUtils.copyFile(src, dest)
                }
                jarFile.close()
            }

            input.directoryInputs.each { DirectoryInput directoryInput ->
                File dest = v.outputProvider.getContentLocation(directoryInput.name, directoryInput.contentTypes, directoryInput.scopes, Format.DIRECTORY)
                String root = directoryInput.file.absolutePath
                if (!root.endsWith(File.separator))
                    root += File.separator
                directoryInput.file.eachFileRecurse { File file ->
                    def path = file.absolutePath.replace(root, '')
                    if (!leftSlash) {
                        path = path.replace('\\', "/")
                    }

                    if (file.isFile()) {
                        InputStream fileStream = new FileInputStream(file)
                        collectARouterClass(fileStream, path)
                        if (initClassFilePath == null && collectARouterInitClass(new FileInputStream(file), path)) {
                            initClassFileEntryName = file.name
                            initClassFilePath = dest.absolutePath + File.separator + path
                        }
                    }
                }
                // copy to dest
                FileUtils.copyDirectory(directoryInput.file, dest)
            }
        }

        if (initClassFilePath != null) {
            injectARouterInitClass(new File(initClassFilePath))
        } else {
            Logger.log(ArouterInitTransform.simpleName + " 未找到初始化类文件!")
        }
        Logger.endCount(ArouterInitTransform.simpleName + " 完成")
    }

    boolean maybeApplication(String path) {
        return path.contains(nameProjectPackage) && path.endsWith('Application.class')
    }

    void collectARouterClass(InputStream inputStream, String path) {
        if (path.contains(ScanSetting.AROUTER_PACKAGE_PREFIX)) scanArouterClass(inputStream)
    }

    boolean collectARouterInitClass(InputStream inputStream, String path) {
        return maybeApplication(path) && scanInitClass(inputStream)
    }

    void injectARouterInitClass(File file) {
        Logger.log(ArouterInitTransform.simpleName + " 找到初始化类文件: " + file.getName())
        ByteCodeGenerator.injectCode(file, registerList)
    }

    boolean scanArouterClass(InputStream inputStream) {
        ClassReader cr = new ClassReader(inputStream)
        ScanClassVisitor cv = new ScanClassVisitor(Opcodes.ASM5)
        cr.accept(cv, ClassReader.EXPAND_FRAMES)
        inputStream.close()
        return cv.isFinded
    }


    boolean scanInitClass(InputStream inputStream) {
        ClassReader cr = new ClassReader(inputStream)
        InitClassVisitor cv = new InitClassVisitor(Opcodes.ASM5)
        cr.accept(cv, ClassReader.EXPAND_FRAMES)
        inputStream.close()
        return cv.isFinded
    }

    //找到 相应接口的实现类, 比如 IRouteRoot IInterceptorGroup
    class ScanClassVisitor extends ClassVisitor {
        boolean isFinded = false

        ScanClassVisitor(int api) {
            super(api)
        }

        ScanClassVisitor(int api, ClassVisitor cv) {
            super(api, cv)
        }

        void visit(int version, int access, String name, String signature,
                   String superName, String[] interfaces) {
            super.visit(version, access, name, signature, superName, interfaces)
            registerList.each { ext ->
                if (ext.interfaceName && interfaces != null) {
                    //当前visit的类的所有接口
                    interfaces.each { itName ->
                        if (itName == ext.interfaceName) {
                            //fix repeated inject init code when Multi-channel packaging
                            if (!ext.classList.contains(name)) {
                                isFinded = true
                                ext.classList.add(name)
                            }
                        }
                    }
                }
            }
        }
    }

    class InitClassVisitor extends ClassVisitor {
        boolean isFinded = false

        public InitClassVisitor(int api) {
            super(api)
        }

        public InitClassVisitor(int api, ClassVisitor cv) {
            super(api, cv)
        }

        boolean IsFinded() {
            return isFinded
        }

        @Override
        void visit(int version, int access, String name, String signature, String superName, String[] interfaces) {
            super.visit(version, access, name, signature, superName, interfaces)
            if ((access & 0x0400) != 0x0400) {
                interfaces.each { String I ->
                    if (!isFinded && I == nameInterface) {
                        isFinded = true
                    }
                }
            }
        }
    }
}
