package com.cv.media.lib.plugin.packing


import com.android.build.gradle.api.ApplicationVariant
import com.android.builder.model.ProductFlavor
import com.android.builder.model.SigningConfig
import com.android.zipflinger.ZipArchive
import com.android.zipflinger.ZipSource
import com.cv.media.lib.plugin.Constants
import com.cv.media.lib.plugin.Logger
import com.cv.media.lib.plugin.ZipUtils
import org.apache.commons.io.FileUtils
import org.gradle.api.DefaultTask
import org.gradle.api.GradleException
import org.gradle.api.tasks.Input
import org.gradle.api.tasks.TaskAction
import org.gradle.internal.impldep.org.apache.commons.compress.archivers.zip.ZipUtil

import java.util.zip.ZipEntry
import java.util.zip.ZipFile

/**
 * 自定义打包任务
 * 职责:
 * @author Damon
 */
class ApkPackingTask extends DefaultTask {
    @Input
    ApplicationVariant mVariant
    @Input
    File mOutputDir
    @Input
    List<String> mChannels
    @Input
    String mTerminal
    @Input
    String mDate

    File toolsDir

    ApkPackingTask() {
        group "打包"
    }

    @TaskAction
    void packing() {
        //准备好工具
        mOutputDir.deleteDir()
        mOutputDir.mkdirs()
        unZipTools()
        //转移->加固->对齐->签名->渠道
        File channelsDir = new File(mOutputDir, "channels")
        channelsDir.mkdirs()
        File parentApk = reSignApk(reAlignApk(makeProtectedApk(moveArtifact())))
        //规整结构
        File originDir = new File(mOutputDir, "origin")
        originDir.mkdirs()
        def platform = mVariant.productFlavors.find { it.dimension == Constants.projectPlatformFlavor }.name
        def env = mVariant.productFlavors.find { it.dimension == Constants.projectEnvFlavor }.name
        def version = mVariant.versionName
        def terminal = mTerminal
        def date = mDate
        //包名-命名规范 : mfc_sourceId_terminal_enviroment_versionName_yearMonthDate.apk
        File newParentApk = new File(originDir, "${platform}_${terminal}_${env}_${version}_${date}.apk")
        parentApk.renameTo(newParentApk)
        //衍生渠道包
        fissionChannelApks(channelsDir, newParentApk, platform, env, version, terminal, date)
        //清除冗余文件
        mOutputDir.eachFile {
            if (!it.isDirectory()) it.delete()
        }
    }

    File unZipTools() {
        toolsDir = new File(project.rootProject.getBuildDir(), "${ApkPackingTask.class.getSimpleName()}")
        if (toolsDir.exists()) {
            def toolsName = [
                    'apksigner.jar',
                    'channel_cli.jar',
                    'encryptiontool-1.2_forsoapi.jar',
            ]

            toolsDir.eachFile {
                if (toolsName.contains(it.name)) toolsName.remove(it.name)
            }

            if (toolsName.isEmpty()) {
                //tools already existed
                return
            }
            toolsDir.deleteDir()
            toolsDir.mkdirs()
        } else {
            toolsDir.mkdirs()
        }
        ZipUtils.unZip(new File(ApkPackingTask.class.getProtectionDomain().getCodeSource().getLocation().path), toolsDir.path)
        toolsDir.eachFile {
            if (it.isDirectory()) it.deleteDir()
        }
    }

    //转移编译产物
    File moveArtifact() {
        logs("产物转移开始")
        FileUtils.deleteDirectory(mOutputDir)
        mOutputDir.mkdirs()
        //复制Apk文件夹
        File apkDir = new File(project.getBuildDir(), "outputs/apk/${mVariant.getDirName()}")
        FileUtils.copyDirectory(apkDir, mOutputDir)
        //复制Mapping文件夹 if need
        if (mVariant.getBuildType().name == "release") {
            File mappingSrcDir = new File(project.getBuildDir(), "outputs/mapping/${mVariant.getName()}")
            File mappingDestDir = new File(mOutputDir, "mapping")
            mappingDestDir.mkdirs()
            FileUtils.copyDirectory(mappingSrcDir, mappingDestDir)
        }
        logs("打包产物 全部已经转移至:${mOutputDir.getPath()}")
        logs("产物转移结束")
        return new File(mOutputDir, "${mVariant.outputs.first().outputFile.name}")
    }

    //制作 加固保护的Apk
    File makeProtectedApk(File apkOriginFile) {
        def args = [
                "itman@valorosoltd.com",
                "${apkOriginFile.path}",
                "${mOutputDir.path}",
                "null",
                "458"
        ]
        logs("加固 开始...")
        Logger.startCount()
        execute("java -jar ${toolsDir.path}${File.separator}encryptiontool-1.2_forsoapi.jar ${args.join(" ")}", "爱加密")
        Logger.endCount("加固任务")
        logs("加固 结束...")

        List<File> apks = mOutputDir.listFiles().findAll {
            it.name.contains("enc_unsigned.apk")
        }


        if (apks.size() != 1) {
            throw new GradleException("Apk加固异常: 找到多个匹配apk")
        }
        File protectApkFile = new File(mOutputDir, "${apkOriginFile.name.replace(".apk", "_protected.apk")}")
        logs("更改加固后apk:${apks[0].name}为: $protectApkFile.name")
        apks[0].renameTo(protectApkFile)
        if (!protectApkFile.exists()) throw new GradleException("Apk加固失败: 加固包不存在")

        logs("开始检查加固后apk: $protectApkFile.name")
        ZipFile zipFileOfPAF = new ZipFile(protectApkFile)
        Enumeration<?> entries = zipFileOfPAF.entries()
        boolean onlyOneDex = true
        while (entries.hasMoreElements()) {
            ZipEntry entry = (ZipEntry) entries.nextElement()
            if (entry.name.startsWith("classes") && entry.name.endsWith(".dex") && entry.name != "classes.dex") {
                onlyOneDex = false
                break
            }
        }

        if (!onlyOneDex) {
            throw new GradleException("Apk加固异常: 不止一个.dex")
        }

        logs("加固成功: $protectApkFile.name")
        return protectApkFile
    }

    //重新字节对齐
    File reAlignApk(File protectApkFile) {
        logs("对齐 开始...")
        Logger.startCount()
        File byteAlignApkFile = new File(mOutputDir, "${protectApkFile.name.replace(".apk", "_align.apk")}")
        def zipSource = new ZipSource(protectApkFile)
        zipSource.entries().each {
            zipSource.select(it.value.name, it.value.name, ZipSource.COMPRESSION_NO_CHANGE, 4L)
        }
        ZipArchive zipArchive = new ZipArchive(byteAlignApkFile)
        zipArchive.add(zipSource)
        zipArchive.close()
        Logger.endCount("对齐")
        logs("对齐 结束...")
        if (!byteAlignApkFile.exists()) throw new GradleException("Apk对齐失败")
        return byteAlignApkFile
    }

    //重新签名
    File reSignApk(File alignedApkFile) {
        logs("签名开始")
        File finalApkFile = new File(mOutputDir, "${alignedApkFile.name.replace(".apk", "_signed.apk")}")
        SigningConfig signingConfig = ((ApplicationVariant) mVariant).getSigningConfig()
        def args = [
                "sign",
                "-v",
                "--ks",
                "${signingConfig.storeFile.path}",
                "--ks-key-alias",
                signingConfig.keyAlias,
                "--ks-pass",
                "pass:${signingConfig.storePassword}",
                "--key-pass",
                "pass:${signingConfig.keyPassword}",
                "--out",
                "${finalApkFile.path}",
                alignedApkFile.path,
        ]
        execute("java -jar ${toolsDir.path}${File.separator}apksigner.jar ${args.join(" ")}", "signapk")
        logs("签名结束")
        if (!finalApkFile.exists()) throw new GradleException("Apk签名失败")
        logs("删除对齐包: ${alignedApkFile.path}")
        alignedApkFile.delete()
        return finalApkFile
    }

    //分裂 频道包
    void fissionChannelApks(File channelsDir, File finalApkFile, String platform, String env, String version, String terminal, String date) {
        logs("制作渠道包开始...")
        Logger.startCount()
        File channelApkDir = channelsDir
        channelApkDir.mkdirs()
        execute("java -jar ${toolsDir}${File.separator}channel_cli.jar put -c ${mChannels.join(",")} ${finalApkFile.path} ${channelApkDir.path}", "渠道包")
        channelsDir.eachFile { File channelApk ->
            String channel = mChannels.find { String channel ->
                channelApk.name.contains(channel)
            }
            channelApk.renameTo(new File(channelsDir, "${platform}_${channel}_${terminal}_${env}_${version}_${date}.apk"))
        }
        Logger.endCount("渠道包")
        logs("制作渠道包结束...")
    }

    void execute(String cmd, String tag) {
        logs("tag:${tag} 执行命令: ${cmd}")
        Process proc = cmd.execute()
        Reader reader = proc.in.newReader("utf-8")
        Reader errReader = proc.err.newReader("utf-8")
        String tmp
        while ((tmp = reader.readLine()) != null || (tmp = errReader.readLine()) != null) {
            logs("${tag}:: " + tmp)
        }
        reader.close()
        errReader.close()
    }

    void logs(String content) {
        println("${ApkPackingTask.simpleName}--> ${content}")
    }
}