package com.cv.media.lib.plugin.mvp

import com.android.SdkConstants
import com.android.build.api.transform.*
import com.android.build.gradle.LibraryPlugin
import com.android.build.gradle.internal.pipeline.TransformManager
import com.cv.media.lib.plugin.Logger
import org.apache.commons.codec.digest.DigestUtils
import org.apache.commons.io.FileUtils
import org.gradle.api.Project
import org.objectweb.asm.ClassReader
import org.objectweb.asm.ClassVisitor
import org.objectweb.asm.Opcodes

import java.util.jar.JarEntry
import java.util.jar.JarFile

/**
 MVP 针对插件化框架
 Transform无法debug, 只能log调试
 @Author Damon
 */
class MVPInitTransform extends Transform {
    Project project
    ArrayList<ScanSetting> registerList
    String nameInterface
    File fileContainsInitClass
    String nameInitClass
    String nameProjectPackage
    static boolean leftSlash = File.separator == '/'


    MVPInitTransform(Project project) {
        this.project = project
        registerList = [new ScanSetting('MVPInfoFiller')]
        String prefix = project.getBuildFile().getParentFile().getAbsolutePath() + File.separator + SdkConstants.FD_SOURCES + File.separator + SdkConstants.FD_MAIN + File.separator + SdkConstants.FD_JAVA + File.separator;
        File javaDir = new File(prefix)
        List<File> files = []
        javaDir.eachFileRecurse() {
            files.add(it)
        }
        int fileIndex = -1
        files.eachWithIndex { File file, int index ->
            File[] subFileslist
            if (fileIndex < 0 && file.isDirectory() && (subFileslist = file.listFiles()).length >= 1) {
                if (subFileslist.length == 1 && subFileslist[0].isDirectory()) return
                fileIndex = index
                return
            }

            if (index == files.size() - 1 && fileIndex < 0) {
                fileIndex = index
            }
        }
        nameProjectPackage = files[fileIndex].getAbsolutePath().replace(prefix, '')
        nameProjectPackage = !leftSlash ? nameProjectPackage.replace('\\', '/') : nameProjectPackage
        nameInterface = ScanSetting.INTERFACE_PACKAGE_PREFIX + 'IMVPInitApplication'
    }

    @Override
    String getName() {
        MVPInitTransform.simpleName
    }

    @Override
    Set<QualifiedContent.ContentType> getInputTypes() {
        TransformManager.CONTENT_CLASS
    }

    @Override
    Set<? super QualifiedContent.Scope> getScopes() {
        TransformManager.PROJECT_ONLY
    }

    @Override
    boolean isIncremental() {
        return false
    }

    @Override
    void transform(TransformInvocation v) throws TransformException, InterruptedException, IOException {
        super.transform(v)
        Logger.startCount()
        v.outputProvider.deleteAll()
        v.inputs.each { TransformInput input ->
            input.jarInputs.each { JarInput jarInput ->
                String destName = jarInput.name
                def hexName = DigestUtils.md5Hex(jarInput.file.absolutePath)
                if (destName.endsWith(SdkConstants.DOT_JAR)) {
                    destName = destName.substring(0, destName.length() - 4)
                }
                File src = jarInput.file
                File dest = v.outputProvider.getContentLocation(destName + "_" + hexName, jarInput.contentTypes, jarInput.scopes, Format.JAR)
                FileUtils.copyFile(src, dest)
                def jarFile = new JarFile(src)
                Enumeration enumeration = jarFile.entries()
                while (enumeration.hasMoreElements()) {
                    JarEntry jarEntry = (JarEntry) enumeration.nextElement()
                    if (!jarEntry.isDirectory() && jarEntry.getName().startsWith(ScanSetting.FILLER_PACKAGE_PREFIX)) {
                        InputStream inputEntry = jarFile.getInputStream(jarEntry)
                        scanInfoFiller(inputEntry)
                        inputEntry.close()
                    }
                }
                jarFile.close()
            }

            input.directoryInputs.each { DirectoryInput directoryInput ->
                File dest = v.outputProvider.getContentLocation(directoryInput.name, directoryInput.contentTypes, directoryInput.scopes, Format.DIRECTORY)
                String root = directoryInput.file.absolutePath
                if (!root.endsWith(File.separator))
                    root += File.separator
                directoryInput.file.eachFileRecurse { File file ->
                    def path = file.absolutePath.replace(root, '')
                    if (!leftSlash) {
                        path = path.replace('\\', "/")
                    }

                    if (file.isFile()) {
                        if (maybeFillerClass(path) && scanInfoFiller(new FileInputStream(file))) {

                        } else if (maybeApplication(path) && scanInitClass(new FileInputStream(file))) {
                            fileContainsInitClass = new File(dest.absolutePath + File.separator + path)
                        }
                    }
                }

                // copy to dest
                FileUtils.copyDirectory(directoryInput.file, dest)
            }
        }

        if (fileContainsInitClass) {
            Logger.log(MVPInitTransform.simpleName + " 找到初始化类")
            ByteCodeGenerator.injectCode(fileContainsInitClass, registerList)
        } else {
            Logger.log(MVPInitTransform.simpleName + " 未找到初始化类")
        }
        Logger.endCount(MVPInitTransform.simpleName + " 完成")
    }

    boolean maybeApplication(String path) {
        return path.contains(nameProjectPackage) && path.endsWith('Application.class')
    }

    boolean maybeFillerClass(String path) {
        return path.contains(ScanSetting.FILLER_PACKAGE_PREFIX)
    }

    boolean scanInfoFiller(InputStream inputStream) {
        ClassReader cr = new ClassReader(inputStream)
        ScanClassVisitor cv = new ScanClassVisitor(Opcodes.ASM5)
        cr.accept(cv, ClassReader.EXPAND_FRAMES)
        inputStream.close()
        return cv.isFinded
    }


    boolean scanInitClass(InputStream inputStream) {
        ClassReader cr = new ClassReader(inputStream)
        InitClassVisitor cv = new InitClassVisitor(Opcodes.ASM5)
        cr.accept(cv, ClassReader.EXPAND_FRAMES)
        inputStream.close()
        return cv.isFinded
    }

    //找到 相应接口的实现类, 比如 IRouteRoot IInterceptorGroup
    class ScanClassVisitor extends ClassVisitor {
        boolean isFinded = false

        ScanClassVisitor(int api) {
            super(api)
        }

        ScanClassVisitor(int api, ClassVisitor cv) {
            super(api, cv)
        }

        void visit(int version, int access, String name, String signature,
                   String superName, String[] interfaces) {
            super.visit(version, access, name, signature, superName, interfaces)
            registerList.each { ext ->
                if (ext.interfaceName && interfaces != null) {
                    //当前visit的类的所有接口
                    interfaces.each { itName ->
                        if (itName == ext.interfaceName) {
                            //fix repeated inject init code when Multi-channel packaging
                            if (!ext.classList.contains(name)) {
                                isFinded = true
                                ext.classList.add(name)
                            }
                        }
                    }
                }
            }
        }
    }

    class InitClassVisitor extends ClassVisitor {
        boolean isFinded = false

        public InitClassVisitor(int api) {
            super(api)
        }

        public InitClassVisitor(int api, ClassVisitor cv) {
            super(api, cv)
        }

        boolean IsFinded() {
            return isFinded
        }

        @Override
        void visit(int version, int access, String name, String signature, String superName, String[] interfaces) {
            super.visit(version, access, name, signature, superName, interfaces)
            interfaces.each { String I ->
                if (!isFinded && I == nameInterface) {
                    isFinded = true
                }
            }
        }
    }
}
