package com.cv.media.lib.plugin.module

import com.android.annotations.NonNull
import com.android.builder.internal.ClassFieldImpl
import com.android.builder.model.ClassField
import com.cv.media.lib.plugin.Logger
import com.google.common.base.Charsets
import com.google.common.collect.Lists
import com.google.common.io.Closer
import com.squareup.javawriter.JavaWriter

import javax.lang.model.element.Modifier

/**
 Config类型文件生成器
 @Author Damon
 */
class ConfigClassGenerator {
    final File genFolder

    final String className

    final String fileName

    String packageName

    static final Set<Modifier> PUBLIC_FINAL = EnumSet.of(Modifier.PUBLIC, Modifier.FINAL)
    static final Set<Modifier> PUBLIC_STATIC_FINAL =
            EnumSet.of(Modifier.PUBLIC, Modifier.STATIC, Modifier.FINAL)

    final List<ClassField> mFields = Lists.newArrayList()

    ConfigClassGenerator(@NonNull String pkgName, @NonNull String clzName, @NonNull File genFolder) {
        this.genFolder = genFolder
        this.className = clzName
        this.fileName = className + ".java"
        this.packageName = pkgName
    }

    ConfigClassGenerator addField(
            @NonNull String type, @NonNull String name, @NonNull String value) {
        mFields.add(new ClassFieldImpl(type, name, value))
        return this
    }

    /**
     * Returns a File representing where the Config class will be.
     */
    File getFolderPath() {
        return new File(genFolder, packageName.replace(".", File.separator))
    }

    File getConfigFile() {
        File folder = getFolderPath()
        return new File(folder, fileName)
    }

    /**
     * Generates the BuildConfig class.
     */
    void generate() throws IOException {
        File pkgFolder = getFolderPath()
        Logger.log("moduleConfigPath:" + pkgFolder.getAbsolutePath())
        if (!pkgFolder.isDirectory()) {
            if (!pkgFolder.mkdirs()) {
                throw new RuntimeException("Failed to create " + pkgFolder.getAbsolutePath())
            }
        }

        File configFile = new File(pkgFolder, fileName)

        Closer closer = Closer.create()
        try {
            FileOutputStream fos = closer.register(new FileOutputStream(configFile))
            OutputStreamWriter out = closer.register(new OutputStreamWriter(fos, Charsets.UTF_8))
            JavaWriter writer = closer.register(new JavaWriter(out))

            writer.emitJavadoc("Automatically generated file. DO NOT MODIFY")
                    .emitPackage(packageName)
                    .beginType(className, "class", PUBLIC_FINAL)

            for (ClassField field : mFields) {
                emitClassField(writer, field)
            }

            writer.endType()
        } catch (Throwable e) {
            throw closer.rethrow(e)
        } finally {
            closer.close()
        }
    }

    private static void emitClassField(JavaWriter writer, ClassField field) throws IOException {
        String documentation = field.getDocumentation()
        if (!documentation.isEmpty()) {
            writer.emitJavadoc(documentation)
        }
        for (String annotation : field.getAnnotations()) {
            writer.emitAnnotation(annotation)
        }
        writer.emitField(
                field.getType(),
                field.getName(),
                PUBLIC_STATIC_FINAL,
                field.getValue())
    }

}