package com.cv.media.lib.plugin

import com.android.build.gradle.AppExtension
import com.android.build.gradle.AppPlugin
import com.android.build.gradle.BaseExtension
import com.android.build.gradle.LibraryExtension
import com.android.build.gradle.api.ApplicationVariant
import com.android.build.gradle.api.BaseVariant
import com.android.build.gradle.api.LibraryVariant
import com.android.build.gradle.internal.api.ApkVariantOutputImpl
import com.android.build.gradle.internal.dsl.SigningConfig
import com.android.build.gradle.tasks.GenerateBuildConfig
import com.android.builder.model.ProductFlavor
import com.cv.media.lib.plugin.packing.BaiduApkPackingTask
import com.cv.media.lib.plugin.packing.PrivateServiceApkPackingTask
import com.google.gson.Gson
import com.cv.media.lib.plugin.config.VariantBuildConfiguration
import com.cv.media.lib.plugin.dependencies.DependenciesInjector
import com.cv.media.lib.plugin.packing.PublicServiceApkPackingTask
import com.cv.media.lib.plugin.module.GenerateOriginModuleInfo
import com.cv.media.lib.plugin.module.ModuleConfig
import org.gradle.api.Action
import org.gradle.api.DefaultTask
import org.gradle.api.GradleException
import org.gradle.api.JavaVersion
import org.gradle.api.Plugin
import org.gradle.api.Project
import org.gradle.api.Task

import java.text.SimpleDateFormat

/**
 Everything For Easy compiling
 Only used by RootProject
 @Author Damon
 */
class PluginForEasy implements Plugin<Project> {
    PluginForEasyConfig config
    DependenciesInjector dependenciesInjector = new DependenciesInjector()

    @Override
    void apply(Project rootProject) {
        int agpVersion = Integer.valueOf(AGPCompat.getAndroidGradlePluginVersionCompat().replace(".", ""))
        if (agpVersion < 420 || agpVersion > 422) {
            throw new GradleException("AGP版本兼容版本为4.2.0~4.2.2")
        }

        if (rootProject != rootProject.getRootProject()) {
            throw new GradleException("${PluginForEasy.name} 只能在根工程build.gradle使用")
        }

//        if (rootProject.buildscript.configurations.classpath.dependencies.size() > 0) {
//            throw new RuntimeException("apply plugin:\'${PluginForEasy.name}\' 请写在根工程 build.gradle 的 buildscript{} 之中")
//        }

        dependenciesInjector.init(rootProject)
        dependenciesInjector.inject(rootProject)
        rootProject.extensions.create("PluginForEasy", PluginForEasyConfig)
        rootProject.getGradle().addListener(new TimesCostWatcher())
        rootProject.afterEvaluate {
            config = rootProject.extensions.getByType(PluginForEasyConfig)
            config.check()
            rootProject.getSubprojects().each {
                //注入 依赖
                dependenciesInjector.inject(it)
                it.beforeEvaluate { Project pjt ->
                    if (isSplitModuleProject(pjt)) {
                        //插件模块工程
                        pjt.apply plugin: 'com.android.dynamic-feature'
                        pjt.apply plugin: 'PluginForArouter'
                        //pjt.apply plugin: 'com.iqiyi.qigsaw.dynamicfeature'
                        if (getModuleConfig(pjt).isMvpArch) {
                            pjt.apply plugin: 'PluginForMVP'
                        }
                    }

                    if (isOriginalModuleProject(pjt)) {
                        //常规模块工程
                        pjt.apply plugin: 'com.android.library'
                        pjt.apply plugin: 'PluginForArouter'
                        if (getModuleConfig(pjt).isMvpArch) {
                            pjt.apply plugin: 'PluginForMVP'
                        }
                    }
                }

                it.afterEvaluate { Project pjt ->
                    config_buildType(pjt)
                    //主工程 app
                    if (pjt.plugins.hasPlugin(AppPlugin)) {
                        pjt.apply plugin: 'PluginForArouter'
                        config_dependencies(pjt);
                        config_compileOptions(pjt)
                        config_ViewBinding(pjt)
                        config_productLine(pjt)
                        config_SpiltsInfo(pjt)
                        if (config.appModuleConfiguration == null || config.appModuleConfiguration.call(pjt)) config_ModulesInfo(pjt)
                    }

                    //主Library工程 aar
                    if (pjt.plugins.hasPlugin(MainLibraryPlugin)) {
                        pjt.apply plugin: 'PluginForArouter'
                        config_compileOptions(pjt)
                        config_ViewBinding(pjt)
                        config_productLine(pjt)
                        config_ModulesInfo(pjt)
                    }

                    //特别的拥有flavor的库
                    if (config.getExtraFlavorProjects() != null && config.getExtraFlavorProjects().contains(pjt.name)) {
                        config_productLine(pjt)
                    }

                    //插件模块工程
                    if (isSplitModuleProject(pjt)) {
                        config_compileOptions(pjt)
                        config_ViewBinding(pjt)
                        config_productLine(pjt)
                    }

                    //常规模块工程
                    if (isOriginalModuleProject(pjt)) {
                        config_compileOptions(pjt)
                        config_ViewBinding(pjt)
                        config_productLine(pjt)
                    }

                    //检查依赖关系
                    DependencyCheck.checkDependency(pjt, config.getModuleOnlyDependProjects())
                }
            }
        }
    }

    boolean isModuleProject(Project pjt) {
        return pjt.name.startsWith(Constants.projectNamePrefix_Module)
    }

    boolean isOriginalModuleProject(Project pjt) {
        if (isModuleProject(pjt)) {
            return !getModuleConfig(pjt).isSplit
        }
        return false
    }

    boolean isSplitModuleProject(Project pjt) {
        if (isModuleProject(pjt)) {
            return getModuleConfig(pjt).isSplit
        }
        return false
    }

    void config_dependencies(Project appProject) {
        appProject.getRootProject().getSubprojects().each {
            if (it != appProject) {
                if (isOriginalModuleProject(it)) {
                    if (config.appModuleConfiguration == null || (config.appModuleConfiguration != null && config.appModuleConfiguration.call(it))) appProject.getDependencies().invokeMethod("implementation", it)
                } else if (isSplitModuleProject(it))
                    if (config.appModuleConfiguration == null || (config.appModuleConfiguration != null && config.appModuleConfiguration.call(it))) it.getDependencies().invokeMethod("implementation", appProject)
            }
        }
    }

    //解析 模块工程配置文件
    ModuleConfig getModuleConfig(Project pjt) {
        if (!isModuleProject(pjt)) throw new RuntimeException("${pjt.name} 工程非模块工程, 非法获取Config")
        ModuleConfig config
        try {
            config = pjt.moduleConfig
        } catch (MissingPropertyException exception) {
            File configFile = pjt.file(Constants.moduleConfigFileName)
            if (!configFile.exists()) throw new RuntimeException("${pjt.name} 工程缺少${Constants.moduleConfigFileName}, 请补充在路径 ${pjt.projectDir.toString()} 下")
            config = new Gson().fromJson(configFile.newReader(), ModuleConfig)
            pjt.metaClass.moduleConfig = config
        }
        return config
    }

    //主工程 配置常规模块 工程信息
    void config_ModulesInfo(Project project) {
        project.afterEvaluate {
            def variantList = []
            if (project.plugins.hasPlugin(AppPlugin)) {
                variantList.addAll(project.android.getApplicationVariants())
            } else if (project.plugins.hasPlugin(MainLibraryPlugin)) {
                variantList.addAll(project.android.getLibraryVariants())
            }

            variantList.each { BaseVariant variant ->
                GenerateOriginModuleInfo generateModuleConfigTask = project.tasks.create("generate${variant.name.capitalize()}ModuleConfig", GenerateOriginModuleInfo.class)
                List<String> names = []
                project.getRootProject().subprojects.each { Project pjt ->
                    if (isOriginalModuleProject(pjt)) {
                        names.add(":$pjt.name".toString())
                    }
                }
                GenerateBuildConfig generateBuildConfigTask = AGPCompat.getGenerateBuildConfigTask(project, variant.name.capitalize())
                generateModuleConfigTask.className = 'ModuleConfig'
                generateModuleConfigTask.outputDir = generateBuildConfigTask.getSourceOutputDir().get().asFile
                generateModuleConfigTask.moduleNames = names
                generateModuleConfigTask.pkgName = generateBuildConfigTask.buildConfigPackageName.get()
                generateBuildConfigTask.finalizedBy generateModuleConfigTask
            }
        }
    }

    //主工程 配置 插件模块 工程信息
    void config_SpiltsInfo(Project appProject) {
        def names = []
        appProject.getRootProject().subprojects.each { Project pjt ->
            if (isSplitModuleProject(pjt)) {
                names.add(":$pjt.name".toString())
            }
        }
        appProject.android.dynamicFeatures.addAll(names)
    }

    //ProGuard文件 配置
    void config_ProGuard(Project project) {
        def release = project.android.buildTypes.release
        project.getRootProject().getAllprojects().each { Project childProject ->
            File file = childProject.file('proguard-rules.pro')
            if (file.exists() && childProject != project) {
                release.proguardFile(childProject.file('proguard-rules.pro'))
            }
        }
    }

    void config_buildType(Project project) {
        try {
            project.android { BaseExtension extension ->
                buildTypes {
                    debug {
                        signingConfig null
                        debuggable true
                    }
                    release {
                        if (project.plugins.hasPlugin(AppPlugin)) minifyEnabled true
                    }
                }
            }
        } catch (Exception e) {

        }
    }

    //配置相应产品线 以切换一些相应产品 代码资源
    void config_productLine(Project project) {
        println "=== 开始config_productLine ==="
        BaseExtension extension = project.android
        if (extension instanceof AppExtension) {
            ((AppExtension) extension).getApplicationVariants().whenObjectAdded(new Action<ApplicationVariant>() {
                @Override
                void execute(ApplicationVariant variant) {
                    VariantBuildConfiguration.config(project, variant)
                    variant.outputs.all { ApkVariantOutputImpl apkVariantOutput ->
                        apkVariantOutput.outputFileName = "${apkVariantOutput.name}-${apkVariantOutput.getVersionNameOverride()}.apk"
                    }
                }
            })

            project.afterEvaluate {
                def platformTasksWithPublic = new HashMap<String, DefaultTask>()
                def platformTasksWithPrivate = new HashMap<String, DefaultTask>()
                def platformTasksWithBaidu = new HashMap<String, DefaultTask>()
                project.android.applicationVariants.all { ApplicationVariant variant ->
                    def platformFlavor = variant.productFlavors.find { it.getDimension() == Constants.projectPlatformFlavor }
                    def envFlavor = variant.productFlavors.find { it.getDimension() == Constants.projectEnvFlavor }
                    if (!platformTasksWithPublic.containsKey(platformFlavor.name)) {
                        platformTasksWithPublic.put(platformFlavor.name, new ArrayList())
                    }
                    if (!platformTasksWithPrivate.containsKey(platformFlavor.name)) {
                        platformTasksWithPrivate.put(platformFlavor.name, new ArrayList())
                    }
                    if (!platformTasksWithBaidu.containsKey(platformFlavor.name)) {
                        platformTasksWithBaidu.put(platformFlavor.name, new ArrayList())
                    }

                    def tasks = platformTasksWithPublic[platformFlavor.name]
                    tasks.add(project.task("packing${variant.name.capitalize()}WithIjmService", type: PublicServiceApkPackingTask) {
                        mVariant = variant;
                        mOutputDir = project.getRootProject().file("outputs/ijmService/${platformFlavor.name}/${envFlavor.name.toLowerCase()}")
                        mChannels = platformFlavor.channels
                        mTerminal = platformFlavor.terminal
                        SimpleDateFormat sdf = new SimpleDateFormat("yyyyMMdd")
                        mDate = sdf.format(new Date(System.currentTimeMillis()))
                        Task cleanTask = project.tasks.findByName('clean')
                        dependsOn cleanTask
                        dependsOn variant.getAssembleProvider().get()
                        variant.getAssembleProvider().get().mustRunAfter(cleanTask)
                    })
                    tasks = platformTasksWithPrivate[platformFlavor.name]
                    tasks.add(project.task("packing${variant.name.capitalize()}WithGooseService", type: PrivateServiceApkPackingTask) {
                        mVariant = variant;
                        mOutputDir = project.getRootProject().file("outputs/gooseService/${platformFlavor.name}/${envFlavor.name.toLowerCase()}")
                        mChannels = platformFlavor.channels
                        mTerminal = platformFlavor.terminal
                        SimpleDateFormat sdf = new SimpleDateFormat("yyyyMMdd")
                        mDate = sdf.format(new Date(System.currentTimeMillis()))
                        Task cleanTask = project.tasks.findByName('clean')
                        dependsOn cleanTask
                        dependsOn variant.getAssembleProvider().get()
                        variant.getAssembleProvider().get().mustRunAfter(cleanTask)
                    })
                    tasks = platformTasksWithBaidu[platformFlavor.name]
                    tasks.add(project.task("packing${variant.name.capitalize()}WithBaiduService", type: BaiduApkPackingTask) {
                        mVariant = variant;
                        mOutputDir = project.getRootProject().file("outputs/baiduService/${platformFlavor.name}/${envFlavor.name.toLowerCase()}")
                        mChannels = platformFlavor.channels
                        mTerminal = platformFlavor.terminal
                        SimpleDateFormat sdf = new SimpleDateFormat("yyyyMMdd")
                        mDate = sdf.format(new Date(System.currentTimeMillis()))
                        Task cleanTask = project.tasks.findByName('clean')
                        dependsOn cleanTask
                        dependsOn variant.getAssembleProvider().get()
                        variant.getAssembleProvider().get().mustRunAfter(cleanTask)
                    })
                }

                platformTasksWithPublic.each { Map.Entry<String, List<DefaultTask>> entry ->
                    project.task("packing${entry.key.capitalize()}ALLReleaseWithIjmService", group: entry.value[0].getGroup(), type: DefaultTask) {
                        dependsOn(entry.value.findAll {
                            it.name.toLowerCase().contains("release")
                        })
                    }.doFirst {
                        org.apache.commons.io.FileUtils.deleteDirectory(project.getRootProject().file("outputs"))
                    }
                }
                platformTasksWithPrivate.each { Map.Entry<String, List<DefaultTask>> entry ->
                    project.task("packing${entry.key.capitalize()}ALLReleaseWithGooseService", group: entry.value[0].getGroup(), type: DefaultTask) {
                        dependsOn(entry.value.findAll {
                            it.name.toLowerCase().contains("release")
                        })
                    }.doFirst {
                        org.apache.commons.io.FileUtils.deleteDirectory(project.getRootProject().file("outputs"))
                    }
                }
                platformTasksWithBaidu.each { Map.Entry<String, List<DefaultTask>> entry ->
                    project.task("packing${entry.key.capitalize()}ALLReleaseWithBaiduService", group: entry.value[0].getGroup(), type: DefaultTask) {
                        dependsOn(entry.value.findAll {
                            it.name.toLowerCase().contains("release")
                        })
                    }.doFirst {
                        org.apache.commons.io.FileUtils.deleteDirectory(project.getRootProject().file("outputs"))
                    }
                }
            }
        } else if (extension instanceof LibraryExtension) {
            ((LibraryExtension) extension).getLibraryVariants().whenObjectAdded(new Action<LibraryVariant>() {
                @Override
                void execute(LibraryVariant variant) {
                    VariantBuildConfiguration.config(project, variant)
                }
            });
        }

        project.android { BaseExtension ext ->
            boolean isApp = ext instanceof AppExtension
            Closure<Properties> configApplicationProperties = { ProductFlavor productFlavor ->
                if (!isApp) return null
                Properties props = new Properties();
                File propsFile = project.getRootProject().file("${productFlavor.name}.properties")
                props.load(new FileInputStream(propsFile))
                //如果Project的Properties中有相同的KeyValue, 需要做替换, Jenkins会配置相应的Properties
                props.keySet().each { def key ->
                    String value = project.properties.get(key)
                    if (value != null && !value.isEmpty()) props.put(key, project.properties.get(key))
                }

                Closure<String> checkKVEntry = { String key ->
                    if (!props.containsKey(key)) throw new PropertyLostException(propsFile, key)
                    key
                }

                productFlavor.versionName = props.get(checkKVEntry("versionName"))
                productFlavor.versionCode = Integer.valueOf(props.get(checkKVEntry("versionCode")))
                productFlavor.applicationId = props.get(checkKVEntry("applicationId"))
                def signConfig = new SigningConfig(productFlavor.name)
                signConfig.setKeyAlias(props.get(checkKVEntry("keyAlias")))
                signConfig.setKeyPassword(props.get(checkKVEntry("keyPassword")))
                signConfig.setStorePassword(props.get(checkKVEntry("keyPassword")))
                signConfig.setStoreFile(project.rootProject.file(props.get(checkKVEntry("signFilePath"))))
                productFlavor.signingConfig = signConfig
                HashSet<String> channels = new HashSet<>()
                props.get(checkKVEntry("channels")).toString().split(",").each {
                    channels.add(it)
                }
                File extraChannelsFile = project.rootProject.file("channels.txt")
                if (extraChannelsFile.exists()) {
                    extraChannelsFile.readLines().each {
                        channels.add(it)
                    }
                }
                productFlavor.metaClass.channels = channels.toList()
                productFlavor.metaClass.terminal = props.get(checkKVEntry("terminal"))
            }

            flavorDimensions Constants.projectPlatformFlavor, Constants.projectEnvFlavor
            config.getEnv().each { String env ->
                ext.productFlavors.create(env) {
                    dimension Constants.projectEnvFlavor
                    if (config.envConfiguration != null) config.envConfiguration(it)
                }
            }

            config.getPlatforms().each { String platform ->
                ext.productFlavors.create(platform) {
                    dimension Constants.projectPlatformFlavor
                    if (config.platformConfiguration != null) config.platformConfiguration(it)
                    FileUtils.createPropertiesFileIfNeed(project.getRootProject().file("${platform}.properties"), "此文件配置 业务平台 数据项")
                    configApplicationProperties(it)
                }
            }
        }
    }

    //ViewBinding
    void config_ViewBinding(Project project) {
        project.android {
            buildFeatures {
                viewBinding true
            }
        }
    }

    void config_compileOptions(Project project) {
        project.android {
            compileOptions {
                sourceCompatibility JavaVersion.VERSION_1_8
                targetCompatibility JavaVersion.VERSION_1_8
            }
        }
    }

}
