package com.cv.media.lib.plugin.packing

import com.cv.media.lib.plugin.Logger
import org.gradle.api.GradleException

import java.util.zip.ZipEntry
import java.util.zip.ZipFile

/**
 * 百度加固打包任务
 * 职责:apkprotect [-hvV] [-url serverAddress] -akey AKEY -skey SKEY -i input_file -o output_file [-type type] [-lib libs] [-shell shell_version] [-t timeout] [-nocache] [-g taskID] [-config configID]
 参数说明
 -h 打印帮助信息
 -v 打印详细日志
 -V 打印版本信息
 -url [url] 加固地址 默认:https://apkprotect.baidu.com
 -akey access_key, 登录后在加固配置处获取
 -skey security_key，登录后在加固配置处获取
 -i [input_file] 输入文件
 -o [output_file] 输出文件
 -type 加固类型：free|standard|ultimate|so|jar，分别对应：免费版|专业版|旗舰版|so|安卓SDK加固
 -lib [lib1:lib2...] 加固类型为so时，输入文件中指定需要加固so文件名（不包括路径）
 -shell 加固服务版本号，高级用户使用
 -t $timeout 设置等待超时时间, 单位秒, 默认300秒
 -nocache 禁用缓存，默认开启
 -g $taskID 用于离线加固，超时的任务在一段时间后通过-g指令获取加固结果
 -config $configID 用于指定页面配置的id进行加固，常用于安卓SDK指定配置文件
 * @author Roger
 */
class BaiduApkPackingTask extends PublicServiceApkPackingTask {

    BaiduApkPackingTask() {
        setGroup("百度加固打包");
    }

    @Override
    File makeProtectedApk(File apkOriginFile) {
        logs("加固 开始...")
        Logger.startCount()
        // 加固语法
        String[] argArr = [
                "${toolsDir.path}${File.separator}apkprotect.exe",
                "-akey",
                "jnAEW4jWREMBiSUc",
                "-skey",
                "yx9VvvGVREE75rDLXy7KCsCyj4BejEUx",
                "-i",
                "${apkOriginFile.path}",
                "-o",
                "${mOutputDir.path}${File.separator}enc_unsigned.apk",
                "-type",
                "free"
//                "ultimate",// 将 "ultimate" 表示旗舰加固
        ]

        String[] javaVersion = new String[2]
        javaVersion[0] = "java"
        javaVersion[1] = "-version"
        execute(javaVersion, "检查java版本服务")
        Logger.startCount()
        execute(argArr, "百度加固服务")
        Logger.endCount("加固任务")
        logs("加固 结束...")

        List<File> apks = mOutputDir.listFiles().findAll {
            it.name.contains("enc_unsigned.apk")
        }

        if (apks.size() != 1) {
            throw new GradleException("Apk加固异常: 找到${apks.size()}个匹配apk")
        }
        File protectApkFile = new File(mOutputDir, "${apkOriginFile.name.replace(".apk", "_protected.apk")}")
        logs("更改加固后apk:${apks[0].name}为: $protectApkFile.name")
        apks[0].renameTo(protectApkFile)
        if (!protectApkFile.exists()) throw new GradleException("Apk加固失败: 加固包不存在")

        logs("开始检查加固后apk: $protectApkFile.name")
        ZipFile zipFileOfPAF = new ZipFile(protectApkFile)
        Enumeration<?> entries = zipFileOfPAF.entries()
        boolean hasBaiduSo = false
        while (entries.hasMoreElements()) {
            ZipEntry entry = (ZipEntry) entries.nextElement()
            if (entry.name.endsWith("baiduprotect.so")) {
                hasBaiduSo = true
                break
            }
        }

        if (!hasBaiduSo) {
            throw new GradleException("Apk加固异常: 未发现baiduprotect.so")
        }

        logs("加固成功: $protectApkFile.name")
        return protectApkFile
    }

    @Override
    void logs(String content) {
        println("${BaiduApkPackingTask.simpleName}--> ${content}");
    }

    void execute(String[] commands, String tag) {
        Process process = Runtime.getRuntime().exec(commands)
        logs("tag:${tag} 执行命令: ${commands.join(" ")}")
        BufferedReader stdoutReader = new BufferedReader(new InputStreamReader(process.getInputStream()))
        BufferedReader stderrReader = new BufferedReader(new InputStreamReader(process.getErrorStream()))
        String line
        while ((line = stdoutReader.readLine()) != null || (line = stderrReader.readLine()) != null) {
            logs("${tag}:: " + line)
        }
        stdoutReader.close();
        stderrReader.close();
    }
}
