package com.cv.media.lib.gradle.publish

import com.android.build.gradle.AppExtension
import com.android.build.gradle.LibraryExtension
import org.gradle.BuildListener
import org.gradle.BuildResult
import org.gradle.api.GradleException
import org.gradle.api.Plugin
import org.gradle.api.Project
import org.gradle.api.initialization.Settings
import org.gradle.api.invocation.Gradle
import org.gradle.api.publish.maven.MavenPublication
import org.gradle.jvm.tasks.Jar

class MavenPublishPlugin implements Plugin<Project> {
    MavenConfigRoot baseConfig
    final String extensionName = "PluginForPublish"

    @Override
    void apply(Project target) {
        if (target == target.getRootProject()) {
            target.extensions.create(extensionName, MavenConfigRoot)
            target.afterEvaluate {
                displayInstructions()
                baseConfig = target.extensions.getByType(MavenConfigRoot)
                baseConfig.artifactId = ""
                baseConfig.projectFilter = baseConfig.projectFilter == null ? { true } : baseConfig.projectFilter
                checkConfig(baseConfig)
                target.getSubprojects().each {
                    it.extensions.create(extensionName, MavenConfig)
                    it.afterEvaluate { Project sub ->
                        if (isAppProject(sub) || !baseConfig.projectFilter.call(sub)) return
                        def setComponentFunc
                        def setSourceCodeFunc
                        def subConfig = sub.extensions.getByType(MavenConfig)
                        if (subConfig.artifactId == null || subConfig.artifactId.isEmpty()) subConfig.artifactId = sub.name
                        subConfig.mergeBase(baseConfig)
                        printConfig(subConfig, sub)
                        if (isJavaLibProject(sub)) {
                            setComponentFunc = { MavenPublication mavenPublication ->
                                sub.afterEvaluate {
                                    if (sub.components.java != null) {
                                        mavenPublication.from(sub.components.java)
                                    }
                                }
                            }
                            setSourceCodeFunc = { MavenPublication mavenPublication ->
                                def srcTask = sub.task(["type": Jar.class], "sourcesJar")
                                srcTask.archiveClassifier.set("sources")
                                srcTask.from(sub.sourceSets.main.allJava.getSrcDirs())
                                mavenPublication.artifact(srcTask)
                            }
                        } else if (isAndroidLibProject(sub)) {
                            setComponentFunc = { MavenPublication mavenPublication ->
                                sub.afterEvaluate {
                                    def releaseComponent = sub.components.find {
                                        it.name.toLowerCase().contains('release')
                                    }
                                    if (releaseComponent != null) {
                                        mavenPublication.from(releaseComponent)
                                    }
                                }
                            }
                            setSourceCodeFunc = { MavenPublication mavenPublication ->
                                def srcTask = sub.task(["type": Jar.class], "sourcesJar")
                                srcTask.archiveClassifier.set("sources")
                                srcTask.from(sub.android.sourceSets.main.java.getSourceFiles().getFiles())
                                mavenPublication.artifact(srcTask)
                            }
                        } else
                            return

                        sub.apply plugin: "maven-publish"
                        sub.publishing {
                            publications {
                                mine(MavenPublication) {
                                    setComponentFunc(it)
                                    setSourceCodeFunc(it)
                                    groupId subConfig.groupId
                                    artifactId subConfig.artifactId
                                    version subConfig.getFullVersion()
                                }
                            }

                            repositories {
                                maven {
                                    url subConfig.getRepo()
                                    credentials {
                                        username subConfig.getRepoUsrName()
                                        password subConfig.getRepoPsw()
                                    }
                                }
                            }
                        }

                        sub.task("type": MavenPublishTask, "publishToRemoteRepo") {
                            it.config = subConfig
                        }
                    }
                }
            }

        } else {
            throw new GradleException("请在根工程使用该插件 子工程:${target.name} 不应该apply PluginForPublish")
        }
    }

    void checkConfig(MavenConfig config) {
        def errorInfo = ""
        config.properties.findAll {
            it.value == null && it.key != 'taskThatOwnsThisObject'
        }.each {
            errorInfo += "\n${extensionName}->${it.key} 没有配置"
        }
        if (!errorInfo.isEmpty()) throw new GradleException(errorInfo)
    }

    void printConfig(MavenConfig config, Project project) {
        println "工程 ${project.name} 打包配置: ${config.groupId}:${config.artifactId}:${config.getFullVersion()} ==>Repo: ${config.getRepo()}"
    }

    void displayInstructions() {
        println "**************************************************************"
        println "**************************************************************"
        println "******************$extensionName 配置说明******************"
        println "请在根工程 下build.gradle中配置: \n" +
                "$extensionName {\n" +
                "\tisSnapshot value(boolean 所有的子工程都会应用, 若要单独控制, 子工程覆写)\n" +
                "\tgroupId value(String 所有的子工程都会应用, 若要单独控制, 子工程覆写)\n" +
                "\tversion value(String 所有的子工程都会应用, 若要单独控制, 子工程覆写)\n" +
                "\trepoRelease value(String 所有的子工程都会应用, 若要单独控制, 子工程覆写)\n" +
                "\trepoReleaseUser value(String 所有的子工程都会应用, 若要单独控制, 子工程覆写)\n" +
                "\trepoReleasePsw value(String 所有的子工程都会应用, 若要单独控制, 子工程覆写)\n" +
                "\trepoSnapshot value(String 所有的子工程都会应用, 若要单独控制, 子工程覆写)\n" +
                "\trepoSnapshotUser value(String 所有的子工程都会应用, 若要单独控制, 子工程覆写)\n" +
                "\trepoSnapshotPsw value(String 所有的子工程都会应用, 若要单独控制, 子工程覆写)\n" +
                "}"
        println "子工程 请在子工程 下build.gradle同样配置 比如:"
        println "$extensionName { \n" +
                "\tartifactId value(String 默认为工程名)\n" +
                "\tversion newValue(String 如果有单独的版本控制, 就设置一个新的版本号)\n" +
                "}\n"
        println "**************************************************************"
        println "**************************************************************"
        println "**************************************************************"
    }

    boolean isAppProject(Project project) {
        return project.extensions.findByName("android") != null && project.extensions.findByName("android") instanceof AppExtension
    }

    boolean isJavaLibProject(Project project) {
        return project.extensions.findByName("java") != null && project.extensions.findByName("android") == null
    }

    boolean isAndroidLibProject(Project project) {
        return project.extensions.findByName("android") != null && project.extensions.findByName("android") instanceof LibraryExtension
    }
}
