package com.cv.media.lib.metric.manager;

import android.content.Context;
import android.text.TextUtils;

import com.cv.media.lib.metric.BuildConfig;
import com.cv.media.lib.metric.data.MetricData;
import com.cv.media.lib.metric.chain.MetricProcessChain;
import com.vms.metric.sender.service.MetricSender;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

public class MetricManager {
    private static MetricManager ourInstance = null;
    private Context mContext;
    private Object lock = new Object();
    private String actionPrefix;
    //播放路径记录
    private Map<String, String> mPlayAction = new HashMap<>();

    private Map<MetricKey, List<MetricProcessChain>> chains = new ConcurrentHashMap<>();
    private List<IMetricCallback> metricCallbacks = new ArrayList<>();

    public enum MetricKey {
        APP,
        PM,
        SLA,
        PCT
    }

    public static MetricManager getInstance() {
        if (null == ourInstance) {
            throw new RuntimeException(MetricManager.class.getSimpleName() + " not instantiated");
        }
        return ourInstance;
    }

    public static void init(Context context, String appId, String urls, Map<String, String[]> extraMap, String actionPrefix) {
        if (null == ourInstance) {
            ourInstance = new MetricManager(context, appId, urls, extraMap, actionPrefix);
        }
    }

    private MetricManager(Context context, String appId, String urls, Map<String, String[]> extraMap, String actionPrefix) {
        this.mContext = context;
        if (null != urls && TextUtils.isEmpty(urls.replace(" ", ""))) {
            return;
        }
        this.actionPrefix = actionPrefix;
        String[] urlsGroups = urls.split("#");
        Map<String, String[]> urlMap = new HashMap<>();
        for (int i = 0; i < urlsGroups.length; i++) {
            String[] urlArray = null;
            if (!TextUtils.isEmpty(urlsGroups[i])) {
                urlArray = urlsGroups[i].split(",");
            }
            if (i < MetricKey.values().length) {
                urlMap.put(MetricKey.values()[i].name(), urlArray);
            }
        }

        if (extraMap != null) {
            urlMap.putAll(extraMap);
        }

        if (BuildConfig.DEBUG) {
            MetricSender.init(appId, "", urlMap, 100, 60000, true);
        } else {
            MetricSender.init(appId, "", urlMap);
        }
    }

    public MetricManager addMetricChain(MetricProcessChain chain) {
        return addMetricChain(MetricKey.APP, chain);
    }

    public MetricManager addMetricChain(MetricKey metricKey, MetricProcessChain chain) {
        synchronized (lock) {
            List<MetricProcessChain> metricProcessChains = chains.get(metricKey);
            metricProcessChains = (metricProcessChains == null ? new ArrayList<>() : metricProcessChains);
            chains.put(metricKey, metricProcessChains);
            metricProcessChains.add(chain);
        }
        return this;
    }

    /**
     * 为null时不更新
     *
     * @param sourcePage
     * @param sourcePageId
     */
    public void savePlayAction(String sourcePage, String sourcePageId) {
        if (sourcePage != null) {
            mPlayAction.put("sourcePage", sourcePage);
        }
        if (sourcePageId != null) {
            mPlayAction.put("sourcePageId", sourcePageId);
        }
    }

    public Map<String, String> getPlayAction() {
        return mPlayAction;
    }

    public void clearPlayAction() {
        mPlayAction.clear();
    }

    public void sendAppMetric(String key, Map<String, Object> map) {
        try {
            MetricManager.getInstance().put(key, new MetricData.Builder().put(map).build());
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    @Deprecated
    public void put(String action) {
        put(action, new MetricData.Builder().build());
    }

    @Deprecated
    public void put(String action, MetricData data) {
        put(MetricKey.APP, action, data);
    }

    public void put(MetricKey metricKey, String action) {
        put(metricKey, action, new MetricData.Builder().build());
    }

    public void put(MetricKey metricKey, String action, MetricData data) {
        put(metricKey, action, data, true);
    }

    public void put(MetricKey metricKey, String action, MetricData data, boolean withChain) {
        if (withChain) {
            synchronized (lock) {
                List<MetricProcessChain> chainList = chains.get(metricKey);
                if (null != chainList) {
                    for (MetricProcessChain chain : chainList) {
                        data = chain.process(data);
                    }
                }
            }
        }


        if (metricKey == MetricKey.APP || metricKey == MetricKey.SLA) {
            action = actionPrefix + action;
        }
        MetricSender.put(metricKey.name(), action, data.value());

        notifyCallback(action, data.value());
    }

    public MetricData getChainSnapshot(MetricKey metricKey) {
        MetricData data = new MetricData.Builder().build();
        synchronized (lock) {
            List<MetricProcessChain> chainList = chains.get(metricKey);
            if (null != chainList) {
                for (MetricProcessChain chain : chainList) {
                    data = chain.process(data);
                }
            }
        }
        return data;
    }

    private void notifyCallback(String eventId, Map<String, Object> fieldsMap) {
        for(IMetricCallback callback : metricCallbacks) {
            callback.onEvent(eventId, fieldsMap);
        }
    }

    public void addMetricCallback(IMetricCallback metricCallback) {
        if (metricCallbacks != null && !metricCallbacks.contains(metricCallback)) {
            metricCallbacks.add(metricCallback);
        }
    }

    public void removeMetricCallback(IMetricCallback metricCallback) {
        if (metricCallbacks != null) {
            metricCallbacks.remove(metricCallback);
        }
    }

    public interface IMetricCallback {
        void onEvent(String eventId, Map<String, Object> fieldsMap);
    }
}
