package com.cv.media.lib.mvx.mvp;

import android.app.Activity;
import android.content.Intent;
import android.os.Bundle;
import android.util.TypedValue;
import android.view.Gravity;
import android.view.ViewTreeObserver;
import android.view.WindowManager;
import android.widget.TextView;

import androidx.appcompat.app.AlertDialog;

import java.util.ArrayList;
import java.util.Objects;

import com.cv.media.lib.common_utils.callback.ICallBack;
import com.cv.media.lib.mvx.R;
import com.cv.media.lib.mvx.base.BaseActivity;
import com.cv.media.lib.mvx.base.BaseDialogFragment;

public abstract class MVPBaseDialog extends BaseDialogFragment implements BaseViewInterface {
    private final ArrayList<LifeListener> lifeListeners = new ArrayList<>();

    //创建Present
    private void createPresenter() {
        PresenterCenter.createPresenter(this);
    }

    public void back() {
        dismiss();
    }

    @Override
    public void addLifeListener(LifeListener listener) {
        if (listener != null) {
            lifeListeners.add(listener);
        }
    }

    @Override
    public void removeLifeListener(LifeListener listener) {
        if (listener != null) {
            lifeListeners.remove(listener);
        }
    }

    public void onViewCreated() {
        super.onViewCreated();
        for (LifeListener listener : lifeListeners) {
            listener.onViewCreated();
        }
    }

    @Override
    public void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        createPresenter();
        for (LifeListener listener : lifeListeners) {
            listener.onCreate(savedInstanceState, getArguments() != null ? getArguments() : new Bundle());
        }
    }

    @Override
    public void onResume() {
        super.onResume();
        for (LifeListener listener : lifeListeners) {
            listener.onResume();
        }
    }

    @Override
    public void onPause() {
        super.onPause();
        for (LifeListener listener : lifeListeners) {
            listener.onPause();
        }
    }

    @Override
    public void onStart() {
        super.onStart();
        for (LifeListener listener : lifeListeners) {
            listener.onStart();
        }
    }

    @Override
    public void onStop() {
        super.onStop();
        for (LifeListener listener : lifeListeners) {
            listener.onStop();
        }
    }

    @Override
    public void onSaveInstanceState(Bundle outState) {
        super.onSaveInstanceState(outState);
        for (LifeListener listener : lifeListeners) {
            listener.onSave(outState);
        }
    }

    @Override
    public void onViewStateRestored(Bundle savedInstanceState) {
        super.onViewStateRestored(savedInstanceState);
        for (LifeListener listener : lifeListeners) {
            listener.onRestore(savedInstanceState);
        }
    }

    @Override
    public void onDestroy() {
        super.onDestroy();
        for (int i = 0; i < lifeListeners.size(); i++) {
            LifeListener listener = lifeListeners.get(i);
            listener.onDestroy();
            lifeListeners.remove(listener);
            i--;
        }
    }

    @Override
    public void onHiddenChanged(boolean hidden) {
        super.onHiddenChanged(hidden);
        if (hidden) {
            for (LifeListener listener : lifeListeners) {
                listener.onStop();
            }
        } else {
            for (LifeListener listener : lifeListeners) {
                listener.onStart();
            }
        }
    }

    @Override
    public void onActivityResult(int requestCode, int resultCode, Intent data) {
        super.onActivityResult(requestCode, resultCode, data);
        for (LifeListener listener : lifeListeners) {
            listener.onActivityResult(requestCode, resultCode, data);
        }
    }

    @Override
    public Activity provideActivity() {
        return getActivity();
    }

    @Override
    public void notifyOnClickBack(ICallBack<ICallBack<Boolean>> cb) {

    }

    @Override
    public void reset() {

    }

    @Override
    public void doWhenFirstTimeInSight(Runnable runnable) {
        doAfterViewReady(new Runnable() {
            @Override
            public void run() {
                getView().getViewTreeObserver().addOnPreDrawListener(new ViewTreeObserver.OnPreDrawListener() {
                    @Override
                    public boolean onPreDraw() {
                        getView().getViewTreeObserver().removeOnPreDrawListener(this);
                        runnable.run();
                        return false;
                    }
                });
            }
        });
    }

    protected void notifyUIAction(String action, Object... uiArgs) {
        if (getActivity() != null) {
            getActivity().runOnUiThread(() -> {
                BasePresenter presenter = PresenterCenter.fetchBindPresenter(this);
                if (presenter != null) {
                    UIEventsCenter uiEventsCenter =  BasePresenter.PresentersManagerGlobal.getInstance().getUIEventsCenter(presenter);
                    if (uiEventsCenter != null)  uiEventsCenter.notifyEvent(action, uiArgs);
                }
            });
        }
    }

    @Override
    public void showDebugContent(String content) {
        try {
            AlertDialog dialog = new AlertDialog.Builder(Objects.requireNonNull(getContext()), R.style.Theme_AppCompat_Dialog)
                    .setTitle("Debug Content Dialog")
                    .setMessage(content)
                    .setPositiveButton("Close", null)
                    .show();
            TextView textView = dialog.findViewById(android.R.id.message);
            if (textView != null) {
                textView.setTextSize(TypedValue.COMPLEX_UNIT_SP, 12);
            }

            WindowManager.LayoutParams lp = dialog.getWindow().getAttributes();
            lp.gravity = Gravity.CENTER;
            dialog.getWindow().setAttributes(lp);
        }catch (Exception e) {
            System.out.println(content);
        }
    }
}
